/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 * Copyright (c) 2016 Codethink Ltd
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The StoryBoard administration module.
 */
angular.module('sb.admin', [ 'sb.services', 'sb.templates', 'sb.util',
    'ui.router'])
    .config(function ($stateProvider, $urlRouterProvider, PermissionResolver) {
        'use strict';

        // Routing Defaults.
        $urlRouterProvider.when('/admin', '/admin/user');

        // Declare the states for this module.
        $stateProvider
            .state('sb.admin', {
                abstract: true,
                views: {
                    'submenu@': {
                        templateUrl: 'app/admin/template/admin_submenu.html'
                    },
                    '@': {
                        template: '<div ui-view></div>'
                    }
                },
                url: '/admin',
                resolve: {
                    isSuperuser: PermissionResolver
                        .requirePermission('is_superuser', true)
                }
            })
            .state('sb.admin.user', {
                url: '/user',
                templateUrl: 'app/admin/template/user.html',
                controller: 'UserAdminController'
            })
            .state('sb.admin.user_edit', {
                url: '/user/:id',
                templateUrl: 'app/admin/template/user_edit.html',
                controller: 'UserEditController',
                resolve: {
                    user: function ($stateParams, User) {
                        return User.get({id: $stateParams.id}).$promise;
                    }
                }
            })
            .state('sb.admin.team', {
                url: '/team',
                templateUrl: 'app/admin/template/team.html',
                controller: 'TeamAdminController'
            })
            .state('sb.admin.team_edit', {
                url: '/team/:id',
                templateUrl: 'app/admin/template/team_edit.html',
                controller: 'TeamEditController',
                resolve: {
                    team: function ($stateParams, Team) {
                        return Team.get({team_id: $stateParams.id}).$promise;
                    },
                    members: function($stateParams, Team) {
                        return Team.UsersController.get({
                            team_id: $stateParams.id
                        }).$promise;
                    },
                    projects: function($stateParams, Team) {
                        return Team.ProjectsController.get({
                            team_id: $stateParams.id
                        }).$promise;
                    }
                }
            });
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * This StoryBoard module contains our authentication and authorization logic.
 */
angular.module('sb.auth', [ 'sb.services', 'sb.templates', 'ui.router',
        'sb.util', 'LocalStorageModule']
).config(function ($stateProvider, SessionResolver) {
        'use strict';

        // Declare the states for this module.
        $stateProvider
            .state('sb.auth', {
                abstract: true,
                template: '<div ui-view></div>',
                url: '/auth'
            })
            .state('sb.auth.authorize', {
                url: '/authorize?error&error_description',
                templateUrl: 'app/auth/template/busy.html',
                controller: 'AuthAuthorizeController',
                resolve: {
                    isLoggedOut: SessionResolver.requireLoggedOut
                }
            })
            .state('sb.auth.deauthorize', {
                url: '/deauthorize',
                templateUrl: 'app/auth/template/busy.html',
                controller: 'AuthDeauthorizeController',
                resolve: {
                    isLoggedIn: SessionResolver.requireLoggedIn
                }
            })
            .state('sb.auth.token', {
                url: '/token?code&state&error&error_description',
                templateUrl: 'app/auth/template/busy.html',
                controller: 'AuthTokenController',
                resolve: {
                    isLoggedOut: SessionResolver.requireLoggedOut
                }
            })
            .state('sb.auth.error', {
                url: '/error?error&error_description',
                templateUrl: 'app/auth/template/error.html',
                controller: 'AuthErrorController'
            });
    })
    .run(function ($rootScope, SessionState, Session, PermissionManager,
                   Notification, Priority) {
        'use strict';

        // Initialize our permission manager.
        PermissionManager.initialize();

        // Always record the logged in state on the root scope.
        Notification.intercept(function (message) {
            switch (message.type) {
                case SessionState.LOGGED_IN:
                    $rootScope.isLoggedIn = true;
                    break;
                case SessionState.LOGGED_OUT:
                    $rootScope.isLoggedIn = false;
                    break;
                default:
                    break;
            }
        }, Priority.LAST);
    });

/*
 * Copyright (c) 2015 Codethink Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The StoryBoard board submodule handles the creation and usage of
 * kanban-style boards.
 */
angular.module('sb.board', ['ui.router', 'sb.services', 'sb.util',
    'ui.bootstrap'])
    .config(function ($stateProvider, $urlRouterProvider) {
        'use strict';

        // URL Defaults.
        $urlRouterProvider.when('/board', '/board/list');

        // Set our page routes.
        $stateProvider
            .state('sb.board', {
                abstract: true,
                url: '/board',
                template: '<div ui-view></div>'
            })
            .state('sb.board.detail', {
                url: '/{boardID:[0-9]+}',
                controller: 'BoardDetailController',
                templateUrl: 'app/boards/template/detail.html'
            })
            .state('sb.board.list', {
                url: '/list',
                controller: 'BoardsListController',
                templateUrl: 'app/boards/template/list.html'
            });
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The storyboard dashboard module. Handles our index page.
 */
angular.module('sb.dashboard',
    ['sb.services', 'sb.templates', 'sb.auth', 'ui.router', 'ui.bootstrap'])
    .config(function ($stateProvider, SessionResolver) {
        'use strict';

        // Set an initial home page.
        $stateProvider
            .state('sb.index', {
                url: '/',
                templateUrl: 'app/dashboard/template/index.html',
                controller: 'HomeController',
                resolve: {
                    sessionState: SessionResolver.resolveSessionState
                }
            })
            .state('sb.dashboard', {
                abstract: true,
                url: '/dashboard',
                resolve: {
                    sessionState: SessionResolver.resolveSessionState,
                    currentUser: SessionResolver.requireCurrentUser
                },
                views: {
                    'submenu@': {
                        templateUrl: 'app/dashboard/template/submenu.html'
                    },
                    '@': {
                        template: '<div ui-view></div>'
                    }
                }
            })
            .state('sb.dashboard.stories', {
                url: '/stories',
                controller: 'DashboardController',
                templateUrl: 'app/dashboard/template/dashboard.html'
            })
            .state('sb.dashboard.boards', {
                url: '/boards',
                controller: 'BoardsWorklistsController',
                templateUrl: 'app/dashboard/template/boards_worklists.html'
            })
            .state('sb.dashboard.subscriptions', {
                url: '/subscriptions',
                templateUrl: 'app/dashboard/template/subscriptions.html',
                controller: 'DashboardSubscriptionsController'
            });
    });

/*
 * Copyright (c) 2016 Codethink Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The StoryBoard due dates submodule handles the creation of task and story
 * due dates.
 */
angular.module('sb.due_date', ['ui.router', 'sb.services', 'sb.util',
                               'ui.bootstrap'])
    .config(function ($stateProvider, $urlRouterProvider) {
        'use strict';

        // URL Defaults.
        $urlRouterProvider.when('/due_date', '/due_date/list');

        // Set our page routes.
        $stateProvider
            .state('sb.due_date', {
                abstract: true,
                url: '/due_date',
                template: '<div ui-view></div>'
            })
            .state('sb.due_date.detail', {
                url: '/{dueDateID:[0-9]+}',
                controller: 'DueDateDetailController',
                templateUrl: 'app/due_dates/template/detail.html'
            });
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * This module acts as the central routing point for all errors that occur
 * within storyboard.
 */
angular.module('sb.notification', []);

/*
 * Copyright (c) 2013 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The StoryBoard pages submodule contains mostly static content pages that
 * require little functionality themselves.
 *
 * @author Michael Krotscheck
 */
angular.module('sb.pages',
        [ 'sb.services', 'sb.templates', 'sb.pages', 'ui.router']
    )
    .config(function ($stateProvider) {
        'use strict';

        // Set our page routes.
        $stateProvider
            .state('sb.page', {
                abstract: true,
                url: '/page',
                template: '<div ui-view></div>'
            })
            .state('sb.page.about', {
                url: '/about',
                templateUrl: 'app/pages/template/about.html'
            });
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The StoryBoard root application module.
 *
 * This module contains the entire, standalone application for the StoryBoard
 * ticket tracking web client.
 *
 * @author Michael Krotscheck
 */
angular.module('sb.profile',
    ['sb.services', 'sb.templates', 'sb.auth', 'ui.router', 'ui.bootstrap']
)
    .config(function ($stateProvider, SessionResolver, $urlRouterProvider) {
        'use strict';

        // URL Defaults.
        $urlRouterProvider.when('/profile', '/profile/preferences');

        // Declare the states for this module.
        $stateProvider
            .state('sb.profile', {
                abstract: true,
                url: '/profile',
                resolve: {
                    isLoggedIn: SessionResolver.requireLoggedIn,
                    currentUser: SessionResolver.requireCurrentUser
                },
                views: {
                    '@': {
                        template: '<div ui-view></div>'
                    }
                }
            })
            .state('sb.profile.preferences', {
                url: '/preferences',
                templateUrl: 'app/profile/template/preferences.html',
                controller: 'ProfilePreferencesController'
            })
            .state('sb.profile.tokens', {
                url: '/tokens',
                templateUrl: 'app/profile/template/tokens.html',
                controller: 'ProfileTokensController',
                resolve: {
                    tokens: function (CurrentUser, UserToken, $q) {
                        var deferred = $q.defer();

                        CurrentUser.resolve().then(
                            function (currentUser) {
                                UserToken.query({
                                        user_id: currentUser.id
                                    }, function (results) {
                                        deferred.resolve(results);
                                    }, function (error) {
                                        deferred.reject(error);
                                    }
                                );
                            },
                            function (error) {
                                deferred.reject(error);
                            }
                        );
                        return deferred.promise;
                    }
                }
            });
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The StoryBoard project group submodule handles most activity involving
 * searching for and reviewing project groups. Administration of project groups
 * has moved from the admin module.
 */
angular.module('sb.project_group',
    ['ui.router', 'sb.services', 'sb.util', 'sb.auth'])
    .config(function ($stateProvider, $urlRouterProvider, PermissionResolver) {
        'use strict';

        // Routing Defaults.
        $urlRouterProvider.when('/project_group', '/project_group/list');

        // Set our page routes.
        $stateProvider
            .state('sb.project_group', {
                abstract: true,
                url: '/project_group',
                template: '<div ui-view></div>',
                resolve: {
                    isSuperuser: PermissionResolver
                        .resolvePermission('is_superuser', true)
                }
            })
            .state('sb.project_group.list', {
                url: '/list',
                templateUrl: 'app/project_group/template/list.html',
                controller: 'ProjectGroupListController'
            })
            .state('sb.project_group.detail', {
                url: '/{id:any}',
                templateUrl: 'app/project_group/template/detail.html',
                controller: 'ProjectGroupDetailController',
                resolve: {
                    projectGroup: function ($stateParams, ProjectGroup, $q) {
                        var deferred = $q.defer();

                        ProjectGroup.get({id: $stateParams.id},
                            function (result) {
                                deferred.resolve(result);
                            }, function (error) {
                                deferred.reject(error);
                            });
                        return deferred.promise;
                    }
                }
        });
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The StoryBoard project submodule handles most activity surrounding the
 * creation and management of projects.
 */
angular.module('sb.projects',
    ['ui.router', 'sb.services', 'sb.util', 'sb.auth'])
    .config(function ($stateProvider, $urlRouterProvider, SessionResolver,
                      PermissionResolver) {
        'use strict';

        // Routing Defaults.
        $urlRouterProvider.when('/project', '/project/list');

        // Set our page routes.
        $stateProvider
            .state('sb.project', {
                abstract: true,
                url: '/project',
                template: '<div ui-view></div>',
                resolve: {
                    isSuperuser: PermissionResolver
                        .resolvePermission('is_superuser', true)
                }
            })
            .state('sb.project.list', {
                url: '/list',
                templateUrl: 'app/projects/template/list.html',
                controller: 'ProjectListController'
            })
            .state('sb.project.detail', {
                url: '/{id:any}',
                templateUrl: 'app/projects/template/detail.html',
                controller: 'ProjectDetailController'
            });
    })
;

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The StoryBoard search module, providing both generic and advanced
 * discovery mechanisms.
 */
angular.module('sb.search',
    ['ui.router', 'sb.services', 'sb.util', 'sb.auth'])
    .config(function ($stateProvider) {
        'use strict';

        // Set our page routes.
        $stateProvider
            .state('sb.search', {
                url: '/search',
                templateUrl: 'app/search/template/index.html',
                controller: 'SearchController'
            });
    });

/*
 * Copyright (c) 2013 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The StoryBoard Services module contains all of the necessary API resources
 * used by the storyboard client. Its resources are available via injection to
 * any module that declares it as a dependency.
 */
angular.module('sb.services', ['ngResource', 'sb.notification']);

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 * Copyright (c) 2017 Adam Coldrick
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The StoryBoard story submodule handles most activity surrounding the
 * creation and management of stories, their tasks, and comments.
 */
angular.module('sb.story', ['ui.router', 'sb.services', 'sb.util',
    'ui.bootstrap'])
    .config(function ($stateProvider, $urlRouterProvider, PreferenceProvider,
                      TimelineEventTypes, SessionResolver) {
        'use strict';

        // URL Defaults.
        $urlRouterProvider.when('/story', '/story/list');

        var queryParams = 'q&status&tags&project_group_id&'
            + 'project_id&assignee_id';

        var creationParams = 'title&description&project_id&'
            + 'private&force_private&security&tags&team_id&user_id';

        // Set our page routes.
        $stateProvider
            .state('sb.story', {
                abstract: true,
                url: '/story',
                template: '<div ui-view></div>'
            })
            .state('sb.story.list', {
                url: '/list?' + queryParams,
                params: {
                    'status': 'active'
                },
                templateUrl: 'app/stories/template/list.html',
                controller: 'StoryListController'
            })
            .state('sb.story.detail', {
                url: '/{storyId:[0-9]+}',
                templateUrl: 'app/stories/template/detail.html',
                controller: 'StoryDetailController',
                resolve: {
                    story: function (Story, $stateParams) {
                        // Pre-resolve the story.
                        return Story.get({
                            id: $stateParams.storyId
                        }).$promise;
                    },
                    creator: function (story, User) {
                        // Pre-resolve the creator after the story has been
                        // resolved.
                        if (!story.creator_id) {
                            return {};
                        } else {
                            return User.get({
                                id: story.creator_id
                            }).$promise;
                        }
                    },
                    tasks: function (Task, $stateParams) {
                        return Task.browse({
                            story_id: $stateParams.storyId
                        }).$promise;
                    },
                    currentUser: function (CurrentUser, Session) {
                        var user = Session.resolveSessionState()
                            .then(function() {
                                if (Session.isLoggedIn()) {
                                    return CurrentUser.resolve();
                                }
                                return {};
                            });
                        return user;
                    },
                    worklists: function(Worklist, $stateParams, CurrentUser) {
                        var userPromise = CurrentUser.resolve();

                        return userPromise.then(function(user) {
                            return Worklist.browse({
                                story_id: $stateParams.storyId,
                                subscriber_id: user.id,
                                hide_lanes: ''
                            }).$promise;
                        }, function() {
                            return [];
                        });
                    }
                }
            })
            .state('sb.story.new', {
                url: '/new?' + creationParams,
                templateUrl: 'app/stories/template/new_page.html',
                controller: 'StoryNewController',
                resolve: {
                    isLoggedIn: SessionResolver.requireLoggedIn,
                    currentUser: SessionResolver.requireCurrentUser
                }
            });

        // Register a preference for filtering timeline events.
        // By default all types of events should be displayed.

        TimelineEventTypes.forEach(function (type) {
            PreferenceProvider.addPreference('display_events_' + type, 'true');
        });
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The StoryBoard root application module.
 *
 * This module contains the entire, standalone application for the StoryBoard
 * ticket tracking web client.
 *
 * @author Michael Krotscheck
 */
angular.module('storyboard',
    [ 'sb.services', 'sb.templates', 'sb.dashboard', 'sb.pages', 'sb.projects',
        'sb.auth', 'sb.story', 'sb.profile', 'sb.notification', 'sb.search',
        'sb.admin', 'sb.subscription', 'sb.project_group', 'sb.worklist',
        'sb.board', 'sb.due_date', 'sb.task', 'ui.router', 'ui.bootstrap',
        'monospaced.elastic', 'angularMoment', 'angular-data.DSCacheFactory',
        'viewhead', 'ngSanitize', 'as.sortable'])
    .constant('angularMomentConfig', {
        preprocess: 'utc',
        timezone: 'UTC'
    })
    .config(function ($urlRouterProvider, $locationProvider, $httpProvider,
                      msdElasticConfig, $stateProvider, SessionResolver,
                      PreferenceResolver) {
        'use strict';

        // Default URL hashbang route
        $urlRouterProvider.otherwise('/');

        // Override the hash prefix for Google's AJAX crawling.
        $locationProvider.hashPrefix('!');

        // Attach common request headers out of courtesy to the API
        $httpProvider.defaults.headers.common['X-Client'] = 'StoryBoard';

        // Globally set an additional amount of whitespace to the end of our
        // textarea elastic resizing.
        msdElasticConfig.append = '\n';

        // Create a root state (named 'sb') that will not resolve until
        // necessary application data has been loaded.
        $stateProvider
            .state('sb', {
                abstract: true,
                url: '',
                template: '<div ui-view></div>',
                resolve: {
                    sessionState: SessionResolver.resolveSessionState,
                    preferences: PreferenceResolver.resolvePreferences
                }
            });
    })
    .run(function ($log, $rootScope, $document, $transitions,
        LastLocation) {
        'use strict';

        var resolvingClassName = 'resolving';
        var body = $document.find('body');

        // Apply a global class to the application when we're in the middle of
        // a state resolution, as well as a global scope variable that UI views
        // can switch on.
        $transitions.onStart({}, function(transition){
            body.addClass(resolvingClassName);
            $rootScope.isResolving = true;
            LastLocation.onStateChange(transition);
        });
        $transitions.onSuccess({}, function(){
            body.removeClass(resolvingClassName);
            $rootScope.isResolving = false;
        });
        $transitions.onError({}, function(){
            body.removeClass(resolvingClassName);
            $rootScope.isResolving = false;
        });
    })
    .run(function ($log, $rootScope, $state, $transitions) {
        'use strict';

        // Listen to changes on the root scope. If it's an error in the state
        // changes (i.e. a 404) take the user back to the index.
        $transitions.onError({}, function(){
            return false;
        });
    })
    .run(function ($http, DSCacheFactory) {
        'use strict';

        DSCacheFactory.createCache('defaultCache', {
            // Items added to this cache expire after 1 minute.
            maxAge: 60000,
            // Items will be deleted from this cache only on check.
            deleteOnExpire: 'passive'
        });

        $http.defaults.cache = DSCacheFactory.get('defaultCache');
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The StoryBoard subscription module. Adds directives and services necessary
 * for a user to subscribe to resource changes in storyboard.
 *
 * @author Michael Krotscheck
 */
angular.module('sb.subscription', ['sb.notification']);

/*
 * Copyright (c) 2018 Adam Coldrick
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A tiny submodule for handling task navigation. Currently just redirects
 * /task/:id to the relevant story.
 */
angular.module('sb.task', ['ui.router'])
    .config(function ($stateProvider) {
        'use strict';

        $stateProvider
            .state('sb.task', {
                url: '/task/{taskId:[0-9]+}',
                resolve: {
                    redirect: function (Task, $stateParams, $q, $state) {
                        Task.get({
                            id: $stateParams.taskId
                        }).$promise.then(function (task) {
                            $state.go('sb.story.detail',
                                      {storyId: task.story_id});
                        });
                    }
                }
            });
    }
);

/*
 * Copyright (c) 2013 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

angular.module('sb.util', ['ui.router', 'LocalStorageModule'])
    .run(function () {
        'use strict';
        angular.element.prototype.hide = function () {
            this.addClass('ng-hide');
        };

        angular.element.prototype.show = function () {
            this.removeClass('ng-hide');
        };
    });

/**
 * Copyright (c) 2015 Codethink Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The StoryBoard story submodule handles most activity surrounding the
 * creation and management of stories, their tasks, and comments.
 */
angular.module('sb.worklist',
              ['ui.router', 'sb.services', 'sb.util',
    'ui.bootstrap'])
    .config(function ($stateProvider, $urlRouterProvider) {
        'use strict';

        // URL Defaults.
        $urlRouterProvider.when('/worklist', '/worklist/list');

        // Set our page routes.
        $stateProvider
            .state('sb.worklist', {
                abstract: true,
                url: '/worklist',
                template: '<div ui-view></div>'
            })
            .state('sb.worklist.detail', {
                url: '/{worklistID:[0-9]+}',
                controller: 'WorklistDetailController',
                templateUrl: 'app/worklists/template/detail.html',
                resolve: {
                    worklist: function (Worklist, $stateParams) {
                        // Pre-resolve the worklist.
                        return Worklist.get({
                            id: $stateParams.worklistID
                        }).$promise;
                    },
                    permissions: function(Worklist, $stateParams) {
                        // Pre-resolve the permissions.
                        return Worklist.Permissions.get({
                            id: $stateParams.worklistID
                        }).$promise;
                    }
                }
            });
    });

/*
 * Copyright (c) 2016 Codethink Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Administration controller for teams.
 */
angular.module('sb.admin').controller('TeamAdminController',
    function ($scope, $modal, Team, Preference) {
        'use strict';

        /**
         * The teams.
         *
         * @type {Array}
         */
        $scope.teams = [];

        /**
         * The search filter query string.
         *
         * @type {string}
         */
        $scope.filterQuery = '';

        /**
         * Launches the add-add-team modal.
         */
        $scope.addTeam = function () {
            $modal.open({
                templateUrl: 'app/admin/template/team_new.html',
                backdrop: 'static',
                controller: 'TeamNewController'
            }).result.then(function () {
                // On success, reload the page.
                $scope.search();
            });
        };

        /**
         * Execute a search.
         */
        var pageSize = Preference.get('page_size');
        $scope.searchOffset = 0;
        $scope.search = function () {
            var searchQuery = $scope.filterQuery || '';

            $scope.teams = Team.browse({
                name: searchQuery,
                offset: $scope.searchOffset,
                limit: pageSize
            }, function(results, headers) {
                $scope.searchTotal =
                    parseInt(headers('X-Total')) || results.length;
                $scope.searchOffset = parseInt(headers('X-Offset')) || 0;
                $scope.searchLimit = parseInt(headers('X-Limit')) || 0;
            });
        };

        /**
         * Update the page size preference and re-search.
         */
        $scope.updatePageSize = function (value) {
            Preference.set('page_size', value).then(
                function () {
                    pageSize = value;
                    $scope.search();
                }
            );
        };

        /**
         * Next page of the results.
         */
        $scope.nextPage = function () {
            $scope.searchOffset += pageSize;
            $scope.search();
        };

        /**
         * Previous page of the results.
         */
        $scope.previousPage = function () {
            $scope.searchOffset -= pageSize;
            if ($scope.searchOffset < 0) {
                $scope.searchOffset = 0;
            }
            $scope.search();
        };

        // Initialize
        $scope.search();
    });

/*
 * Copyright (c) 2016 Codethink Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Edit/view team controller
 */
angular.module('sb.admin').controller('TeamEditController',
    function($scope, team, members, projects, $state, Team, User, Project,
             DSCacheFactory, storyboardApiBase, $q) {
        'use strict';

        $scope.team = team;
        $scope.members = members;
        $scope.projects = projects;
        $scope.editing = false;
        $scope.isUpdating = false;

        $scope.save = function() {
            $scope.isUpdating = true;
            $scope.team.$update(function(updated) {
                DSCacheFactory.get('defaultCache').put(
                    storyboardApiBase + '/teams/' + $scope.team.id,
                    updated);
                $scope.isUpdating = false;
                $scope.editing = false;
            });
        };

        var oldName = $scope.team.name;
        var oldSecurity = $scope.team.security;
        $scope.toggleEdit = function() {
            if (!$scope.editing) {
                oldName = $scope.team.name;
                oldSecurity = $scope.team.security;
            } else if ($scope.editing) {
                $scope.team.name = oldName;
                $scope.team.security = oldSecurity;
            }
            $scope.editing = !$scope.editing;
        };

        $scope.toggleAddMember = function() {
            $scope.adding = !$scope.adding;
        };

        $scope.addUser = function(model) {
            $scope.members.push(model);
            Team.UsersController.create({
                team_id: $scope.team.id,
                user_id: model.id
            }, function() {
                DSCacheFactory.get('defaultCache').remove(
                    storyboardApiBase + '/teams/' +
                    $scope.team.id + '/users');
            });
        };

        $scope.removeUser = function(user) {
            var idx = $scope.members.indexOf(user);
            $scope.members.splice(idx, 1);
            Team.UsersController.delete({
                team_id: $scope.team.id,
                user_id: user.id
            }, function() {
                DSCacheFactory.get('defaultCache').remove(
                    storyboardApiBase + '/teams/' +
                    $scope.team.id + '/users');
            });
        };

        /**
         * User typeahead search method.
         */
        $scope.searchUsers = function(value) {
            var memberIds = $scope.members.map(function(user) {
                return user.id;
            });
            var deferred = $q.defer();

            User.browse({full_name: value, limit: 10},
                function(searchResults) {
                    var results = [];
                    angular.forEach(searchResults, function(result) {
                        if (memberIds.indexOf(result.id) === -1) {
                            results.push(result);
                        }
                    });
                    deferred.resolve(results);
                }
            );
            return deferred.promise;
        };

        $scope.toggleAddProject = function() {
            $scope.addingProject = !$scope.addingProject;
        };

        $scope.addProject = function(model) {
            $scope.projects.push(model);
            Team.ProjectsController.create({
                team_id: $scope.team.id,
                project_id: model.id
            }, function() {
                DSCacheFactory.get('defaultCache').remove(
                    storyboardApiBase + '/teams/' +
                    $scope.team.id + '/projects');
            });
        };

        $scope.removeProject = function(project) {
            var idx = $scope.projects.indexOf(project);
            $scope.projects.splice(idx, 1);
            Team.ProjectsController.delete({
                team_id: $scope.team.id,
                project_id: project.id
            }, function() {
                DSCacheFactory.get('defaultCache').remove(
                    storyboardApiBase + '/teams/' +
                    $scope.team.id + '/projects');
            });
        };

        /**
         * Project typeahead search method.
         */
        $scope.searchProjects = function(value) {
            var projectIds = $scope.projects.map(function(project) {
                return project.id;
            });
            var deferred = $q.defer();

            Project.browse({name: value, limit: 10},
                function(searchResults) {
                    var results = [];
                    angular.forEach(searchResults, function(result) {
                        if (projectIds.indexOf(result.id) === -1) {
                            results.push(result);
                        }
                    });
                    deferred.resolve(results);
                }
            );
            return deferred.promise;
        };
    });

/*
 * Copyright (c) 2016 Codethink Ltd
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * New Team modal controller.
 */
angular.module('sb.admin').controller('TeamNewController',
    function ($log, $scope, $modalInstance, Team, User, Project, $q) {
        'use strict';

        /**
         * Flag for the UI to indicate that we're saving.
         *
         * @type {boolean}
         */
        $scope.isSaving = false;

        /**
         * The new team.
         *
         * @type {Team}
         */
        $scope.team = new Team();

        /**
         * The members of the new team.
         *
         * @type {[User]}
         */
        $scope.members = [];

        /**
         * The projects related to the new team.
         *
         * @type {[Project]}
         */
        $scope.projects = [];

        /**
         * Saves the team
         */
        $scope.save = function () {
            $scope.isSaving = true;

            // Create a new team
            $scope.team.$create(function(team) {
                var users = [];
                angular.forEach($scope.members, function(member) {
                    users.push(Team.UsersController.create({
                        team_id: team.id,
                        user_id: member.id
                    }).$promise);
                });
                $q.all(users).then(function() {
                    $modalInstance.close(team);
                });
            }, function (error) {
                $scope.isSaving = false;
                $log.error(error);
                $modalInstance.dismiss(error);
            });
        };

        /**
         * Close this modal without saving.
         */
        $scope.close = function () {
            $modalInstance.dismiss('cancel');
        };

        /**
         * Toggle the "Add Member" text box.
         */
        $scope.toggleAddMember = function() {
            $scope.addingMember = !$scope.addingMember;
        };

        /**
         * Add a user to the list of members.
         */
        $scope.addUser = function(model) {
            $scope.members.push(model);
        };

        /**
         * Remove a user from the list of members.
         */
        $scope.removeUser = function(user) {
            var idx = $scope.members.indexOf(user);
            $scope.members.splice(idx, 1);
        };

        /**
         * User typeahead search method.
         */
        $scope.searchUsers = function(value) {
            var memberIds = $scope.members.map(function(user) {
                return user.id;
            });
            var deferred = $q.defer();

            User.browse({full_name: value, limit: 10},
                function(searchResults) {
                    var results = [];
                    angular.forEach(searchResults, function(result) {
                        if (memberIds.indexOf(result.id) === -1) {
                            results.push(result);
                        }
                    });
                    deferred.resolve(results);
                }
            );
            return deferred.promise;
        };

        $scope.toggleAddProject = function() {
            $scope.addingProject = !$scope.addingProject;
        };

        $scope.addProject = function(model) {
            $scope.projects.push(model);
        };

        $scope.removeProject = function(project) {
            var idx = $scope.projects.indexOf(project);
            $scope.projects.splice(idx, 1);
        };

        /**
         * Project typeahead search method.
         */
        $scope.searchProjects = function(value) {
            var projectIds = $scope.projects.map(function(project) {
                return project.id;
            });
            var deferred = $q.defer();

            Project.browse({name: value, limit: 10},
                function(searchResults) {
                    var results = [];
                    angular.forEach(searchResults, function(result) {
                        if (projectIds.indexOf(result.id) === -1) {
                            results.push(result);
                        }
                    });
                    deferred.resolve(results);
                }
            );
            return deferred.promise;
        };
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Administration controller for users.
 */
angular.module('sb.admin').controller('UserAdminController',
    function ($scope, $modal, User, Preference) {
        'use strict';

        /**
         * The project groups.
         *
         * @type {Array}
         */
        $scope.users = [];

        /**
         * The search filter query string.
         *
         * @type {string}
         */
        $scope.filterQuery = '';

        /**
         * Launches the add-add-user modal.
         */
        $scope.addUser = function () {
            $modal.open(
                {
                    templateUrl: 'app/admin/template/user_new.html',
                    backdrop: 'static',
                    controller: 'UserNewController'
                }).result.then(function () {
                    // On success, reload the page.
                    $scope.search();
                });
        };

        /**
         * Execute a search.
         */
        var pageSize = Preference.get('page_size');
        $scope.searchOffset = 0;
        $scope.search = function () {
            var searchQuery = $scope.filterQuery || '';

            $scope.users = User.browse({
                full_name: searchQuery,
                offset: $scope.searchOffset,
                limit: pageSize
            }, function(results, headers) {
                $scope.searchTotal =
                    parseInt(headers('X-Total')) || results.length;
                $scope.searchOffset = parseInt(headers('X-Offset')) || 0;
                $scope.searchLimit = parseInt(headers('X-Limit')) || 0;
            });
        };

        /**
         * Update the page size preference and re-search.
         */
        $scope.updatePageSize = function (value) {
            Preference.set('page_size', value).then(
                function () {
                    pageSize = value;
                    $scope.search();
                }
            );
        };

        /**
         * Next page of the results.
         */
        $scope.nextPage = function () {
            $scope.searchOffset += pageSize;
            $scope.search();
        };

        /**
         * Previous page of the results.
         */
        $scope.previousPage = function () {
            $scope.searchOffset -= pageSize;
            if ($scope.searchOffset < 0) {
                $scope.searchOffset = 0;
            }
            $scope.search();
        };

        // Initialize
        $scope.search();
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Edit user Controller.
 */
angular.module('sb.admin').controller('UserEditController',
    function ($scope, user, $state) {
        'use strict';

        $scope.user = user;

        $scope.save = function () {
            /**
             * Delete the email field to avoid trying to save a
             * user with a blank email address.
             */
            if (!$scope.user.email) {
                delete $scope.user.email;
            }
            $scope.user.$update(function () {
                $state.go('sb.admin.user');
            });
        };

        $scope.cancel = function () {
            $state.go('sb.admin.user');
        };
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * New User modal controller.
 */
angular.module('sb.admin').controller('UserNewController',
    function ($log, $scope, $modalInstance, User) {
        'use strict';

        /**
         * Flag for the UI to indicate that we're saving.
         *
         * @type {boolean}
         */
        $scope.isSaving = false;

        /**
         * The new user.
         *
         * @type {User}
         */
        $scope.user = new User();

        /**
         * Saves the user group
         */
        $scope.save = function () {
            $scope.isSaving = true;

            // Create a new user group
            $scope.user.$create(function (user) {
                $modalInstance.close(user);
            }, function (error) {
                $scope.isSaving = false;
                $log.error(error);
                $modalInstance.dismiss(error);
            });
        };

        /**
         * Close this modal without saving.
         */
        $scope.close = function () {
            $modalInstance.dismiss('cancel');
        };
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/*
 * This controller is responsible for getting an authorization code
 * having a state and an openid.
 *
 * @author Nikita Konovalov
 */

angular.module('sb.auth').controller('AuthAuthorizeController',
    function ($stateParams, $state, $log, OpenId) {
        'use strict';

        // First, check for the edge case where the API returns an error code
        // back to us. This should only happen when it fails to properly parse
        // our redirect_uri and thus just sends the error back to referrer, but
        // we should still catch it.
        if (!!$stateParams.error) {
            $log.debug('Error received, redirecting to auth.error.');
            $state.go('sb.auth.error', $stateParams);
            return;
        }

        // We're not an error, let's fire the authorization.
        OpenId.authorize();
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * This controller deauthorizes the session and destroys all tokens.
 */

angular.module('sb.auth').controller('AuthDeauthorizeController',
    function (Session, $state, $log) {
        'use strict';

        $log.debug('Logging out');
        Session.destroySession();
        $state.go('sb.index');
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * View controller for authorization error conditions.
 */
angular.module('sb.auth').controller('AuthErrorController',
    function ($scope, $stateParams) {
        'use strict';

        $scope.error = $stateParams.error || 'Unknown';
        $scope.errorDescription = $stateParams.error_description ||
            'No description received from server.';
    });

/*
 * Copyright (c) 2014 Mirantis Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/*
 * This controller is responsible for getting an access_token and
 * a refresh token having an authorization_code.
 *
 * @author Nikita Konovalov
 */

angular.module('sb.auth').controller('AuthTokenController',
    function ($state, $log, OpenId, Session, $searchParams, UrlUtil,
        LastLocation) {
        'use strict';

        // First, check for the edge case where the API returns an error code
        // back to us. This should only happen when it fails to properly parse
        // our redirect_uri and thus just sends the error back to referrer, but
        // we should still catch it.
        if (!!$searchParams.error) {
            $log.debug('Error received, redirecting to auth.error.');
            $state.go('sb.auth.error', $searchParams);
            return;
        }

        // Looks like there's no error, so let's see if we can resolve a token.
        // TODO: Finish implementing.
        OpenId.token($searchParams)
            .then(
            function (token) {
                Session.updateSession(token)
                    .then(function () {
                        LastLocation.go('sb.page.about', {});
                    });
            },
            function (error) {
                Session.destroySession();
                $state.go('sb.auth.error', error);
            }
        );
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A simple modal controller for the login require modal.
 */

angular.module('sb.auth').controller('LoginRequiredModalController',
    function ($state, $scope, $modalInstance) {
        'use strict';

        $scope.login = function () {
            $state.go('sb.auth.authorize');
            $modalInstance.dismiss('success');
        };

        $scope.close = function () {
            $modalInstance.dismiss('cancel');
        };
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Permission provider, which hides particular controls based on whether the
 * passed permission flag has been set.
 */
angular.module('sb.util').directive('permission',
    function ($log, PermissionManager) {
        'use strict';

        return {
            restrict: 'A',
            link: function ($scope, element, attrs) {
                // Start by hiding it.
                element.hide();

                var permName = attrs.permission;
                var permValue = attrs.permissionValue || true;

                PermissionManager.listen($scope, permName,
                    function (actualValue) {

                        if (!!actualValue &&
                            actualValue.toString() === permValue.toString()) {
                            element.show();
                        } else {
                            element.hide();
                        }
                    }
                );
            }
        };
    });

/*
 * Copyright (c) 2016 Hewlett Packard Enterprise Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * This $http interceptor ensures that the OAuth Token is always valid, fresh,
 * and reissued when needed.
 */
angular.module('sb.auth').factory('httpOAuthTokenInterceptor',
    function (AccessToken, $injector, $q, $log, $window) {

        'use strict';

        /**
         * Promise deferral for when we're in-flight of a refresh token request.
         *
         * @type {Promise}
         */
        var refreshPromise = null;

        /**
         * Returns the current access token, if available.
         * @returns {*} The access token.
         */
        function getCurrentToken() {
            return $q.when({
                type: AccessToken.getTokenType() || null,
                value: AccessToken.getAccessToken() || null,
                expired: AccessToken.isExpired() || AccessToken.expiresSoon(),
                refresh: AccessToken.getRefreshToken() || null
            });
        }

        /**
         * This method checks a token (see above) to see whether it needs to
         * be refreshed. If yes, it will refresh that token, and return a
         * promise that will update the token in the promise chain.
         * Otherwise, it will simply pass the token along to the header
         * decorator.
         *
         * @param {{}} token The token to check.
         * @returns {Promise}
         */
        function refreshIfNeeded(token) {
            // If it's not expired, just pass it on.
            if (!token.expired) {
                return $q.when(token);
            }

            // If we don't have a refresh token, pass it on.
            if (!token.refresh) {
                return $q.when(token);
            }

            // If the refresh promise is already in flight, return that. We
            // don't want to get caught in a situation where we try to refresh
            // more than once.
            if (refreshPromise) {
                $log.debug('Returning in-flight refresh promise.');
                return refreshPromise;
            }

            // We have to refresh.
            $log.debug('Attempting to refresh auth token.');
            var deferred = $q.defer();
            refreshPromise = deferred.promise;

            // Issue a refresh token request. We have to manually grab the
            // service from the injector here, because else we'd have a
            // circular injection dependency.
            var OpenId = $injector.get('OpenId');
            OpenId.token({
                grant_type: 'refresh_token',
                refresh_token: token.refresh
            }).then(
                function (data) {
                    AccessToken.setToken(data);
                },
                function () {
                    AccessToken.clear();
                    $window.location.reload();
                }
            ).finally(function () {
                // Inject the token, whether or not it exists, back into the
                // promise chain.
                deferred.resolve(getCurrentToken());

                // Clear the promise;
                refreshPromise = null;
            });

            return deferred.promise;
        }

        return {
            /**
             * The request interceptor ensures that the OAuth token, if
             * available, is attached to the HTTP request.
             *
             * @param request The request.
             * @returns {*} A promise that will resolve once we're sure we
             *              have a good token.
             */
            request: function (httpConfig) {
                /**
                 * Decorate the request with the header token, IFF it's
                 * available.
                 *
                 * @param token The token received from the promise chain.
                 * @returns {Promise} A promise that resolves the httpconfig
                 */
                function decorateHeader(token) {
                    if (token.type && token.value) {
                        httpConfig.headers.Authorization =
                            token.type + ' ' + token.value;
                    }
                    return $q.when(httpConfig);
                }

                // Are we an OpenID request? These get skipped.
                if(httpConfig.url.indexOf('/openid/') > -1) {
                    return $q.when(httpConfig);
                }

                // Build the interceptor promise chain for each request.
                return $q.when(getCurrentToken())
                    .then(refreshIfNeeded)
                    .then(decorateHeader);
            }
        };
    })
    // Attach the HTTP interceptor.
    .config(function ($httpProvider) {
        'use strict';

        $httpProvider.interceptors.unshift('httpOAuthTokenInterceptor');
    });

/*
 * Copyright (c) 2014 Mirantis Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

angular.module('sb.auth').run(
    function($log, $modal, Notification, Session, Priority) {
        'use strict';

        function handle_403() {
            var modalInstance = $modal.open({
                templateUrl: 'app/auth/template/modal/superuser_required.html',
                backdrop: 'static',
                controller: function($modalInstance, $scope) {
                    $scope.close = function () {
                        $modalInstance.dismiss('cancel');
                    };
                }
            });
            return modalInstance.result;
        }


        // We're using -1 as the priority, to ensure that this is
        // intercepted before anything else happens.
        Notification.intercept(function (message) {
            if (message.type === 'http') {
                if (message.message === 403) {
                    // Forbidden error. A user should be warned tha he is
                    // doing something wrong.
                    handle_403();
                }

                return false; // Stop processing this notifications.
            }
        }, Priority.BEFORE);

    }
);

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the 'License'); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

angular.module('sb.auth').run(
    function ($log, $modal, Notification, SessionState, Priority) {
        'use strict';


        // We're using -1 as the priority, to ensure that this is
        // intercepted before anything else happens.
        Notification.intercept(function (message) {

            switch (message.type) {
                case SessionState.LOGGED_IN:
                    // Logged in messages get filtered out.
                    return true;
                case SessionState.LOGGED_OUT:
                    message.message = 'You have been logged out.';
                    break;
                default:
                    break;
            }
        }, Priority.AFTER);

    }
);

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A list of constants used by the session service to maintain the user's
 * current authentication state.
 */
angular.module('sb.auth').value('SessionState', {

    /**
     * Session state constant, used to indicate that the user is logged in.
     */
    LOGGED_IN: 'logged_in',

    /**
     * Session state constant, used to indicate that the user is logged out.
     */
    LOGGED_OUT: 'logged_out',

    /**
     * Session state constant, used during initialization when we're not quite
     * certain yet whether we're logged in or logged out.
     */
    PENDING: 'pending'

});

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The permission resolver allows us to require certain permissions for specific
 * UI routes.
 */
angular.module('sb.auth').constant('PermissionResolver',
    {
        /**
         * Rejects the route if the current user does not have the required
         * permission.
         */
        requirePermission: function (permName, requiredValue) {
            'use strict';

            return function ($q, $log, PermissionManager) {
                var deferred = $q.defer();

                PermissionManager.resolve(permName).then(
                    function (value) {
                        $log.debug('permission:', permName, requiredValue,
                            value);
                        if (value === requiredValue) {
                            deferred.resolve(value);
                        } else {
                            deferred.reject(value);
                        }
                    },
                    function (error) {
                        $log.debug('permission:', error);
                        deferred.reject(error);
                    }
                );

                return deferred.promise;
            };

        },

        /**
         * Resolves the value of the provided permission.
         */
        resolvePermission: function (permName) {
            'use strict';

            return function ($q, $log, PermissionManager) {
                var deferred = $q.defer();

                PermissionManager.resolve(permName).then(
                    function (value) {
                        deferred.resolve(value);
                    },
                    function () {
                        deferred.resolve(false);
                    }
                );

                return deferred.promise;
            };

        }
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A set of utility methods that may be used during state declaration to enforce
 * session state. They return asynchronous promises which will either resolve
 * or reject the state change, depending on what you're asking for.
 */
angular.module('sb.auth').constant('SessionResolver',
    (function () {
        'use strict';

        /**
         * Resolve the promise based on the current session state. We can't
         * inject here, since the injector's not ready yet.
         */
        function resolveSessionState(deferred, desiredSessionState, Session) {
            return function () {
                var sessionState = Session.getSessionState();
                if (sessionState === desiredSessionState) {
                    deferred.resolve(sessionState);
                } else {
                    deferred.reject(sessionState);
                }
            };
        }

        return {
            /**
             * This resolver simply checks to see whether a user is logged
             * in or not, and returns the session state.
             */
            resolveSessionState: function ($q, $log, Session, SessionState) {
                var deferred = $q.defer();

                $log.debug('Resolving session state...');
                Session.resolveSessionState().then(
                    function (sessionState) {
                        deferred.resolve(sessionState);
                    },
                    function (error) {
                        $log.error(error);
                        deferred.resolve(SessionState.LOGGED_OUT);
                    }
                );

                return deferred.promise;
            },

            /**
             * This resolver asserts that the user is logged
             * out before allowing a route. Otherwise it fails.
             */
            requireLoggedOut: function ($q, $log, Session, SessionState) {

                $log.debug('Resolving logged-out-only route...');
                var deferred = $q.defer();
                var resolveLoggedOut = resolveSessionState(deferred,
                    SessionState.LOGGED_OUT, Session);

                // Do we have to wait for state resolution?
                if (Session.getSessionState() === SessionState.PENDING) {
                    Session.resolveSessionState().then(resolveLoggedOut);
                } else {
                    resolveLoggedOut();
                }

                return deferred.promise;
            },

            /**
             * This resolver asserts that the user is logged
             * in before allowing a route. Otherwise it fails.
             */
            requireLoggedIn: function ($q, $log, Session, $rootScope,
                                       SessionState) {

                $log.debug('Resolving logged-in-only route...');
                var deferred = $q.defer();
                var resolveLoggedIn = resolveSessionState(deferred,
                    SessionState.LOGGED_IN, Session);

                // Do we have to wait for state resolution?
                if (Session.getSessionState() === SessionState.PENDING) {
                    Session.resolveSessionState().then(resolveLoggedIn);
                } else {
                    resolveLoggedIn();
                }

                return deferred.promise;
            },

            /**
             * This resolver ensures that the currentUser has been resolved
             * before the route resolves.
             */
            requireCurrentUser: function ($q, $log, CurrentUser) {
                $log.debug('Resolving current user...');
                return CurrentUser.resolve();
            }
        };
    })());

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * AccessToken storage service, an abstraction layer between our token storage
 * and the rest of the system. This feature uses localStorage, which means that
 * our application will NOT support IE7. Once that becomes a requirement, we'll
 * have to use this abstraction layer to store data in a cookie instead.
 */
angular.module('sb.auth').factory('AccessToken',
    function (localStorageService, preExpireDelta) {
        'use strict';

        /**
         * Our local storage key name constants
         */
        var TOKEN_TYPE = 'token_type';
        var ACCESS_TOKEN = 'access_token';
        var REFRESH_TOKEN = 'refresh_token';
        var ID_TOKEN = 'id_token';
        var EXPIRES_IN = 'expires_in';
        var ISSUE_DATE = 'issue_date';

        return {

            /**
             * Clears the token
             */
            clear: function () {
                localStorageService.remove(TOKEN_TYPE);
                localStorageService.remove(ACCESS_TOKEN);
                localStorageService.remove(REFRESH_TOKEN);
                localStorageService.remove(ID_TOKEN);
                localStorageService.remove(EXPIRES_IN);
                localStorageService.remove(ISSUE_DATE);
            },

            /**
             * Sets all token properties at once.
             */
            setToken: function (jsonToken) {
                this.setTokenType(jsonToken.token_type);
                this.setAccessToken(jsonToken.access_token);
                this.setRefreshToken(jsonToken.refresh_token);
                this.setIdToken(jsonToken.id_token);
                this.setIssueDate(jsonToken.issue_date);
                this.setExpiresIn(jsonToken.expires_in);
            },

            /**
             * Is the current access token expired?
             */
            isExpired: function () {
                var expiresIn = this.getExpiresIn() || 0;
                var issueDate = this.getIssueDate() || 0;
                var now = Math.round((new Date()).getTime() / 1000);

                return issueDate + expiresIn < now;
            },

            /**
             * Will this token expire in an hour
             */
            expiresSoon: function () {
                var expiresIn = this.getExpiresIn() || 0;
                var issueDate = this.getIssueDate() || 0;
                var now = Math.round((new Date()).getTime() / 1000);

                return issueDate + expiresIn - preExpireDelta < now;
            },

            /**
             * Get the token type. Bearer, etc.
             */
            getTokenType: function () {
                return localStorageService.get(TOKEN_TYPE);
            },

            /**
             * Set the token type.
             */
            setTokenType: function (value) {
                return localStorageService.set(TOKEN_TYPE, value);
            },

            /**
             * Retrieve the date this token was issued.
             */
            getIssueDate: function () {
                return parseInt(localStorageService.get(ISSUE_DATE)) || null;
            },

            /**
             * Set the issue date for the current access token.
             */
            setIssueDate: function (value) {
                return localStorageService.set(ISSUE_DATE, parseInt(value));
            },

            /**
             * Get the number of seconds after the issue date when this token
             * is considered expired.
             */
            getExpiresIn: function () {
                return parseInt(localStorageService.get(EXPIRES_IN)) || 0;
            },

            /**
             * Set the number of seconds from the issue date when this token
             * will expire.
             */
            setExpiresIn: function (value) {
                return localStorageService.set(EXPIRES_IN, parseInt(value));
            },

            /**
             * Retrieve the access token.
             */
            getAccessToken: function () {
                return localStorageService.get(ACCESS_TOKEN) || null;
            },

            /**
             * Set the access token.
             */
            setAccessToken: function (value) {
                return localStorageService.set(ACCESS_TOKEN, value);
            },

            /**
             * Retrieve the refresh token.
             */
            getRefreshToken: function () {
                return localStorageService.get(REFRESH_TOKEN) || null;
            },

            /**
             * Set the refresh token.
             */
            setRefreshToken: function (value) {
                return localStorageService.set(REFRESH_TOKEN, value);
            },

            /**
             * Retrieve the id token.
             */
            getIdToken: function () {
                return localStorageService.get(ID_TOKEN) || null;
            },

            /**
             * Set the id token.
             */
            setIdToken: function (value) {
                return localStorageService.set(ID_TOKEN, value);
            }
        };
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The current user service. It pays attention to changes in the application's
 * session state, and loads the user found in the AccessToken when a valid
 * session is detected.
 */
angular.module('sb.auth').factory('CurrentUser',
    function (SessionState, Session, AccessToken, $rootScope, $log, $q, User,
              Notification, Priority) {
        'use strict';

        /**
         * The current user
         */
        var currentUser = null;
        var currentPromise = null;

        /**
         * A promise that only resolves if we're currently logged in.
         */
        function resolveLoggedInSession() {
            var deferred = $q.defer();

            Session.resolveSessionState().then(
                function (sessionState) {

                    if (sessionState === SessionState.LOGGED_IN) {
                        deferred.resolve(sessionState);
                    } else {
                        deferred.reject(sessionState);
                    }
                },
                function (error) {
                    deferred.reject(error);
                }
            );

            return deferred.promise;
        }

        /**
         * Resolve a current user.
         */
        function resolveCurrentUser() {
            // If we've got an in-flight promise, just return that and let
            // the consumers chain off of that.
            if (!!currentPromise) {
                return currentPromise;
            }

            // Construct a new resolution promise.
            var deferred = $q.defer();
            currentPromise = deferred.promise;

            // Make sure we have a logged-in session.
            resolveLoggedInSession().then(
                function () {
                    // Now that we know we're logged in, do we have a
                    // currentUser yet?
                    if (!!currentUser) {
                        deferred.resolve(currentUser);
                    } else {
                        // Ok, we have to load.
                        User.get(
                            {
                                id: AccessToken.getIdToken()
                            },
                            function (user) {
                                currentUser = user;
                                deferred.resolve(user);
                            },
                            function (error) {
                                currentUser = null;
                                deferred.reject(error);
                            }
                        );
                    }
                },
                function (error) {
                    currentUser = null;
                    deferred.reject(error);
                }
            );

            // Chain a resolution that'll make the currentPromise clear itself.
            currentPromise.then(
                function () {
                    currentPromise = null;
                },
                function () {
                    currentPromise = null;
                }
            );

            return currentPromise;
        }

        // Add event listeners.
        Notification.intercept(function (message) {
            switch (message.type) {
                case SessionState.LOGGED_IN:
                    resolveCurrentUser();
                    break;
                case SessionState.LOGGED_OUT:
                    currentUser = null;
                    break;
                default:
                    break;
            }
        }, Priority.LAST);

        // Expose the methods for this service.
        return {

            /**
             * Resolves the current user with a promise.
             */
            resolve: function () {
                return resolveCurrentUser();
            }
        };
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Our OpenID token resource, which adheres to the OpenID connect specification
 * found here; http://openid.net/specs/openid-connect-basic-1_0.html
 */
angular.module('sb.auth').factory('OpenId',
    function ($location, $window, $log, $http, $q, StringUtil, UrlUtil,
              storyboardApiBase, localStorageService) {
        'use strict';

        var storageKey = 'openid_authorize_state';
        var authorizeUrl = storyboardApiBase + '/openid/authorize';
        var tokenUrl = storyboardApiBase + '/openid/token';
        var redirectUri = UrlUtil.buildApplicationUrl('/auth/token');
        var clientId = $location.host();

        return {
            /**
             * Asks the OAuth endpoint for an authorization token given
             * the passed parameters.
             */
            authorize: function () {
                // Create and store a random state parameter.
                var state = StringUtil.randomAlphaNumeric(20);
                localStorageService.set(storageKey, state);

                var openIdParams = {
                    response_type: 'code',
                    client_id: clientId,
                    redirect_uri: redirectUri,
                    scope: 'user',
                    state: state
                };

                $window.location.href = authorizeUrl + '?' +
                    UrlUtil.serializeParameters(openIdParams);
            },

            /**
             * Asks our OpenID endpoint to convert an authorization code or a
             * refresh token to an access token.
             */
            token: function (params) {
                var deferred = $q.defer();
                var grant_type = params.grant_type || 'authorization_code';
                var authorizationCode = params.code;
                var refreshToken = params.refresh_token;

                var tokenParams = {
                    grant_type: grant_type
                };

                if (grant_type === 'authorization_code') {
                    tokenParams.code = authorizationCode;
                } else {
                    tokenParams.refresh_token = refreshToken;
                }

                var url = tokenUrl + '?' +
                    UrlUtil.serializeParameters(tokenParams);

                $http({method: 'POST', url: url})
                    .then(function (response) {
                        $log.debug('Token creation succeeded.');
                        // Extract the data
                        var data = response.data;

                        // Derive an issue date, from the Date header if
                        // possible.
                        var dateHeader = response.headers('Date');
                        if (!dateHeader) {
                            data.issue_date = Math.floor(Date.now() / 1000);
                        } else {
                            data.issue_date = Math.floor(
                                new Date(dateHeader) / 1000
                            );
                        }

                        deferred.resolve(data);
                    },
                    function (response) {
                        $log.debug('Token creation failed.');

                        // Construct a conformant error response.
                        var error = response.data;
                        if (!error.hasOwnProperty('error')) {
                            error = {
                                error: response.status,
                                error_description: response.data
                            };
                        }
                        deferred.reject(error);
                    });

                return deferred.promise;
            }
        };
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * This service maintains our permission state while the client is running.
 * Rather than be based on a per-request basis whose responses can quickly
 * become stale, it broadcasts events which views/directives can use to
 * update their permissions.
 *
 * At the moment this is a fairly naive implementation, which assumes that
 * permissions are defined as key/value pairs, and are globally scoped.
 * For example, this is possible:
 *
 *      isSuperuser: true
 *
 * But this is not.
 *
 *      Project ID 4, canEdit: false
 *
 * We'll need to update this once we know what our permission structure
 * looks like.
 */
angular.module('sb.auth').factory('PermissionManager',
    function ($log, $q, $rootScope, Session, SessionState, CurrentUser,
              Notification, Priority) {
        'use strict';

        // Our permission resolution cache.
        var permCache = {};
        var NOTIFY_PERMISSIONS = 'notify_permissions';

        /**
         * Resolve a permission.
         */
        function resolvePermission(permName) {
            var deferred = $q.defer();

            if (permCache.hasOwnProperty(permName)) {
                deferred.resolve(permCache[permName]);
            } else {
                CurrentUser.resolve().then(
                    function (user) {
                        permCache[permName] = user[permName];
                        deferred.resolve(permCache[permName]);
                    },
                    function (error) {
                        deferred.reject(error);
                    }
                );
            }

            return deferred.promise;
        }

        /**
         * Clear the permission cache and notify the system that it needs
         * to re-resolve the permissions.
         */
        function clearPermissionCache() {
            $log.debug('Resetting permission cache.');
            permCache = {};
            $rootScope.$broadcast(NOTIFY_PERMISSIONS);
        }

        /**
         * Wrapper function which resolves the permission we're looking
         * for and then invokes the passed handler.
         */
        function permissionListenHandler(permName, handler) {
            return function () {
                resolvePermission(permName).then(
                    function (value) {
                        handler(value);
                    },
                    function () {
                        handler(null);
                    }
                );
            };
        }

        return {
            /**
             * Initialize the permission manager on the passed scope.
             */
            initialize: function () {
                $log.debug('Initializing permissions');


                // Always record the logged in state on the root scope.
                var removeNotifier = Notification.intercept(function (message) {
                    switch (message.type) {
                        case SessionState.LOGGED_IN:
                        case SessionState.LOGGED_OUT:
                            clearPermissionCache();
                            break;
                        default:
                            break;
                    }
                }, Priority.LAST);

                $rootScope.$on('$destroy', removeNotifier);

                // Run update if the session state has already resolved.
                // Otherwise wait for the above listeners.
                if (Session.getSessionState() !== SessionState.PENDING) {
                    clearPermissionCache();
                }
            },

            /**
             * Convenience method which allows a
             * @param scope The view scope that wants to listen to permission
             * changes.
             * @param permName The name of the permission.
             * @param handler The response handler
             */
            listen: function (scope, permName, handler) {
                var messageHandler = permissionListenHandler(permName, handler);

                scope.$on('$destroy',
                    $rootScope.$on(NOTIFY_PERMISSIONS, messageHandler)
                );

                // Trigger the handler once.
                messageHandler();
            },

            /**
             * Resolve a specific permission. Loads from a resolution cache
             * if this permission is currently unknown.
             */
            resolve: function (permName) {
                return resolvePermission(permName);
            }
        };
    });

/*
 * Copyright (c) 2014 Mirantis Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/*
 * A constant time in seconds when an access token is considered "expires soon"
 * default 10 minutes.
 */

angular.module('sb.auth').constant('preExpireDelta', 600);

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Session management service - keeps track of our current session state, mostly
 * by verifying the token state returned from the OpenID service.
 */
angular.module('sb.auth').factory('Session',
    function (SessionState, AccessToken, $rootScope, $log, $q, $state,
              SystemInfo, Notification, Severity) {
        'use strict';

        /**
         * The current session state.
         *
         * @type String
         */
        var sessionState = SessionState.PENDING;

        /**
         * Handles state updates and broadcasts.
         */
        function updateSessionState(newState) {
            if (newState !== sessionState) {
                sessionState = newState;
                Notification.send(newState, newState, Severity.SUCCESS);
            }
        }

        /**
         * Validate the token.
         */
        function validateToken() {

            /**
             * Try fresh call is necessary here because a User may try to
             * validate a token after a long break in using StoryBoard.
             * Even if refresh is not necessary right now the tryRefresh method
             * will just resolve immediately.
             */

            var deferred = $q.defer();
            return SystemInfo.get({},
                function (info) {
                    if (AccessToken.getAccessToken()) {
                        deferred.resolve(info);
                    } else {
                        deferred.reject(info);
                    }
                }).$promise;
        }

        /**
         * Initialize the session.
         */
        function initializeSession() {
            var deferred = $q.defer();

            if (!AccessToken.getAccessToken()) {
                $log.debug('No token found');
                updateSessionState(SessionState.LOGGED_OUT);
                deferred.resolve();
            } else {
                // Validate the token currently in the cache.
                validateToken()
                    .then(function () {
                        $log.debug('Token validated');
                        updateSessionState(SessionState.LOGGED_IN);
                        deferred.resolve(sessionState);
                    }, function () {
                        $log.debug('Token not validated');
                        AccessToken.clear();
                        updateSessionState(SessionState.LOGGED_OUT);
                        deferred.resolve(sessionState);
                    });
            }

            return deferred.promise;
        }

        /**
         * Destroy the session (Clear the token).
         */
        function destroySession() {
            AccessToken.clear();
            updateSessionState(SessionState.LOGGED_OUT);
            $state.reload();
        }

        /**
         * Initialize and test our current session token.
         */
        initializeSession();

        // Expose the methods for this service.
        return {
            /**
             * The current session state.
             */
            getSessionState: function () {
                return sessionState;
            },

            /**
             * Resolve the current session state, as a promise.
             */
            resolveSessionState: function () {
                var deferred = $q.defer();
                if (sessionState !== SessionState.PENDING) {
                    deferred.resolve(sessionState);
                } else {
                    var unwatch = $rootScope.$watch(function () {
                        return sessionState;
                    }, function () {
                        if (sessionState !== SessionState.PENDING) {
                            deferred.resolve(sessionState);
                            unwatch();
                        }
                    });
                }

                return deferred.promise;
            },

            /**
             * Are we logged in?
             */
            isLoggedIn: function () {
                return sessionState === SessionState.LOGGED_IN;
            },

            /**
             * Destroy the session.
             */
            destroySession: function () {
                destroySession();
            },

            /**
             * Update the session with a new (or null) token.
             */
            updateSession: function (token) {
                var deferred = $q.defer();
                if (!token) {
                    destroySession();
                    deferred.resolve(sessionState);
                } else {
                    AccessToken.setToken(token);
                    initializeSession().then(
                        function () {
                            deferred.resolve(sessionState);
                        },
                        function () {
                            deferred.resolve(sessionState);
                        }
                    );
                }

                return deferred.promise;
            }
        };
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Session Modal service, which provides us with some session/auth related
 * modals.
 */
angular.module('sb.auth')
    .factory('SessionModalService', function ($modal) {
        'use strict';

        return {

            /**
             * Show a modal that kindly tells our user that they should
             * log in first.
             */
            showLoginRequiredModal: function () {
                var modalInstance = $modal.open(
                    {
                        templateUrl: 'app/auth/template' +
                            '/modal/login_required.html',
                        backdrop: 'static',
                        controller: 'LoginRequiredModalController'
                    }
                );

                // Return the modal's promise.
                return modalInstance.result;
            }
        };
    }
);

/*
 * Copyright (c) 2015-2016 Codethink Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Controller for the "new board" modal popup.
 */
angular.module('sb.board').controller('AddBoardController',
    function ($scope, $modalInstance, $state, params, Board, Project,
              Worklist, $q, BoardHelper) {
        'use strict';

        /**
         * Create the new board.
         */
        function saveBoard() {
            $scope.board.$create(
                function (result) {
                    $scope.isSaving = false;
                    $modalInstance.dismiss('success');
                    $state.go('sb.board.detail', {boardID: result.id});
                }
            );
        }

        /**
         * Return a function which adds a "lane" to the board in the given
         * position. This lane is just a reference to the worklist used to
         * represent it.
         */
        function addLaneDetails(position) {
            return function(lane) {
                $scope.board.lanes.push({
                    board_id: $scope.board.id,
                    list_id: lane.id,
                    position: position
                });
            };
        }

        /**
         * Create worklists to represent the lanes of the board, then save
         * the board.
         */
        function saveBoardWithLanes() {
            $scope.board.lanes = [];
            var lanePromises = [];
            for (var i = 0; i < $scope.lanes.length; i++) {
                var lane = $scope.lanes[i];
                var addLane = addLaneDetails(i);
                lane.project_id = $scope.board.project_id;
                lane.private = $scope.board.private;
                lanePromises.push(lane.$create(addLane));
            }
            $q.all(lanePromises).then(saveBoard);
        }

        /**
         * Saves the board, and any worklists created to serve as lanes.
         */
        $scope.save = function() {
            $scope.isSaving = true;
            if ($scope.lanes.length > 0) {
                saveBoardWithLanes();
            } else {
                saveBoard();
            }
        };

        /**
         * Close this modal without saving.
         */
        $scope.close = function() {
            $modalInstance.dismiss('cancel');
        };

        /**
         * Add a lane.
         */
        $scope.addLane = function() {
            $scope.lanes.push(new Worklist({
                title: '',
                editing: true
            }));
        };

        /**
         * Remove a lane.
         */
        $scope.removeLane = function(lane) {
            var idx = $scope.lanes.indexOf(lane);
            $scope.lanes.splice(idx, 1);
        };

        /**
         * Toggle editing of a lane title.
         */
        $scope.toggleEdit = function(lane) {
            lane.editing = !lane.editing;
        };

        /**
         * Config for the lanes sortable.
         */
        $scope.lanesSortable = {
            dragMove: BoardHelper.maybeScrollContainer('new-board')
        };

        // Create a blank Board to begin with.
        $scope.isSaving = false;
        $scope.lanes = [];
        $scope.board = new Board({
            title: '',
            lanes: []
        });
    });

/*
 * Copyright (c) 2015 Codethink Limited
 *
 * Licensed under the Apache License, Version 2.0 (the 'License'); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Controller for "archive board" modal
 */
angular.module('sb.board').controller('BoardArchiveController',
    function ($log, $scope, $state, board, $modalInstance) {
        'use strict';

        $scope.board = board;

        // Set our progress flags and clear previous error conditions.
        $scope.isUpdating = true;
        $scope.error = {};

        $scope.remove = function () {
            $scope.board.$delete(
                function () {
                    $modalInstance.dismiss('success');
                    $state.go('sb.dashboard.boards');
                }
            );
        };

        $scope.close = function () {
            $modalInstance.dismiss('cancel');
        };
    });

/*
 * Copyright (c) 2015-2016 Codethink Limited
 * Copyright (c) 2019 Adam Coldrick
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A controller for a kanban board.
 */
angular.module('sb.board').controller('BoardDetailController',
    function ($scope, Worklist, $modal, Board, Project, $stateParams,
              BoardHelper, DueDate, $document, User, $q, moment) {
        'use strict';

        $scope.board = new Board();

        /**
         * Load the board. If onlyContents is true then assume $scope.board
         * is a board and reload its contents.
         */
        function loadBoard() {
            var params = {id: $stateParams.boardID};
            Board.Permissions.get(params, function(perms) {
                $scope.permissions = {
                    editBoard: perms.indexOf('edit_board') > -1,
                    moveCards: perms.indexOf('move_cards') > -1
                };
            });
            Board.get(params, function(board) {
                var offsets = BoardHelper.recordLaneScrollbars($scope.board);
                $scope.board = board;
                $scope.owners = [];
                $scope.users = [];
                angular.forEach(board.owners, function(id) {
                    $scope.owners.push(User.get({id: id}));
                });
                angular.forEach(board.users, function(id) {
                    $scope.users.push(User.get({id: id}));
                });
                BoardHelper.scrollLanes(board, offsets);
            });
        }

        /**
         * Show a modal to handle adding cards to a lane. Pass a validation
         * function to this modal which returns `false` if the card is already
         * in one of the lanes of the board.
         */
        function showAddItemModal(worklist) {
            var modalInstance = $modal.open({
                size: 'lg',
                templateUrl: 'app/worklists/template/additem.html',
                backdrop: 'static',
                controller: 'WorklistAddItemController',
                resolve: {
                    worklist: function() {
                        return worklist;
                    },
                    valid: function() {
                        var board = $scope.board;
                        return function(item) {
                            var valid = true;
                            angular.forEach(board.lanes, function(lane) {
                                var items = lane.worklist.items;
                                angular.forEach(items, function(listItem) {
                                    var type = item.type.toLowerCase();
                                    if (!item.hasOwnProperty('value')) {
                                        item.value = item.id;
                                    }
                                    if (item.value === listItem.item_id &&
                                        type === listItem.item_type) {
                                        valid = false;
                                        item.invalid = item.type +
                                                       ' is already in' +
                                                       ' the board (' +
                                                       lane.worklist.title +
                                                       ' lane).';
                                    }
                                });
                            });
                            return valid;
                        };
                    }
                }
            });

            return modalInstance.result;
        }

        /**
         * Add a card to a lane.
         */
        $scope.addItem = function(worklist) {
            loadBoard();
            showAddItemModal(worklist)
                .finally(function() {
                    loadBoard();
                });
        };

        /**
         * Toggle the edit form for the board title and description.
         */
        $scope.toggleEditMode = function() {
            if ($scope.showEditForm) {
                loadBoard();
            }
            $scope.showEditForm = !$scope.showEditForm;
        };

        /**
         * Save changes to the board.
         */
        $scope.update = function() {
            $scope.isUpdating = true;
            var params = {id: $scope.board.id};
            var owners = {
                codename: 'edit_board',
                users: $scope.board.owners
            };
            var users = {
                codename: 'move_cards',
                users: $scope.board.users
            };
            $scope.board.$update().then(function() {
                var updating = [
                    Board.Permissions.update(params, owners).$promise,
                    Board.Permissions.update(params, users).$promise
                ];
                $q.all(updating).then(function() {
                    $scope.isUpdating = false;
                    $scope.toggleEditMode();
                });
            });
        };

        $scope.unarchive = function() {
            $scope.board.archived = false;
            $scope.board.$update().then(function() {
                angular.forEach($scope.board.lanes, function(lane) {
                    lane.worklist.archived = false;
                    Worklist.update(lane.worklist);
                });
            });
        };

        /**
         * Open a modal to handle archiving the board.
         */
        $scope.remove = function() {
            var modalInstance = $modal.open({
                templateUrl: 'app/boards/template/archive.html',
                backdrop: 'static',
                controller: 'BoardArchiveController',
                resolve: {
                    board: function() {
                        return $scope.board;
                    }
                }
            });

            return modalInstance.result;
        };

        /**
         * Add a lane to the board.
         */
        $scope.addLane = function () {
            var modalInstance = $modal.open({
                size: 'lg',
                templateUrl: 'app/worklists/template/new.html',
                backdrop: 'static',
                controller: 'AddWorklistController',
                resolve: {
                    params: function() {
                        return {};
                    },
                    redirect: function() {
                        return false;
                    }
                }
            });

            modalInstance.result.then(function(worklist) {
                $scope.board.lanes.push({
                    list_id: worklist.id,
                    worklist: Worklist.get({id: worklist.id}),
                    position: $scope.board.lanes.length,
                    board_id: $scope.board.id
                });
                Board.update($scope.board);
            });
        };

        /**
         * Remove a lane from the board.
         */
        $scope.removeLane = function (lane) {
            var modalInstance = $modal.open({
                templateUrl: 'app/worklists/template/delete.html',
                backdrop: 'static',
                controller: 'WorklistDeleteController',
                resolve: {
                    worklist: function() {
                        return lane.worklist;
                    },
                    redirect: false
                }
            });

            modalInstance.result.then(function() {
                var idx = $scope.board.lanes.indexOf(lane);
                $scope.board.lanes.splice(idx, 1);
                $scope.board.$update();
            });
        };

        /**
         * Remove a card from a lane.
         */
        $scope.removeCard = function (worklist, item) {
            Worklist.ItemsController.delete({
                id: worklist.id,
                item_id: item.id
            }).$promise.then(function() {
                var idx = worklist.items.indexOf(item);
                worklist.items.splice(idx, 1);
            });
        };

        /**
         * Save changes to the ordering of and additions to the lanes.
         */
        function updateBoardLanes() {
            for (var i = 0; i < $scope.board.lanes.length; i++) {
                $scope.board.lanes[i].position = i;
            }
            Board.update($scope.board);
        }

        /**
         * Toggle edit mode on a lane. Save changes if toggling on->off, and
         * create a worklist to represent the lane if one doesn't exist
         * already.
         */
        $scope.editWorklist = function(worklist) {
            var modalInstance = $modal.open({
                size: 'lg',
                templateUrl: 'app/worklists/template/new.html',
                backdrop: 'static',
                controller: 'WorklistEditController',
                resolve: {
                    worklist: function() {
                        return worklist;
                    },
                    board: function() {
                        return $scope.board;
                    }
                }
            });

            modalInstance.result.finally(function() {
                loadBoard();
            });
        };

        /**
         * User typeahead search method.
         */
        $scope.searchUsers = function (value, array) {
            var deferred = $q.defer();

            User.browse({full_name: value, limit: 10},
                function(searchResults) {
                    var results = [];
                    angular.forEach(searchResults, function(result) {
                        if (array.indexOf(result.id) === -1) {
                            results.push(result);
                        }
                    });
                    deferred.resolve(results);
                }
            );
            return deferred.promise;
        };

        /**
         * Formats the user name.
         */
        $scope.formatUserName = function (model) {
            if (!!model) {
                return model.name;
            }
            return '';
        };

        /**
         * Add a new user to one of the permission levels.
         */
        $scope.addUser = function (model, modelArray, idArray) {
            idArray.push(model.id);
            modelArray.push(model);
        };

        /**
         * Remove a user from one of the permission levels.
         */
        $scope.removeUser = function (model, modelArray, idArray) {
            var idIdx = idArray.indexOf(model.id);
            idArray.splice(idIdx, 1);

            var modelIdx = modelArray.indexOf(model);
            modelArray.splice(modelIdx, 1);
        };

        $scope.newDueDate = function() {
            var modalInstance = $modal.open({
                templateUrl: 'app/due_dates/template/new.html',
                backdrop: 'static',
                controller: 'DueDateNewController',
                resolve: {
                    board: function() {
                        return $scope.board;
                    }
                }
            });

            modalInstance.result.finally(function() {
                loadBoard();
            });
        };

        $scope.editDueDate = function(dueDate) {
            var modalInstance = $modal.open({
                templateUrl: 'app/due_dates/template/new.html',
                backdrop: 'static',
                controller: 'DueDateEditController',
                resolve: {
                    board: function() {
                        return $scope.board;
                    },
                    dueDate: function() {
                        return dueDate;
                    }
                }
            });

            modalInstance.result.finally(function() {
                loadBoard();
            });
        };

        $scope.removeDueDate = function(dueDate) {
            var modalInstance = $modal.open({
                templateUrl: 'app/due_dates/template/remove_from_board.html',
                backdrop: 'static',
                controller: 'DueDateRemoveController',
                resolve: {
                    board: function() {
                        return $scope.board;
                    },
                    dueDate: function() {
                        return dueDate;
                    }
                }
            });

            modalInstance.result.finally(function() {
                loadBoard();
            });
        };

        $scope.isDue = function(card) {
            if (card.item_type === 'task') {
                if (card.task.status === 'merged') {
                    return false;
                }
            }
            var now = moment();
            var tomorrow = now.clone();
            tomorrow.add(1, 'day');
            if (!card.resolved_due_date) {
                return false;
            }
            if ((now.isSame(card.resolved_due_date.date) ||
                 now.isBefore(card.resolved_due_date.date)) &&
                (tomorrow.isSame(card.resolved_due_date.date) ||
                 tomorrow.isAfter(card.resolved_due_date.date))) {
                return true;
            } else {
                return false;
            }
        };

        $scope.isLate = function(card) {
            if (card.item_type === 'task') {
                if (card.task.status === 'merged') {
                    return false;
                }
            }
            var now = moment();
            if (!card.resolved_due_date) {
                return false;
            }
            if (now.isAfter(card.resolved_due_date.date)) {
                return true;
            } else {
                return false;
            }
        };

        $scope.showCardDetail = function(card, lane) {
            var modalInstance = $modal.open({
                templateUrl: 'app/boards/template/card_details.html',
                backdrop: 'static',
                controller: 'CardDetailController',
                resolve: {
                    card: function() {
                        return card;
                    },
                    board: function() {
                        return $scope.board;
                    },
                    worklist: function() {
                        return lane.worklist;
                    },
                    permissions: function() {
                        return $scope.permissions;
                    }
                }
            });

            modalInstance.result.finally(function() {
                loadBoard();
            });
        };

        /**
         * Config for the lanes sortable.
         */
        $scope.lanesSortable = {
            orderChanged: updateBoardLanes,
            dragMove: BoardHelper.maybeScrollContainer('kanban-board'),
            accept: function (sourceHandle, dest) {
                return sourceHandle.itemScope.sortableScope.$id === dest.$id;
            }
        };

        /**
         * Config for the cards sortable.
         */
        $scope.cardsSortable = {
            orderChanged: BoardHelper.moveCard,
            itemMoved: BoardHelper.moveCard,
            dragMove: BoardHelper.maybeScrollContainer('kanban-board'),
            accept: function (sourceHandle, dest) {
                var srcParent = sourceHandle.itemScope.sortableScope.$parent;
                var dstParentSortable = dest.$parent.sortableScope;
                if (!$scope.permissions.editBoard) {
                    return true;
                }
                if (!srcParent.sortableScope) {
                    return false;
                }
                return srcParent.sortableScope.$id === dstParentSortable.$id;
            }
        };

        /**
         * Add an event listener to prevent default dragging behaviour from
         * interfering with dragging items around.
         */
        $document[0].ondragstart = function (event) {
            event.preventDefault();
        };

        // Load the board and permissions on page load.
        loadBoard();
        $scope.showEditForm = false;
        $scope.showAddOwner = false;
        $scope.isUpdating = false;
    });

/*
 * Copyright (c) 2016 Codethink Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A controller that manages a view of all worklists and boards.
 */
angular.module('sb.board').controller('BoardsListController',
    function ($scope) {
        'use strict';

        $scope.boardResourceTypes = ['Board'];
        $scope.worklistResourceTypes = ['Worklist'];

    });

/*
 * Copyright (c) 2016 Codethink Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Controller for the card detail modal.
 */
angular.module('sb.board').controller('CardDetailController',
    function ($scope, card, board, worklist, permissions, Story, Task,
              DueDate, Worklist, $document, $timeout, $modalInstance,
              $modal) {
        'use strict';

        /**
         * Story/Task title/description/notes
         */
        $scope.modifications = {
            title: '',
            description: '',
            notes: ''
        };
        $scope.toggleEditTitle = function() {
            if (!(permissions.moveCards || permissions.editBoard)) {
                return false;
            }
            if (!$scope.editingTitle) {
                if (card.item_type === 'story') {
                    $scope.modifications.title = card.story.title;
                } else if (card.item_type === 'task') {
                    $scope.modifications.title = card.task.title;
                }
            }
            $scope.editingTitle = !$scope.editingTitle;
        };

        $scope.editTitle = function() {
            var params = {};
            if (card.item_type === 'story') {
                params = {
                    id: card.story.id,
                    title: $scope.modifications.title
                };
                Story.update(params, function(updated) {
                    $scope.toggleEditTitle();
                    card.story.title = updated.title;
                });
            } else if (card.item_type === 'task') {
                params = {
                    id: card.task.id,
                    title: $scope.modifications.title
                };
                Task.update(params, function(updated) {
                    $scope.toggleEditTitle();
                    card.task.title = updated.title;
                });
            }
        };

        /**
         * Story description
         */
        $scope.toggleEditDescription = function() {
            if (!(permissions.moveCards || permissions.editBoard)) {
                return false;
            }
            if (!$scope.editingTitle) {
                $scope.modifications.description = $scope.story.description;
            }
            $scope.editingDescription = !$scope.editingDescription;
        };


        $scope.editStoryDescription = function() {
            var params = {
                id: $scope.story.id,
                description: $scope.modifications.description
            };
            Story.update(params, function(updated) {
                $scope.toggleEditDescription();
                $scope.story.description = updated.description;
            });
        };


        /**
        * Task Notes
        */
        $scope.toggleEditNotes = function() {
            if (!(permissions.moveCards || permissions.editBoard)) {
                return false;
            }
            if (!$scope.editingTitle) {
                $scope.modifications.notes = $scope.card.task.link;
            }
            $scope.editingNotes = !$scope.editingNotes;
        };

        $scope.editTaskNotes = function() {
            var params = {
                id: card.task.id,
                link: $scope.modifications.notes
            };
            Task.update(params, function(updated) {
                $scope.toggleEditNotes();
                $scope.card.task.link = updated.link;
            });
        };

         /**
         * Due dates
         */
        $scope.noDate = {
            id: -1,
            date: null
        };

        $scope.getRelevantDueDates = function() {
            $scope.relevantDates = [];
            angular.forEach(board.due_dates, function(date) {
                if (date.assignable) {
                    $scope.relevantDates.push(date);
                }
            });

        };

        $scope.toggleEditDueDate = function() {
            if (permissions.moveCards || permissions.editBoard) {
                $scope.editingDueDate = !$scope.editingDueDate;
            }
        };

        $scope.toggleDueDateDropdown = function() {
            var dropdown = $document[0].getElementById('due-dates-dropdown');
            var button = dropdown.getElementsByTagName('button')[0];
            $timeout(function() {
                button.click();
            }, 0);
        };

        function cardHasDate(date) {
            for (var i = 0; i < card[card.item_type].due_dates.length; i++) {
                if (card[card.item_type].due_dates[i] === date.id) {
                    return true;
                }
            }
            return false;
        }

        function assignDueDate(date) {
            if (card.item_type === 'task') {
                date.tasks.push(card.task);
            } else if (card.item_type === 'story') {
                date.stories.push(card.story);
            }
            var params = {
                id: date.id,
                tasks: date.tasks,
                stories: date.stories
            };
            DueDate.update(params).$promise.then(function(updated) {
                if (card.item_type === 'task') {
                    card.task.due_dates.push(updated.id);
                    $scope.getRelevantDueDates(card.task.due_dates);
                } else if (card.item_type === 'story') {
                    card.story.due_dates.push(updated.id);
                    $scope.getRelevantDueDates(card.story.due_dates);
                }
                $scope.assigningDueDate = false;
            });
        }

        $scope.setDisplayDate = function(date) {
            if (!cardHasDate(date) && date.id !== -1) {
                assignDueDate(date);
            }
            card.resolved_due_date = date;
            var params = {
                id: card.list_id,
                item_id: card.id,
                list_position: card.list_position,
                display_due_date: date.id
            };
            Worklist.ItemsController.update(params, function() {
                $scope.editingDueDate = false;
            });
        };

        $scope.validDueDate = function(dueDate) {
            return dueDate && !(dueDate === $scope.noDate);
        };

        /**
         * Task assignee
         */
        $scope.toggleAssigneeTypeahead = function() {
            var typeahead = $document[0].getElementById('assignee');
            var assignLink = typeahead.getElementsByTagName('a')[0];
            $timeout(function() {
                assignLink.click();
            }, 0);
        };

        $scope.toggleEditAssignee = function() {
            $scope.editingAssignee = !$scope.editingAssignee;
        };

        $scope.updateTask = function(task) {
            var params = {
                id: task.id,
                assignee_id: task.assignee_id
            };
            Task.update(params, function() {
                $scope.editingAssignee = false;
            });
        };

        /**
         * Other
         */
        $scope.deleteCard = function() {
            Worklist.ItemsController.delete({
                id: $scope.card.list_id,
                item_id: $scope.card.id
            }, function() {
                $modalInstance.close('deleted');
            });
        };

        $scope.close = function() {
            $modalInstance.close('closed');
        };

        $scope.newDueDate = function() {
            var modalInstance = $modal.open({
                templateUrl: 'app/due_dates/template/new.html',
                backdrop: 'static',
                controller: 'DueDateNewController',
                resolve: {
                    board: function() {
                        return $scope.board;
                    }
                }
            });

            modalInstance.result.then(function(dueDate) {
                if (dueDate.hasOwnProperty('date')) {
                    board.due_dates.push(dueDate);
                }

                $scope.getRelevantDueDates();
                $scope.setDisplayDate(dueDate);
            });
        };

        if (card.item_type === 'task') {
            $scope.story = Story.get({id: card.task.story_id});
        } else if (card.item_type === 'story') {
            $scope.story = card.story;
        }
        $scope.getRelevantDueDates();

        $scope.card = card;
        $scope.board = board;
        $scope.permissions = permissions;
        $scope.worklist = worklist;
        $scope.showDescription = true;
        $scope.showTaskNotes = true;
        $scope.assigningDueDate = false;
        $scope.editingDueDate = false;
        $scope.editingDescription = false;
        $scope.editingNotes = false;
        $scope.editingAssignee = false;
    }
);

/*
 * Copyright (c) 2015 Codethink Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A service to help with use of a kanban board.
 */
angular.module('sb.board').factory('BoardHelper',
    function($document, $window, $timeout, Worklist) {
        'use strict';

        /**
         * Return a function which scrolls the element with id="elementID"
         * horizontally when an as-sortable-item is being dragged to the left
         * or right.
         */
        function scrollFunction(elementID) {
            return function(itemPosition, containment, eventObj) {
                if (eventObj) {
                    var container = document.getElementById(elementID);
                    var offsetX = ($window.pageXOffset ||
                                   $document[0].documentElement.scrollLeft);
                    var targetX = eventObj.pageX - offsetX;

                    var leftBound = container.clientLeft + container.offsetLeft;
                    var parent = container.offsetParent;
                    while (parent) {
                        leftBound += parent.offsetLeft;
                        parent = parent.offsetParent;
                    }

                    var rightBound = leftBound + container.clientWidth;

                    if (targetX < leftBound) {
                        container.scrollLeft -= 10;
                    } else if (targetX > rightBound) {
                        container.scrollLeft += 10;
                    }
                }
            };
        }

        /**
         * ng-sortable callback for orderChanged and itemMoved events.
         *
         * This is called when a card has changed position or moved between
         * two different lanes. It updates the WorklistItem which represents
         * the card.
         */
        function moveCard(result) {
            var list = result.dest.sortableScope.$parent.lane.worklist;
            var position = result.dest.index;
            var item = list.items[position];

            item.list_position = position;
            Worklist.ItemsController.update({
                id: list.id,
                item_id: item.id,
                list_position: item.list_position,
                list_id: list.id
            });
        }

        /**
         * Function to record scrollbar positions for the lanes of a board.
         *
         * This is used to track where a user has scrolled each lane to before
         * refreshing the board UI. It returns a mapping of lane IDs to scroll
         * offsets which can later be used to re-scroll to the correct point.
         */
        function recordLaneScrollbars(board) {
            var scrollbars = {};
            angular.forEach(board.lanes, function(lane) {
                var elem = $document[0].getElementById('lane-' + lane.id);
                if (!!elem) {
                    scrollbars[lane.id] = elem.scrollTop;
                }
            });
            return scrollbars;
        }

        /**
         * Function to scroll lanes to a previously recorded position.
         *
         * Takes a board and a mapping of lane IDs to scroll offsets as
         * produced by `recordLaneScrollbars` and scrolls the corresponding
         * lane containers by the given offsets.
         */
        function scrollLanes(board, scrollbars) {
            angular.forEach(board.lanes, function(lane) {
                $timeout(function() {
                    var elem = $document[0].getElementById('lane-' + lane.id);
                    if (!!elem) {
                        elem.scrollTop = scrollbars[lane.id];
                    }
                });
            });
        }

        return {
            maybeScrollContainer: scrollFunction,
            moveCard: moveCard,
            recordLaneScrollbars: recordLaneScrollbars,
            scrollLanes: scrollLanes
        };
    }
);

/*
 * Copyright (c) 2015 Codethink Limited.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */


angular.module('sb.board').factory('NewBoardService',
    function ($modal, $log, Session, SessionModalService) {
        'use strict';

        return {
            showNewBoardModal: function (userId) {
                if (!Session.isLoggedIn()) {
                    return SessionModalService.showLoginRequiredModal();
                } else {
                    var modalInstance = $modal.open(
                        {
                            size: 'lg',
                            templateUrl: 'app/boards/template/new.html',
                            backdrop: 'static',
                            controller: 'AddBoardController',
                            resolve: {
                                params: function () {
                                    return {
                                        userId: userId || null
                                    };
                                }
                            }
                        }
                    );

                    // Return the modal's promise.
                    return modalInstance.result;
                }
            }
        };
    }
);

/*
 * Copyright (c) 2015 Codethink Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A controller that manages the dashboard for worklists and boards.
 */
angular.module('sb.dashboard').controller('BoardsWorklistsController',
    function ($scope, currentUser, Worklist, Board, SubscriptionList) {
        'use strict';

        var params = {user_id: currentUser.id};
        $scope.loadingBoards = true;
        $scope.loadingWorklists = true;

        // Load the boards belonging to the logged in user.
        Board.browse(params).$promise.then(function(boards) {
            $scope.loadingBoards = false;
            $scope.boards = boards;
        });

        // Load the worklists belonging to the logged in user.
        Worklist.browse(params).$promise.then(function(worklists) {
            $scope.loadingWorklists = false;
            $scope.worklists = worklists;
        });

        $scope.worklistSubscriptions = SubscriptionList.subsList(
            'worklist', currentUser);
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A controller that manages our logged-in dashboard
 */
angular.module('sb.dashboard').controller('DashboardController',
    function ($q, $scope, currentUser, Story, SubscriptionList,
            SubscriptionEvent, Task) {
        'use strict';

        // Load the list of current assigned tasks.

        $scope.filterTasks = Task.browse({
            assignee_id: currentUser.id
        }, function(tasks) {
            var todo = [];
            var progress = [];
            var review = [];
            var invalid = [];

            angular.forEach(tasks, function(task) {
                task.type = 'Task';
                if (task.status === 'review') {
                    review.push(task);
                }
                else if (task.status === 'todo') {
                    todo.push(task);
                }
                else if (task.status === 'inprogress') {
                    progress.push(task);
                }
                else {
                    invalid.push(task);
                }
            });

            $scope.reviewTasks = review;
            $scope.todoTasks = todo;
            $scope.progressTasks = progress;
            $scope.invalidTasks = invalid;
        });

        /**
         * Updates the task list.
         */
        $scope.updateTask = function (task, fieldName, value) {

            if(!!fieldName) {
                task[fieldName] = value;
            }

            task.$update(function () {
                $scope.showTaskEditForm = false;
            });
        };

        $scope.createdStories = Story.browse({
            creator_id: currentUser.id
        });


        /**
         * Filter the stories.
         */
        $scope.showActive = true;
        $scope.showMerged = true;
        $scope.showInvalid = true;

        /**
         * Reload the stories in this view based on user selected filters.
         */
        $scope.filterStories = function () {
            var status = [];
            if ($scope.showActive) {
                status.push('active');
            }
            if ($scope.showMerged) {
                status.push('merged');
            }
            if ($scope.showInvalid) {
                status.push('invalid');
            }

            if (status.length === 0) {
                $scope.createdStories = [];
                return;
            }

            Story.browse({
                    sort_field: 'id',
                    sort_dir: 'desc',
                    status: status,
                    creator_id: currentUser.id
                },
                function (result) {
                    $scope.createdStories = result;
                }
            );
        };


        function loadEvents() {
            // Load the user's subscription events.
            $scope.subscriptionEvents = null;
            SubscriptionEvent.browse({
                subscriber_id: currentUser.id,
                offset: 0,
                limit: 50,
                sort_dir: 'desc',
                sort_field: 'created_at'
            }, function (results) {

                // First go through the results and decode the event info.
                results.forEach(function (row) {
                    if (row.hasOwnProperty('event_info')) {
                        row.event_info = JSON.parse(row.event_info);
                    } else {
                        row.event_info = {};
                    }
                });

                $scope.subscriptionEvents = results;
                $scope.collapsedEvents = results.length > 1;
            });
        }

        loadEvents();

        $scope.removeEvent = function (event) {
            var deferred = $q.defer();
            deferred.resolve([
                event.$delete(function () {
                    var idx = $scope.subscriptionEvents.indexOf(event);
                    $scope.subscriptionEvents.splice(idx, 1);
                })]);
            return deferred.promise;
        };

        $scope.removeAllEvents = function () {
            // delete all events
            var promises = [];
            for (var i = 0; i < $scope.subscriptionEvents.length; i++) {
                var event = $scope.subscriptionEvents[i];
                var promise = $scope.removeEvent(event);
                promises.push(promise);
            }

            // reload new events
            $q.all(promises).then(loadEvents);
        };

        $scope.storySubscriptions = SubscriptionList.subsList(
                'story', currentUser);

    });

/*
 * Copyright (c) 2015 Codethink Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Subscriptions controller. Provides a view of all subscribed objects.
 */
angular.module('sb.dashboard').controller('DashboardSubscriptionsController',
    function ($scope, Story, Project, ProjectGroup, SubscriptionList,
              Worklist, currentUser) {
        'use strict';

        $scope.storySubscriptions = SubscriptionList.subsList(
            'story', currentUser);
        $scope.stories = Story.browse({subscriber_id: currentUser.id});

        $scope.projectSubscriptions = SubscriptionList.subsList(
            'project', currentUser);
        $scope.projects = Project.browse({subscriber_id: currentUser.id});

        $scope.projectGroupSubscriptions = SubscriptionList.subsList(
            'project_group', currentUser);
        $scope.projectGroups = ProjectGroup.browse(
            {subscriber_id: currentUser.id});

        $scope.worklistSubscriptions = SubscriptionList.subsList(
            'worklist', currentUser);
        $scope.worklists = Worklist.browse({subscriber_id: currentUser.id});
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Controller for our home(index) page.
 */
angular.module('sb.dashboard').controller('HomeController',
    function ($state, sessionState, SessionState) {
        'use strict';

        // If we're logged in, go to the dashboard instead.
        if (sessionState === SessionState.LOGGED_IN) {
            $state.go('sb.dashboard.stories');
        } else {
            $state.go('sb.page.about');
        }
    });

/*
 * Copyright (c) 2016 Codethink Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A controller for Worklist event notifications.
 */
angular.module('sb.dashboard').controller('WorklistEventController',
    function ($scope, Worklist) {
        'use strict';

        $scope.worklist = Worklist.get({
            id: $scope.evt.event_info.worklist_id
        });
    }
);

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A directive that renders subscription event messages.
 *
 * @see storyboardApiSignature
 */
angular.module('sb.dashboard').directive('subscriptionEvent',
    function ($log, User) {
        'use strict';

        return {
            restrict: 'A',
            scope: {
                subscriptionEvent: '@'
            },
            link: function (scope) {
                try {
                    var evt = JSON.parse(scope.subscriptionEvent);
                    scope.evt = evt;
                    scope.author = User.get({id: evt.author_id});
                    scope.event_type = evt.event_type;
                    scope.created_at = evt.created_at;
                } catch (e) {
                    $log.error(e);
                }
            },
            templateUrl: 'app/dashboard/template/subscription_event.html'
        };
    });

/*
 * Copyright (c) 2016 Codethink Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Controller for the "new due date" modal popup.
 */
angular.module('sb.due_date').controller('DueDateEditController',
    function ($scope, $modalInstance, $state, $q, DueDate, User,
              board, dueDate) {
        'use strict';
        $scope.owners = [];
        $scope.users = [];
        angular.forEach(dueDate.owners, function(id) {
            $scope.owners.push(User.get({id: id}));
        });
        angular.forEach(dueDate.users, function(id) {
            $scope.users.push(User.get({id: id}));
        });

        /**
         * User typeahead search method.
         */
        $scope.searchUsers = function (value, array) {
            var deferred = $q.defer();

            User.browse({full_name: value, limit: 10},
                function(searchResults) {
                    var results = [];
                    angular.forEach(searchResults, function(result) {
                        if (array.indexOf(result.id) === -1) {
                            results.push(result);
                        }
                    });
                    deferred.resolve(results);
                }
            );
            return deferred.promise;
        };

        /**
         * Formats the user name.
         */
        $scope.formatUserName = function (model) {
            if (!!model) {
                return model.name;
            }
            return '';
        };

        /**
         * Add a new user to one of the permission levels.
         */
        $scope.addUser = function (model, modelArray, idArray) {
            idArray.push(model.id);
            modelArray.push(model);
        };

        /**
         * Remove a user from one of the permission levels.
         */
        $scope.removeUser = function (model, modelArray, idArray) {
            var idIdx = idArray.indexOf(model.id);
            idArray.splice(idIdx, 1);

            var modelIdx = modelArray.indexOf(model);
            modelArray.splice(modelIdx, 1);
        };

        /**
         * Save changes to the due date.
         */
        function saveDueDate() {
            DueDate.update($scope.dueDate, function (result) {
                    var params = {id: $scope.dueDate.id};
                    var owners = {
                        codename: 'edit_date',
                        users: $scope.dueDate.owners
                    };
                    var users = {
                        codename: 'assign_date',
                        users: $scope.dueDate.users
                    };
                    DueDate.Permissions.update(params, users)
                        .$promise.then(function() {
                            DueDate.Permissions.update(params, owners)
                                .$promise.then(function() {
                                    $scope.isSaving = false;
                                    $modalInstance.close(result);
                                });
                        });
                }
            );
        }

        /**
         * Saves the due date.
         */
        $scope.save = function() {
            $scope.isSaving = true;
            $scope.dueDate.date.second(0);
            saveDueDate();
        };

        /**
         * Close this modal without saving.
         */
        $scope.close = function() {
            $modalInstance.dismiss('cancel');
        };

        // Create a blank DueDate to begin with.
        $scope.isSaving = false;
        $scope.dueDate = dueDate;
        $scope.modalTitle = 'Edit Due Date';
    });

/*
 * Copyright (c) 2016 Codethink Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Controller for the "new due date" modal popup.
 */
angular.module('sb.due_date').controller('DueDateNewController',
    function ($scope, $modalInstance, $state, $q, DueDate, board, CurrentUser,
              User) {
        'use strict';

        var currentUser = CurrentUser.resolve();
        $scope.owners = [];
        $scope.users = [];
        angular.forEach(board.owners, function(id) {
            $scope.owners.push(User.get({id: id}));
        });
        angular.forEach(board.users, function(id) {
            $scope.users.push(User.get({id: id}));
        });

        /**
         * Create the new due date.
         */
        function saveDueDate() {
            if ($scope.mode === 'edit') {
                $scope.dueDate.$create(
                    function (result) {
                        $scope.isSaving = false;
                        $modalInstance.close(result);
                    }
                );
            } else if ($scope.mode === 'find') {
                DueDate.update({
                    id: $scope.selected.id,
                    board_id: board.id
                }, function (result) {
                    $scope.isSaving = false;
                    $modalInstance.close(result);
                });
            }
        }

        /**
         * User typeahead search method.
         */
        $scope.searchUsers = function (value, array) {
            var deferred = $q.defer();

            User.browse({full_name: value, limit: 10},
                function(searchResults) {
                    var results = [];
                    angular.forEach(searchResults, function(result) {
                        if (array.indexOf(result.id) === -1) {
                            results.push(result);
                        }
                    });
                    deferred.resolve(results);
                }
            );
            return deferred.promise;
        };

        /**
         * Formats the user name.
         */
        $scope.formatUserName = function (model) {
            if (!!model) {
                return model.name;
            }
            return '';
        };

        /**
         * Add a new user to one of the permission levels.
         */
        $scope.addUser = function (model, modelArray, idArray) {
            idArray.push(model.id);
            modelArray.push(model);
        };

        /**
         * Remove a user from one of the permission levels.
         */
        $scope.removeUser = function (model, modelArray, idArray) {
            var idIdx = idArray.indexOf(model.id);
            idArray.splice(idIdx, 1);

            var modelIdx = modelArray.indexOf(model);
            modelArray.splice(modelIdx, 1);
        };

        /**
         * Saves the due date.
         */
        $scope.save = function() {
            $scope.isSaving = true;
            $scope.dueDate.date.second(0);
            saveDueDate();
        };

        /**
         * Close this modal without saving.
         */
        $scope.close = function() {
            $modalInstance.dismiss('cancel');
        };

        $scope.setMode = function(mode) {
            $scope.mode = mode;
        };

        $scope.searchDueDates = function() {
            DueDate.browse($scope.criteria, function(results) {
                var existing = [];
                angular.forEach(board.due_dates, function(date) {
                    existing.push(date.id);
                });
                $scope.results = results;
                angular.forEach(results, function(result) {
                    if (existing.indexOf(result.id) !== -1) {
                        var idx = $scope.results.indexOf(result);
                        $scope.results.splice(idx, 1);
                    }
                });
            });
        };

        $scope.select = function(dueDate) {
            $scope.selected = dueDate;
        };

        $scope.selected = {};
        $scope.criteria = {};
        currentUser.then(function(result) {
            currentUser = result;
            $scope.criteria = {
                user: currentUser.id
            };
        });

        // Create a blank DueDate to begin with.
        $scope.isSaving = false;
        $scope.newDueDate = true;
        $scope.modalTitle = 'New Due Date';
        $scope.mode = 'edit';
        $scope.dueDate = new DueDate({
            name: ''
        });
        if (!!board) {
            $scope.dueDate.board_id = board.id;
            $scope.dueDate.owners = board.owners;
            $scope.dueDate.users = board.users;
        }
    });

/*
 * Copyright (c) 2015 Codethink Limited
 *
 * Licensed under the Apache License, Version 2.0 (the 'License'); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Controller for "archive board" modal
 */
angular.module('sb.board').controller('DueDateRemoveController',
    function ($log, $scope, $state, board, dueDate, $modalInstance, DueDate) {
        'use strict';

        $scope.board = board;
        $scope.dueDate = dueDate;

        // Set our progress flags and clear previous error conditions.
        $scope.isUpdating = true;
        $scope.error = {};

        $scope.remove = function () {
            DueDate.delete({
                    id: dueDate.id,
                    board_id: board.id
                },
                function () {
                    $modalInstance.close('success');
                }
            );
        };

        $scope.close = function () {
            $modalInstance.dismiss('cancel');
        };
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * This module acts as the central routing point for all errors that occur
 * within storyboard.
 */
angular.module('sb.notification').controller('NotificationsController',
    function ($scope, Notification) {
        'use strict';

        var defaultDisplayCount = 5;

        $scope.displayCount = defaultDisplayCount;

        $scope.notifications = [];

        /**
         * Remove a notification from the display list.
         *
         * @param notification
         */
        $scope.remove = function (notification) {
            var idx = $scope.notifications.indexOf(notification);
            if (idx > -1) {
                $scope.notifications.splice(idx, 1);
            }

            // If the notification list length drops below default, make
            // sure we reset the limit.
            if ($scope.notifications.length <= defaultDisplayCount) {
                $scope.displayCount = defaultDisplayCount;
            }
        };

        /**
         * Reveal more notifications, either current count + 5 or the total
         * number of messages, whichever is smaller.
         */
        $scope.showMore = function () {
            // Set this to something big.
            $scope.displayCount = Math.min($scope.notifications.length,
                    $scope.displayCount + 5);
        };

        /**
         * Set up a notification subscriber, and make sure it's removed when
         * the scope is destroyed.
         */
        $scope.$on('$destroy', Notification.subscribe(
                function (notification) {
                    $scope.notifications.push(notification);
                }
            )
        );
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * This directive is a notification list renderer with all the trimmings.
 * Errors broadcast throughout the system will be collected and displayed here.
 */
angular.module('sb.notification').directive('notifications',
    function () {
        'use strict';

        return {
            restrict: 'E',
            templateUrl: 'app/notification/template/notifications.html',
            controller: 'NotificationsController'
        };
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */


/**
 * Useful priority constants.
 */

angular.module('sb.notification').constant('Priority', {
    BEFORE: -1,
    FIRST: 0,
    LAST: 999,
    AFTER: 1000
});

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */


/**
 * A list of severity levels used within this module.
 */

angular.module('sb.notification').constant('Severity', {
    ERROR: 'error',
    WARNING: 'warning',
    INFO: 'info',
    SUCCESS: 'success'
});

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The centralized notification service, aka the central routing point for all
 * broadcast notifications. You use it by registering interceptors and
 * subscribers, and handling any messages that are sent().
 *
 * Interceptors are intended to be both filters and decorators, where
 * individual components can handle messages before either terminating
 * the dispatch chain, or passing them on to the next interceptor. In this
 * fashion it is easy to handle specific messages in one context while
 * other messages in another.
 *
 * Subscribers are processors that handle all messages vetted by our
 * interceptors.
 */
angular.module('sb.notification').factory('Notification',
    function ($log, Severity, Priority) {
        'use strict';

        var subscribers = [];
        var interceptors = [];

        return {
            /**
             * Send a notification.
             *
             * @param type A type identifier, such as a string. Use this for
             * your subscribers to determine what kind of a message you're
             * working with.
             * @param message A human readable message for this notification.
             * @param severity The severity of this message, any of the
             * constants provided in Severity.
             * @param cause The cause of this message, perhaps a large amount
             * of debugging information.
             * @param callback If this message prompts the user to do
             * something, then pass a function here and it'll be rendered
             * in the message.
             * @param callbackLabel A custom label for the callback.
             */
            send: function (type, message, severity, cause, callback,
                            callbackLabel) {
                // Return the type.
                if (!type || !message) {
                    $log.warn('Invoked Notification.send() without a type' +
                        ' or message.');
                    return;
                }

                // sanitize our data.
                var n = {
                    'type': type,
                    'message': message,
                    'severity': severity || Severity.INFO,
                    'cause': cause || null,
                    'callback': callback || null,
                    'callbackLabel': callbackLabel || null,
                    'date': new Date()
                };

                // Iterate through the interceptors.
                for (var i = 0; i < interceptors.length; i++) {
                    if (!!interceptors[i].method(n)) {
                        return;
                    }
                }

                // Iterate through the subscribers.
                for (var j = 0; j < subscribers.length; j++) {
                    subscribers[j](n);
                }
            },

            /**
             * Add a message interceptor to the notification system, in order
             * to determine which messages you'd like to handle.
             *
             * @param interceptor A method that accepts a notification. You can
             * return true from the interceptor method to indicate that this
             * message has been handled and should not be processed further.
             * @param priority An optional priority (default 999).
             * Interceptors with a lower priority will go first.
             * @returns {Function} A method that may be called to remove the
             * interceptor at a later time.
             */
            intercept: function (interceptor, priority) {

                var i = {
                    'priority': priority || Priority.LAST,
                    'method': interceptor
                };

                // Add and sort the interceptors. We're using unshift here so
                // that the sorting algorithm ends up being a single-pass
                // bubble sort.
                interceptors.unshift(i);
                interceptors.sort(function (a, b) {
                    return a.priority - b.priority;
                });

                return function () {
                    var idx = interceptors.indexOf(i);
                    interceptors.splice(idx, 1);
                };
            },

            /**
             * Subscribe to all messages that make it through our interceptors.
             *
             * @param subscriber A subscriber method that receives a
             * notification.
             * @returns {Function}  A method that may be called to remove the
             * subscriber at a later time.
             */
            subscribe: function (subscriber) {
                subscribers.push(subscriber);

                return function () {
                    subscribers.remove(subscriber);
                };
            }
        };
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Preferences controller for our user profile. Allows explicit editing of
 * individual preferences.
 */
angular.module('sb.profile').controller('ProfilePreferencesController',
    function ($scope, Preference, Notification, Severity) {
        'use strict';

        $scope.preferences = Preference.getAll();

        /**
         * Save all the preferences.
         */
        $scope.save = function () {
            $scope.saving = true;

            Preference.saveAll($scope.preferences).then(
                function () {
                    Notification.send(
                        'preferences',
                        'Preferences Saved!',
                        Severity.SUCCESS
                    );
                    $scope.saving = false;
                },
                function () {
                    $scope.saving = false;
                }
            );
        };
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Issue token controller.
 */
angular.module('sb.profile').controller('ProfileTokenNewController',
    function ($q, $log, $scope, $modalInstance, UserToken, user) {
        'use strict';

        /**
         * Flag for the UI to indicate that we're saving.
         *
         * @type {boolean}
         */
        $scope.isSaving = false;

        /**
         * The new token.
         *
         * @type {UserToken}
         */
        $scope.token = new UserToken({
            user_id: user.id,
            expires_in: 3600
        });

        /**
         * Saves the project group
         */
        $scope.save = function () {
            $scope.isSaving = true;
            $scope.token.$create(
                function (token) {
                    $modalInstance.close(token);
                    $scope.isSaving = false;
                },
                function () {
                    $scope.isSaving = false;
                }
            );
        };

        /**
         * Close this modal without saving.
         */
        $scope.close = function () {
            $modalInstance.dismiss('cancel');
        };
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * User profile controller for all of a user's auth tokens.
 */
angular.module('sb.profile').controller('ProfileTokensController',
    function ($scope, UserToken, tokens, $modal) {
        'use strict';

        $scope.tokens = tokens;

        $scope.deleteToken = function (token) {
            token.$delete(function () {
                var idx = $scope.tokens.indexOf(token);
                if (idx > -1) {
                    $scope.tokens.splice(idx, 1);
                }
            });
        };

        $scope.issueToken = function () {
            $modal.open({
                templateUrl: 'app/profile/template/token_new.html',
                backdrop: 'static',
                controller: 'ProfileTokenNewController',
                resolve: {
                    user: function (CurrentUser) {
                        return CurrentUser.resolve();
                    }
                }
            }).result.then(function (token) {
                    // On success, append the token.
                    $scope.tokens.push(token);
                });
        };
    });


/**
 * Controller for a single token row within the profile token view.
 */
angular.module('sb.profile').controller('ProfileTokenItemController',
    function ($scope, AccessToken) {
        'use strict';

        var now = new Date();

        // Render the expiration date.
        $scope.created = new Date($scope.token.created_at);
        $scope.expires = new Date($scope.token.created_at);
        $scope.expires.setSeconds($scope.expires.getSeconds() +
            $scope.token.expires_in);

        $scope.expired = $scope.expires.getTime() < now.getTime();
        $scope.current =
            $scope.token.access_token === AccessToken.getAccessToken();
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Controller for the project group delete modal popup.
 */
angular.module('sb.project_group').controller('ProjectGroupDeleteController',
    function ($scope, projectGroup, $modalInstance) {
        'use strict';

        $scope.projectGroup = projectGroup;

        // Set our progress flags and clear previous error conditions.
        $scope.isUpdating = true;
        $scope.error = {};

        /**
         *
         */
        $scope.remove = function () {
            $scope.projectGroup.$delete(
                function () {
                    $modalInstance.close('success');
                }
            );
        };

        $scope.close = function () {
            $modalInstance.dismiss('cancel');
        };
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
/**
 * Project group detail controller, for general user use of project groups.
 * From a feature standpoint this really just means viewing the group, member
 * projects, and any stories that belong under this project group.
 */
angular.module('sb.project_group').controller('ProjectGroupDetailController',
    function ($scope, $stateParams, projectGroup, Story, Project,
              Preference, SubscriptionList, CurrentUser, Subscription,
              $q, ProjectGroupItem, ArrayUtil, $log) {
        'use strict';

        var projectPageSize = Preference.get(
            'project_group_detail_projects_page_size') || 0;
        var storyPageSize = Preference.get(
            'project_group_detail_stories_page_size') || 0;

        /**
         * The project group we're viewing right now.
         *
         * @type ProjectGroup
         */
        $scope.projectGroup = projectGroup;

        /**
         * The list of projects in this group
         *
         * @type [Project]
         */
        $scope.projects = [];
        $scope.isSearchingProjects = false;

        $scope.editMode = false;

        $scope.toggleEdit = function() {
            $scope.editMode = !$scope.editMode;
        };

        /**
         * List the projects in this Project Group
         */
        $scope.listProjects = function () {
            $scope.isSearchingProjects = true;
            Project.browse({
                    project_group_id: projectGroup.id,
                    offset: $scope.projectSearchOffset,
                    limit: projectPageSize,
                    sort_dir: 'desc'
                },
                function (result, headers) {
                    // Successful search results, apply the results to the
                    // scope and unset our progress flag.
                    $scope.projectCount =
                        parseInt(headers('X-Total')) || result.length;
                    $scope.projectSearchOffset =
                        parseInt(headers('X-Offset')) || 0;
                    $scope.projectSearchLimit =
                        parseInt(headers('X-Limit')) || 0;
                    $scope.projects = result;
                    $scope.isSearchingProjects = false;
                },
                function (error) {
                    // Error search results, show the error in the UI and
                    // unset our progress flag.
                    $scope.error = error;
                    $scope.isSearchingProjects = false;
                }
            );
        };

        /**
         * The list of stories in this project group
         *
         * @type [Story]
         */
        $scope.stories = [];

        /**
         * Filter the stories.
         */
        $scope.selectedStatus = {};
        $scope.selectedStatus.showActive = true;
        $scope.selectedStatus.showMerged = false;
        $scope.selectedStatus.showInvalid = false;

        /**
         * Reload the stories in this view based on user selected filters.
         */
        $scope.filterStories = function () {
            var status = [];
            if ($scope.selectedStatus.showActive) {
                status.push('active');
            }
            if ($scope.selectedStatus.showMerged) {
                status.push('merged');
            }
            if ($scope.selectedStatus.showInvalid) {
                status.push('invalid');
            }

            // If we're asking for nothing, just return a [];
            if (status.length === 0) {
                $scope.stories = [];
                return;
            }

            Story.browse({
                    project_group_id: projectGroup.id,
                    sort_field: 'id',
                    sort_dir: 'desc',
                    status: status,
                    offset: $scope.storySearchOffset,
                    limit: storyPageSize
                },
                function (result, headers) {
                    // Successful search results, apply the results to the
                    // scope and unset our progress flag.
                    $scope.storyCount =
                        parseInt(headers('X-Total')) || result.length;
                    $scope.storySearchOffset =
                        parseInt(headers('X-Offset')) || 0;
                    $scope.storySearchLimit =
                        parseInt(headers('X-Limit')) || 0;
                    $scope.stories = result;
                    $scope.isSearchingStories = false;
                },
                function (error) {
                    // Error search results, show the error in the UI and
                    // unset our progress flag.
                    $scope.error = error;
                    $scope.isSearchingStories = false;
                }
            );
        };

        /**
         * Next page of the results.
         *
         * @param type The name of the result set to be paged,
         * expects 'stories' or 'projects'.
         */
        $scope.nextPage = function (type) {
            if (type === 'stories') {
                $scope.storySearchOffset += storyPageSize;
                $scope.filterStories();
            } else if (type === 'projects') {
                $scope.projectSearchOffset += projectPageSize;
                $scope.listProjects();
            }
        };

        /**
         * Previous page of the results.
         *
         * @param type The name of the result set to be paged,
         * expects 'stories' or 'projects'.
         */
        $scope.previousPage = function (type) {
            if (type === 'stories') {
                $scope.storySearchOffset -= storyPageSize;
                if ($scope.storySearchOffset < 0) {
                    $scope.storySearchOffset = 0;
                }
                $scope.filterStories();
            } else if (type === 'projects') {
                $scope.projectSearchOffset -= projectPageSize;
                if ($scope.projectSearchOffset < 0) {
                    $scope.projectSearchOffset = 0;
                }
                $scope.listProjects();
            }
        };

        /**
         * Update the page size preference and re-search.
         *
         * @param type The name of the result set to change the page
         * size for, expects 'stories' or 'projects'.
         * @param value The value to set the page size preference to.
         */
        $scope.updatePageSize = function (type, value) {
            if (type === 'stories') {
                Preference.set(
                    'project_group_detail_stories_page_size', value).then(
                        function () {
                            storyPageSize = value;
                            $scope.filterStories();
                        }
                    );
            } else if (type === 'projects') {
                Preference.set(
                    'project_group_detail_projects_page_size', value).then(
                        function () {
                            projectPageSize = value;
                            $scope.listProjects();
                        }
                    );
            }
        };


        /**
         * UI flag, are we saving?
         *
         * @type {boolean}
         */
        $scope.isSaving = false;

        /**
         * Project typeahead search method.
         */
        $scope.searchProjects = function (value) {
            var deferred = $q.defer();
            Project.browse({name: value, limit: 10},
                function (results) {
                    // Dedupe the results.
                    var idxList = [];
                    for (var i = 0; i < $scope.projects.length; i++) {
                        var project = $scope.projects[i];
                        if (!!project) {
                            idxList.push(project.id);
                        }
                    }

                    for (var j = results.length - 1; j >= 0; j--) {
                        var resultId = results[j].id;
                        if (idxList.indexOf(resultId) > -1) {
                            results.splice(j, 1);
                        }
                    }

                    deferred.resolve(results);
                },
                function (error) {
                    $log.error(error);
                    deferred.resolve([]);
                });
            return deferred.promise;
        };

        /**
         * Formats the project name.
         */
        $scope.formatProjectName = function (model) {
            if (!!model) {
                return model.name;
            }
            return '';
        };

        /**
         * Remove a project from the list
         */
        $scope.removeProject = function (index) {
            $scope.projects.splice(index, 1);
        };

        /**
         * Save the project and the associated groups
         */
        $scope.save = function () {
            $scope.isSaving = true;

            ProjectGroupItem.browse({projectGroupId: $scope.projectGroup.id},
                function(results) {
                    var loadedIds = [];
                    results.forEach(function (project) {
                        loadedIds.push(project.id);
                    });
                    var promises = [];

                    // Get the desired ID's.
                    var desiredIds = [];
                    $scope.projects.forEach(function (project) {
                        desiredIds.push(project.id);
                    });

                    // Intersect loaded vs. current to get a list of project
                    // reference to delete.
                    var idsToDelete = ArrayUtil.difference(
                        loadedIds, desiredIds);
                    idsToDelete.forEach(function (id) {

                        // Get a deferred promise...
                        var removeProjectDeferred = $q.defer();

                        // Construct the item.
                        var item = new ProjectGroupItem({
                            id: id,
                            projectGroupId: projectGroup.id
                        });

                        // Delete the item.
                        item.$delete(function (result) {
                                removeProjectDeferred.resolve(result);
                            },
                            function (error) {
                                removeProjectDeferred.reject(error);
                            }
                        );

                        promises.push(removeProjectDeferred.promise);
                    });

                    // Intersect current vs. loaded to get a list of project
                    // reference to add.
                    var idsToAdd = ArrayUtil.difference(desiredIds, loadedIds);
                    idsToAdd.forEach(function (id) {

                        // Get a deferred promise...
                        var addProjectDeferred = $q.defer();

                        // Construct the item.
                        var item = new ProjectGroupItem({
                            id: id,
                            projectGroupId: projectGroup.id
                        });

                        // Delete the item.
                        item.$create(function (result) {
                                addProjectDeferred.resolve(result);
                            },
                            function (error) {
                                addProjectDeferred.reject(error);
                            }
                        );

                        promises.push(addProjectDeferred.promise);
                    });


                    // Save the project group itself.
                    var deferred = $q.defer();
                    promises.push(deferred.promise);
                    $scope.projectGroup.$update(function (success) {
                        deferred.resolve(success);
                    }, function (error) {
                        $log.error(error);
                        deferred.reject(error);
                    });

                    // Roll all the promises into one big happy promise.
                    $q.all(promises).then(
                        function () {
                            $scope.editMode = false;
                            $scope.isSaving = false;
                        },
                        function (error) {
                            $log.error(error);
                        }
                    );
                }
            );
        };

        /**
         * Add project.
         */
        $scope.addProject = function () {
            $scope.projects.push({});
        };

        /**
         * Insert item into the project list.
         */
        $scope.selectNewProject = function (index, model) {
            // Put our model into the array
            $scope.projects[index] = model;
        };

        /**
         * Check that we have valid projects on the list
         */
        $scope.checkValidProjects = function () {
            if ($scope.projects.length === 0) {
                return false;
            }

            // check if projects contain a valid project_id
            for (var i = 0; i < $scope.projects.length; i++) {
                var project = $scope.projects[i];
                if (!project.id) {
                    return false;
                }
            }
            return true;
        };


        $scope.listProjects();
        $scope.filterStories();

        //GET subscriptions
        var cuPromise = CurrentUser.resolve();

        $scope.resolvedUser = false;
        cuPromise.then(function(user){
            $scope.projectSubscriptions = SubscriptionList.subsList(
                'project', user);
            $scope.storySubscriptions = SubscriptionList.subsList(
                'story', user);
            $scope.projectGroupSubscription = Subscription.browse({
                target_type: 'project_group',
                target_id: $scope.projectGroup.id,
                user_id: user.id
            });
            $scope.resolvedUser = true;
        });
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Controller for the project group member list.
 */
angular.module('sb.project_group').controller('ProjectGroupItemController',
    function ($scope, $log, ProjectGroupItem) {
        'use strict';

        $scope.projectGroupItems = [];
        $scope.loadingProjectGroupItems = false;

        if (!$scope.projectGroup) {
            return;
        }

        var id = $scope.projectGroup.id;

        $scope.loadingProjectGroupItems = true;
        ProjectGroupItem.browse({
                projectGroupId: id
            },
            function (results) {
                $scope.loadingProjectGroupItems = false;
                $scope.projectGroupItems = results;
                $scope.collapsed = results.length > 1;
            }, function (error) {
                $log.error(error);
                $scope.loadingProjectGroupItems = false;
            });
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A Project
 */
angular.module('sb.project_group').controller('ProjectGroupListController',
    function ($scope, $modal, SubscriptionList, CurrentUser, $state,
              $location, SearchHelper) {
        'use strict';

        // search results must be of type "ProjectGroup"
        $scope.resourceTypes = ['ProjectGroup'];

        var params = $location.search();
        $scope.defaultCriteria = SearchHelper.parseParameters(params);

        /**
         * Create a new project-group.
         */
        $scope.newProjectGroup = function () {
            $scope.modalInstance = $modal.open(
                {
                    templateUrl: 'app/project_group/template/new.html',
                    backdrop: 'static',
                    controller: 'ProjectGroupNewController'
                });

            $scope.modalInstance.result.then(function (projectGroup) {
                    // On success, go to the project group detail.
                    $scope.showMobileNewMenu = false;
                    $state.go(
                        'sb.project_group.detail',
                        {id: projectGroup.id}
                    );
                });
        };

        //GET list of project group subscriptions
        var cuPromise = CurrentUser.resolve();

        cuPromise.then(function(user){
            $scope.projectGroupSubscriptions =
            SubscriptionList.subsList(
                'project_group', user);
        });

    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * New Project Group modal controller.
 */
angular.module('sb.project_group').controller('ProjectGroupNewController',
    function ($q, $log, $scope, $modalInstance, ProjectGroup, ProjectGroupItem,
              Project) {
        'use strict';

        /**
         * Flag for the UI to indicate that we're saving.
         *
         * @type {boolean}
         */
        $scope.isSaving = false;

        /**
         * The list of projects.
         *
         * @type {{}[]}
         */
        $scope.projects = [
            {}
        ];

        /**
         * The new project group.
         *
         * @type {ProjectGroup}
         */
        $scope.projectGroup = new ProjectGroup();

        $scope.error = {};
        /**
         * Project typeahead search method.
         */
        $scope.searchProjects = function (value) {
            var deferred = $q.defer();
            Project.browse({name: value, limit: 10},
                function (results) {
                    // Dedupe the results.
                    var idxList = [];
                    for (var i = 0; i < $scope.projects.length; i++) {
                        var project = $scope.projects[i];
                        if (!!project) {
                            idxList.push(project.id);
                        }
                    }

                    for (var j = results.length - 1; j >= 0; j--) {
                        var resultId = results[j].id;
                        if (idxList.indexOf(resultId) > -1) {
                            results.splice(j, 1);
                        }
                    }

                    deferred.resolve(results);
                },
                function (error) {
                    $log.error(error);
                    // We've encountered an error.
                    $scope.isSaving = false;
                });
            return deferred.promise;
        };

        /**
         * Formats the project name.
         */
        $scope.formatProjectName = function (model) {
            if (!!model) {
                return model.name;
            }
            return '';
        };

        /**
         * Add project.
         */
        $scope.addProject = function () {
            $scope.projects.push({});
        };

        /**
         * Insert item into the project list.
         */
        $scope.selectNewProject = function (index, model) {
            // Put our model into the array
            $scope.projects[index] = model;
        };

        /**
         * Remove a project from the list.
         */
        $scope.removeProject = function (index) {
            $scope.projects.splice(index, 1);
        };

        /**
         * Saves the project group
         */
        $scope.save = function () {
            $scope.isSaving = true;

            // Create a new project group
            $scope.projectGroup.$save(function (projectGroup) {
                var promises = [];
                $scope.projects.forEach(
                    function (project) {
                        // Get a deferred promise...
                        var deferred = $q.defer();

                        // Construct the item.
                        var item = new ProjectGroupItem({
                            id: project.id,
                            projectGroupId: projectGroup.id
                        });

                        // Create the item.
                        item.$create(function (result) {
                                deferred.resolve(result);
                            },
                            function (error) {
                                deferred.reject(error);
                            }
                        );

                        promises.push(deferred.promise);
                    }
                );

                // Wait for all the promises to finish.
                $q.all(promises).then(function () {
                    $modalInstance.close(projectGroup);
                }, function (error) {
                    $log.error(error.data.faultstring);
                    $scope.error = error.data.faultstring;
                    $modalInstance.dismiss('cancel');
                    $scope.isSaving = false;
                });


            }, function (error) {
                $modalInstance.dismiss('cancel');
                $log.error(error.data.faultstring);
                $scope.error = error.data.faultString;
                $scope.isSaving = false;
            });
        };

        /**
         * Close this modal without saving.
         */
        $scope.close = function () {
            $modalInstance.dismiss('cancel');
        };

        /**
         * Check that we have valid projects on the list
         */
        $scope.checkValidProjects = function () {
            if ($scope.projects.length === 0) {
                return false;
            }

            // check if projects contain a valid project_id
            for (var i = 0; i < $scope.projects.length; i++) {
                var project = $scope.projects[i];
                if (!project.id) {
                    return false;
                }
            }
            return true;
        };

    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
/**
 * Project detail &  manipulation controller. Usable for any view that wants to
 * view, edit, or delete a project, though views don't have to use all the
 * functions therein. Includes flags for busy time, error responses and more.
 *
 * This controller assumes that the $stateParams object is both injectable and
 * contains an ":id" property that indicates which project should be loaded. At
 * the moment it will only set a 'isLoading' flag to indicate that data is
 * loading. If loading the data is anticipated to take longer than 3 seconds,
 * this will need to be updated to display a sane progress.
 *
 * Do not allow loading of this (or any) controller to take longer than 10
 * seconds. 3 is preferable.
 */
angular.module('sb.projects').controller('ProjectDetailController',
    function ($scope, $rootScope, $state, $stateParams, Project, Story,
              Session, isSuperuser, CurrentUser, Subscription) {
        'use strict';

        // Parse the ID
        var id = $stateParams.hasOwnProperty('id') ?
            $stateParams.id : null;

        if (!isNaN(id)) {
            id = parseInt(id, 10);
        }

        if (id === null) {
            $state.go('sb.index');
            return;
        }

        /**
         * UI flag for when we're initially loading the view.
         *
         * @type {boolean}
         */
        $scope.isLoading = true;

        /**
         * UI view for when a change is round-tripping to the server.
         *
         * @type {boolean}
         */
        $scope.isUpdating = false;

        /**
         * Any error objects returned from the services.
         *
         * @type {{}}
         */
        $scope.error = {};

        /**
         * Generic service error handler. Assigns errors to the view's scope,
         * and unsets our flags.
         */
        function handleServiceError(error) {
            // We've encountered an error.
            $scope.error = error;
            $scope.isLoading = false;
            $scope.isUpdating = false;
        }

        /**
         * Resets our loading flags.
         */
        $scope.projectSubscription = {};
        $scope.resolvedUser = false;
        function handleServiceSuccess() {
            $scope.isLoading = false;
            $scope.isUpdating = false;
            // Get subscriptions
            var cuPromise = CurrentUser.resolve();

            cuPromise.then(function(user){
                $scope.projectSubscription = Subscription.browse({
                    target_type: 'project',
                    target_id: $scope.project.id,
                    user_id: user.id
                });
                $scope.resolvedUser = true;
            });
        }

        /**
         * Load the project
         */
        function loadProject() {
            return Project.get(
                {'id': id},
                function (result) {
                    handleServiceSuccess();
                    return result;
                },
                handleServiceError
            );
        }

        /**
         * The project we're manipulating right now.
         *
         * @type Project
         */
        $scope.project = loadProject();

        /**
         * Toggles the form back.
         */
        $scope.cancel = function () {
            loadProject();
            $scope.showEditForm = false;
        };

        /**
         * Toggle/display the edit form
         */
        $scope.toggleEditMode = function () {
            if (isSuperuser) {
                $scope.showEditForm = !$scope.showEditForm;

                // Deferred timeout request for a re-rendering of elastic
                // text fields, since the size calculation breaks when
                // visible: false
                setTimeout(function () {
                    $rootScope.$broadcast('elastic:adjust');
                }, 1);
            } else {
                $scope.showEditForm = false;
            }
        };

        /**
         * Scope method, invoke this when you want to update the project.
         */
        $scope.update = function () {
            // Set our progress flags and clear previous error conditions.
            $scope.isUpdating = true;
            $scope.error = {};

            // Invoke the save method and wait for results.
            $scope.project.$update(
                function () {
                    // Unset our loading flag and navigate to the detail view.
                    $scope.isUpdating = false;
                    $scope.showEditForm = false;
                    handleServiceSuccess();
                },
                handleServiceError
            );
        };
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The project list controller handles discovery for all projects, including
 * search. Note that it is assumed that we implemented a search (inclusive),
 * rather than a browse (exclusive) approach.
 */
angular.module('sb.projects').controller('ProjectListController',
    function ($scope, $modal, isSuperuser, SubscriptionList, CurrentUser,
              $location, SearchHelper) {
        'use strict';

        // inject superuser flag to properly adjust UI.
        $scope.is_superuser = isSuperuser;

        // search results must be of type "project"
        $scope.resourceTypes = ['Project'];

        var params = $location.search();
        $scope.defaultCriteria = SearchHelper.parseParameters(params);

        /**
         * Launches the add-project modal.
         */
        $scope.addProject = function () {
            $scope.modalInstance = $modal.open({
                size: 'lg',
                templateUrl: 'app/projects/template/new.html',
                backdrop: 'static',
                controller: 'ProjectNewController'
            });
        };

        var cuPromise = CurrentUser.resolve();

        cuPromise.then(function(user){
            $scope.projectSubscriptions = SubscriptionList.subsList(
                'project', user);
        });
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * View controller for the new project form. Includes an intermediary 'saving'
 * flag as well as room for an error response (though until we get a real API
 * that'll be a bit tricky to test).
 */
angular.module('sb.projects').controller('ProjectNewController',
    function ($scope, $state, $modalInstance, Project) {
        'use strict';

        // View parameters.
        $scope.newProject = new Project();
        $scope.isCreating = false;
        $scope.error = {};

        /**
         * Submits the newly created project. If an error response is received,
         * assigns it to the view and unsets various flags. The template
         * should know how to handle it.
         */
        $scope.createProject = function () {

            // Clear everything and set the progress flag...
            $scope.isCreating = true;
            $scope.error = {};

            $scope.newProject.$create(
                function (project) {
                    $modalInstance.dismiss('success');
                    $state.go('sb.project.detail', {id: project.id});
                },
                function (error) {
                    // Error received. Ho hum.
                    $scope.isCreating = false;
                    $scope.error = error;
                }
            );
        };

        /**
         * Close this modal without saving.
         */
        $scope.close = function () {
            $modalInstance.dismiss('cancel');
        };
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * This controller manages stories within the scope of a particular project.
 */
angular.module('sb.projects').controller('ProjectStoryListController',
    function ($scope, $state, $stateParams, Story, NewStoryService,
              Preference, SubscriptionList, CurrentUser) {
        'use strict';

        var pageSize = Preference.get('project_detail_page_size');

        // Variables and methods available to the template...
        function resetScope() {
            $scope.storyCount = 0;
            $scope.stories = [];
            $scope.error = {};
        }

        $scope.searchQuery = '';
        $scope.isSearching = false;
        $scope.filter = 'active';

        $scope.offsets = {
            'active': 0,
            'merged': 0,
            'invalid': 0
        };
        /**
         * Set the filter and refresh the search.
         */
        $scope.setFilter = function (state) {
            $scope.offsets[$scope.filter] = $scope.searchOffset;
            $scope.filter = state;
            $scope.searchOffset = $scope.offsets[$scope.filter];
            $scope.search();
        };
        /**
         * The search method.
         */
        $scope.search = function () {
            // Clear the scope and set the progress flag.
            resetScope();
            $scope.isSearching = true;

            // Execute the story query.
            Story.browse({
                    project_id: $scope.project.id,
                    status: $scope.filter || null,
                    offset: $scope.searchOffset,
                    limit: pageSize,
                    sort_dir: 'desc'
                },
                function (result, headers) {

                    // Successful search results, apply the results to the
                    // scope and unset our progress flag.
                    $scope.storyCount =
                        parseInt(headers('X-Total')) || result.length;
                    $scope.searchOffset = parseInt(headers('X-Offset')) || 0;
                    $scope.searchLimit = parseInt(headers('X-Limit')) || 0;
                    $scope.stories = result;
                    $scope.isSearching = false;
                },
                function (error) {
                    // Error search results, show the error in the UI and
                    // unset our progress flag.
                    $scope.error = error;
                    $scope.isSearching = false;
                }
            );
        };

        /**
         * Update the page size preference and re-search.
         */
        $scope.updatePageSize = function (value) {
            Preference.set('project_detail_page_size', value).then(
                function () {
                    pageSize = value;
                    $scope.search();
                }
            );
        };

        /**
         * Next page of the results.
         */
        $scope.nextPage = function () {
            $scope.searchOffset += pageSize;
            $scope.search();
        };

        /**
         * Previous page of the results.
         */
        $scope.previousPage = function () {
            $scope.searchOffset -= pageSize;
            if ($scope.searchOffset < 0) {
                $scope.searchOffset = 0;
            }
            $scope.search();
        };

        $scope.newStory = function () {
            NewStoryService.showNewStoryModal($scope.project.id)
                .then(function (story) {
                    // On success, go to the story detail
                    $state.go('sb.story.detail', {storyId: story.id});
                }
            );
        };

        // Initialize the view with a default search.
        resetScope();
        $scope.project.$promise.then(function() {
            $scope.search();
        });

        // GET list of story subscriptions
        var cuPromise = CurrentUser.resolve();

        cuPromise.then(function(user){
            $scope.storySubscriptions = SubscriptionList.subsList(
                'story', user);
        });
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * This controller provides initialization logic for the generic search view.
 */
angular.module('sb.search').controller('SearchController',
    function ($log, $q, $scope, Criteria, $location, SearchHelper) {
        'use strict';

        /**
         * Default criteria, potentially populated by the q param.
         *
         * @type {Array}
         */
        var params = $location.search();
        $scope.defaultCriteria = SearchHelper.parseParameters(params);

        /**
         * List of resource types which this view will be searching on.
         *
         * @type {string[]}
         */
        $scope.resourceTypes = ['TaskStatus', 'Story', 'Project', 'User',
            'Task', 'ProjectGroup', 'Board', 'Worklist'];
    }
);

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the 'License'); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The sole purpose of this controller is to allow a user to search for valid
 * search/filter criteria for various resources, and expose chosen criteria
 * to the scope. These criteria may be static or asynchronously loaded, and
 * may be property filters (title = foo) or resource filters (story_id = 22).
 */
angular.module('sb.search').controller('SearchCriteriaController',
    function ($log, $q, $scope, $location, $injector, Criteria) {
        'use strict';

        /**
         * Valid sets of resources that can be searched on. The default
         * assumes no resources may be searched.
         */
        var resourceTypes = [];

        /**
         * Managed list of active criteria tags.
         *
         * @type {Array}
         */
        $scope.criteria = [];

        /**
         * Initialize this controller with different resource types and
         * default search criteria.
         *
         * @param types
         * @param defaultCriteria
         */
        $scope.init = function (types, defaultCriteria) {
            resourceTypes = types || $scope.resourceTypes || resourceTypes;
            if (!!defaultCriteria) {
                defaultCriteria.then(function(criteria) {
                    $scope.criteria = criteria;
                });
            } else {
                $scope.criteria = [];
            }
            $scope.searchForCriteria =
                Criteria.buildCriteriaSearch(resourceTypes, 5);
        };

        $scope.$on('refresh-types', function() {
            $scope.init();
        });

        $scope.rewriteQueryString = function() {
            var params = {};
            angular.forEach(resourceTypes, function(resourceName) {
                var resource = $injector.get(resourceName);
                angular.forEach($scope.criteria, function() {
                    var criteriaMap = resource.criteriaMap($scope.criteria);
                    angular.extend(params, criteriaMap);
                });
            });
            $location.search(params);
        };

        /**
         * When a criteria is added, make sure we remove all previous criteria
         * that have the same type.
         */
        $scope.addCriteria = function (item) {
            for (var i = $scope.criteria.length - 1; i >= 0; i--) {
                var cItem = $scope.criteria[i];

                // Don't remove exact duplicates.
                if (cItem === item) {
                    continue;
                }

                if (item.type === cItem.type && item.type !== 'Tags') {
                    $scope.criteria.splice(i, 1);
                }
            }
            $scope.rewriteQueryString();
        };

        /**
         * Remove a criteria
         */
        $scope.removeCriteria = function (item) {
            var idx = $scope.criteria.indexOf(item);
            if (idx > -1) {
                $scope.criteria.splice(idx, 1);
            }
            $scope.rewriteQueryString();
        };

        /**
         * Validate criteria when the list changes.
         */
        $scope.$watchCollection(function () {
            return $scope.criteria;
        }, function () {
            // Now, check all search resources to see if we have _any_ valid
            // criteria.
            $scope.hasSomeValidCriteria = false;
            resourceTypes.forEach(function (resourceName) {
                var validCriteria = Criteria
                    .filterCriteria(resourceName, $scope.criteria);

                if (validCriteria.length === $scope.criteria.length) {
                    $scope.hasSomeValidCriteria = true;
                }
            });
        });

        /**
         * Search for available search criteria.
         */
        $scope.searchForCriteria = function () {
            var deferred = $q.defer();
            deferred.resolve([]);
            return deferred.promise;
        };
    }
);

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A directive that displays a list of projects in a table.
 *
 * @see ProjectListController
 */
angular.module('sb.search').directive('searchResults',
    function ($log, $parse, Criteria, $injector, Preference) {
        'use strict';

        return {
            restrict: 'A',
            scope: true,
            link: function ($scope, $element, args) {

                // Extract the resource type.
                var resourceName = args.searchResource;
                var pageSize = args.searchPageSize ||
                    Preference.get('page_size');
                var searchWithoutCriteria =
                    args.searchWithoutCriteria === 'true';
                var criteria = [];

                $scope.isSearching = false;
                $scope.searchResults = [];

                /**
                 * The field to sort on.
                 *
                 * @type {string}
                 */
                $scope.sortField = 'updated_at';

                /**
                 * The direction to sort on.
                 *
                 * @type {string}
                 */
                $scope.sortDirection = 'desc';

                /**
                 * Handle error result.
                 */
                function handleErrorResult() {
                    $scope.isSearching = false;
                }

                /**
                 * Handle search result.
                 *
                 * @param results
                 */
                function handleSearchResult(results, headers) {
                    $scope.searchTotal =
                        parseInt(headers('X-Total')) || results.length;
                    $scope.searchOffset = parseInt(headers('X-Offset')) || 0;
                    $scope.searchLimit = parseInt(headers('X-Limit')) || 0;
                    $scope.searchResults = results;
                    $scope.isSearching = false;
                }

                /**
                 * Update the results when the criteria change
                 */
                function updateResults() {

                    // Extract the valid criteria from the provided ones.
                    $scope.validCriteria = Criteria
                        .filterCriteria(resourceName, criteria);

                    // You have criteria, but they may not be valid.
                    $scope.hasCriteria = criteria.length > 0;

                    // You have criteria, and all of them are valid for
                    // this resource.
                    $scope.hasValidCriteria =
                        searchWithoutCriteria ||
                        ($scope.validCriteria.length === criteria.length &&
                        $scope.hasCriteria);

                    // No need to search if our criteria aren't valid.
                    if (!$scope.hasValidCriteria) {
                        $scope.searchResults = [];
                        $scope.isSearching = false;
                        return;
                    }

                    var params = Criteria.mapCriteria(resourceName,
                        $scope.validCriteria);
                    var resource = $injector.get(resourceName);

                    if (!resource) {
                        $log.error('Invalid resource name: ' +
                        resourceName);
                        return;
                    }

                    // Apply paging.
                    params.limit = pageSize;
                    params.offset = $scope.searchOffset;

                    // If we don't actually have search criteria, issue a
                    // browse. Otherwise, issue a search.
                    $scope.isSearching = true;
                    if (!params.hasOwnProperty('q')) {
                        params.sort_field = $scope.sortField;
                        params.sort_dir = $scope.sortDirection;

                        resource.browse(params,
                            handleSearchResult,
                            handleErrorResult);
                    } else {
                        resource.search(params,
                            handleSearchResult,
                            handleErrorResult);
                    }
                }

                /**
                 * Allowing sorting stories in search results by certain fields.
                 */
                $scope.stories_sort_field = 'Sort Field';
                $scope.sort_stories_by_field = function(selected){
                    $scope.stories_sort_field = selected.toString();
                    var res = $scope.searchResults;
                    switch(selected) {
                        case 'Title':
                            res.sort(function compare(a, b){
                                if (a.title < b.title){
                                    return -1;
                                }
                                if (a.title > b.title){
                                    return 1;
                                }
                                return 0;
                            });
                            break;
                        case 'Tags':
                            res.sort(function compare(a, b){
                                if (a.tags < b.tags){
                                    return -1;
                                }
                                if (a.tags > b.tags){
                                    return 1;
                                }
                                return 0;
                            });
                            break;
                        case 'Status':
                            res.sort(function compare(a, b){
                                if (a.status < b.status){
                                    return -1;
                                }
                                if (a.status > b.status){
                                    return 1;
                                }
                                return 0;
                            });
                            break;
                        case 'Updated':
                            res.sort(function compare(a, b){
                                if (a.updated_at < b.updated_at){
                                    return -1;
                                }
                                if (a.updated_at > b.updated_at){
                                    return 1;
                                }
                                return 0;
                            });
                            break;
                    }
                };

                /**
                 * Allowing sorting tasks in search results by certain fields.
                 */
                $scope.tasks_sort_field = 'Sort Field';
                $scope.sort_tasks_by_field = function(selected){
                    $scope.tasks_sort_field = selected.toString();
                    var res = $scope.searchResults;
                    switch(selected) {
                        case 'Story':
                            res.sort(function compare(a, b){
                                if (a.story_id < b.story_id){
                                    return -1;
                                }
                                if (a.story_id > b.story_id){
                                    return 1;
                                }
                                return 0;
                            });
                            break;
                        case 'Status':
                            res.sort(function compare(a, b){
                                if (a.status < b.status){
                                    return -1;
                                }
                                if (a.status > b.status){
                                    return 1;
                                }
                                return 0;
                            });
                            break;
                        case 'Title':
                            res.sort(function compare(a, b){
                                if (a.title < b.title){
                                    return -1;
                                }
                                if (a.title > b.title){
                                    return 1;
                                }
                                return 0;
                            });
                            break;
                        case 'Project':
                            res.sort(function compare(a, b){
                                if (a.project_id < b.project_id){
                                    return -1;
                                }
                                if (a.project_id > b.project_id){
                                    return 1;
                                }
                                return 0;
                            });
                            break;
                        case 'Created at':
                            res.sort(function compare(a, b){
                                if (a.created_at < b.created_at){
                                    return -1;
                                }
                                if (a.created_at > b.created_at){
                                    return 1;
                                }
                                return 0;
                            });
                            break;
                        case 'Updated Since':
                            res.sort(function compare(a, b){
                                if (a.updated_at < b.updated_at){
                                    return -1;
                                }
                                if (a.updated_at > b.updated_at){
                                    return 1;
                                }
                                return 0;
                            });
                            break;
                    }
                };

                /**
                 * Update the page size preference and re-search.
                 */
                $scope.updatePageSize = function (value) {
                    Preference.set('page_size', value).then(
                        function () {
                            pageSize = value;
                            updateResults();
                        }
                    );
                };

                /**
                 * Toggle the filter ID and direction in the UI.
                 *
                 * @param fieldName
                 */
                $scope.toggleFilter = function (fieldName) {
                    if ($scope.sortField === fieldName) {
                        $scope.sortDirection =
                            $scope.sortDirection === 'asc' ? 'desc' : 'asc';
                    } else {
                        $scope.sortField = fieldName;
                        $scope.sortDirection = 'desc';
                    }
                    updateResults();
                };

                /**
                 * Next page of the results.
                 */
                $scope.nextPage = function () {
                    $scope.searchOffset += pageSize;
                    updateResults();
                };

                /**
                 * Previous page in the results.
                 */
                $scope.previousPage = function () {
                    $scope.searchOffset -= pageSize;
                    if ($scope.searchOffset < 0) {
                        $scope.searchOffset = 0;
                    }
                    updateResults();
                };

                // Watch for changing criteria
                $scope.$watchCollection(
                    $parse(args.searchCriteria),
                    function (results) {
                        criteria = results;
                        updateResults();
                    });
            }
        };
    });

/**
 * Copyright (c) 2016 Codethink Ltd
 * Copyright (c) 2017 Adam Coldrick
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A service providing helper functions for search views.
 */
angular.module('sb.search').factory('SearchHelper',
    function(User, Project, ProjectGroup, Story, Task, Criteria,
             $filter, $q) {
        'use strict';

        /**
         * Create search criteria based on some given parameters.
         */
        function parseParameters(params) {
            var criteria = [];
            var promises = [];
            if (params.q) {
                criteria.push(
                    Criteria.create('Text', params.q)
                );
            }
            if (params.title) {
                criteria.push(
                    Criteria.create('Text', params.title)
                );
            }
            if (params.status) {
                criteria.push(
                    Criteria.create('StoryStatus', params.status,
                                    $filter('capitalize')(params.status))
                );
            }
            if (params.tags) {
                if (params.tags.constructor === Array) {
                    angular.forEach(params.tags, function(tag) {
                        criteria.push(
                            Criteria.create('Tags', tag, tag)
                        );
                    });
                } else {
                    criteria.push(
                        Criteria.create('Tags', params.tags, params.tags)
                    );
                }
            }
            if (params.assignee_id || params.creator_id) {
                var id = params.assignee_id || params.creator_id;
                var userPromise = User.get({'id': id}).$promise;
                promises.push(userPromise);

                userPromise.then(function(result) {
                        criteria.push(
                            Criteria.create('User',
                                            params.assignee_id,
                                            result.full_name + ' <'
                                            + result.email + '>')
                        );
                    }
                );
            }
            if (params.project_id) {
                var projectParams = {'id': params.project_id};
                var projectPromise = Project.get(projectParams).$promise;
                promises.push(projectPromise);

                projectPromise.then(function(result) {
                        criteria.push(
                            Criteria.create('Project',
                                            params.project_id,
                                            result.name)
                        );
                    }
                );
            }
            if (params.project_group_id) {
                var groupParams = {'id': params.project_group_id};
                var groupPromise = ProjectGroup.get(groupParams).$promise;
                promises.push(groupPromise);

                groupPromise.then(function(result) {
                        criteria.push(
                            Criteria.create('ProjectGroup',
                                            params.project_group_id,
                                            result.title)
                        );
                    }
                );
            }
            if (params.story_id) {
                var storyParams = {'id': params.story_id};
                var storyPromise = Story.get(storyParams).$promise;
                promises.push(storyPromise);

                storyPromise.then(function(result) {
                        criteria.push(
                            Criteria.create('Story',
                                            params.story_id,
                                            result.title)
                        );
                    }
                );
            }
            if (params.task_id) {
                var taskParams = {'id': params.task_id};
                var taskPromise = Task.get(taskParams).$promise;
                promises.push(taskPromise);

                taskPromise.then(function(result) {
                        criteria.push(
                            Criteria.create('Task',
                                            params.task_id,
                                            result.title)
                        );
                    }
                );
            }

            var deferred = $q.defer();
            $q.all(promises).then(function() {
                deferred.resolve(criteria);
            });

            return deferred.promise;
        }

        return {
            parseParameters: parseParameters
        };
    }
);

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the 'License'); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A service which centralizes management of search criteria: Creation,
 * validation, filtering, criteria-to-parameter mapping, and more.
 */
angular.module('sb.services').service('Criteria',
    function ($q, $log, $injector, Preference) {
        'use strict';

        return {

            /**
             * This method takes a set of criteria, and filters out the
             * ones not valid for the passed resource.
             *
             * @param resourceName The name of the resource to filter for.
             * @param criteria The list of criteria.
             * @return {Array} A map of URL parameters.
             */
            filterCriteria: function (resourceName, criteria) {

                var resource = $injector.get(resourceName);

                // Sanity check: If we don't have this resource, wat?
                if (!resource || !resource.hasOwnProperty('criteriaFilter')) {
                    $log.warn('Attempting to filter criteria for unknown ' +
                        'resource "' + resourceName + '"');
                    return [];
                }

                return resource.criteriaFilter(criteria);
            },

            /**
             * This method takes a set of criteria, and maps them against the
             * query parameters available for the provided resource. It will
             * skip any items not valid for this resource, and return an
             * array of criteria that are valid
             *
             * @param resourceName
             * @param criteria
             * @return A map of URL parameters.
             */
            mapCriteria: function (resourceName, criteria) {
                var resource = $injector.get(resourceName);

                // Sanity check: If we don't have this resource, wat?
                if (!resource || !resource.hasOwnProperty('criteriaMap')) {
                    $log.warn('Attempting to map criteria for unknown ' +
                        'resource "' + resourceName + '"');
                    return {};
                }

                return resource.criteriaMap(criteria);
            },

            /**
             * Create a new build criteria object.
             *
             * @param type The type of the criteria tag.
             * @param value Value of the tag. Unique DB ID, or text string.
             * @param title The title of the criteria tag.
             * @returns {Criteria}
             */
            create: function (type, value, title) {
                title = title || value;
                return {
                    'type': type,
                    'value': value,
                    'title': title
                };
            },

            /**
             * Rather than actually performing a search, this method returns a
             * customized lambda that will perform our browse search for us.
             *
             * @param types An array of resource types to browse.
             * @param pageSize An optional page size for the criteria. Defaults
             * to the global page_size preference.
             */
            buildCriteriaSearch: function (types, pageSize) {
                pageSize = pageSize || Preference.get('page_size');

                var resolvers = [];
                types.forEach(function (type) {
                    // Retrieve an instance of the declared resource.
                    var resource = $injector.get(type);

                    if (!resource.hasOwnProperty('criteriaResolvers')) {
                        $log.warn('Resource type "' + type +
                            '" does not implement criteriaResolvers.');
                        return;
                    }

                    resource.criteriaResolvers().forEach(function (resolver) {
                        if (resolvers.indexOf(resolver) === -1) {
                            resolvers.push(resolver);
                        }
                    });
                });

                /**
                 * Construct the search lambda that issues the search
                 * and assembles the results.
                 */
                return function (searchString) {
                    var deferred = $q.defer();

                    // Clear the criteria
                    var promises = [];

                    resolvers.forEach(function (resolver) {
                        promises.push(resolver(searchString, pageSize));
                    });

                    // Wrap everything into a collective promise
                    $q.all(promises).then(function (results) {
                        var criteria = [];

                        results.forEach(function (result) {
                            result.forEach(function (item) {
                                criteria.push(item);
                            });
                        });
                        deferred.resolve(criteria);
                    });

                    // Return the search promise.
                    return deferred.promise;
                };
            },

            /**
             * This method takes a set of criteria, and filters out the
             * ones not valid for the passed resource.
             *
             * @param parameterMap A map of criteria types and parameters
             * in the search query they correspond to.
             * @return {Function} A criteria filter for the passed parameters.
             */
            buildCriteriaFilter: function (parameterMap) {
                return function (criteria) {
                    var filteredCriteria = [];

                    criteria.forEach(function (item) {
                        if (parameterMap.hasOwnProperty(item.type)) {
                            filteredCriteria.push(item);
                        }
                    });
                    return filteredCriteria;
                };
            },

            /**
             * This method takes a set of criteria, and maps them against the
             * query parameters available for the provided resource. It will
             * skip any items not valid for this resource, and return an
             * array of criteria that are valid
             *
             * @param parameterMap A map of criteria types and parameters
             * in the search query they correspond to.
             * @return {Function} A criteria mapper for the passed parameters.
             */
            buildCriteriaMap: function (parameterMap) {
                return function (criteria) {
                    var params = {};

                    criteria.forEach(function (item) {
                        if (parameterMap.hasOwnProperty(item.type)) {
                            if (parameterMap[item.type] === 'tags') {
                                if (!('tags' in params)) {
                                    params.tags = [item.value];
                                } else {
                                    params.tags.push(item.value);
                                }
                            } else {
                                params[parameterMap[item.type]] = item.value;
                            }
                        }
                    });
                    return params;
                };
            }
        };
    }
);

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */


/**
 * This criteria resolver may be injected by individual resources that accept a
 * Story Status search parameters.
 */
angular.module('sb.services').factory('StoryStatus',
    function (Criteria, $q) {
        'use strict';

        /**
         * A list of valid story status items.
         *
         * @type {*[]}
         */
        var validStatusCriteria = [
            Criteria.create('StoryStatus', 'active', 'Active'),
            Criteria.create('StoryStatus', 'merged', 'Merged'),
            Criteria.create('StoryStatus', 'invalid', 'Invalid')
        ];

        /**
         * Return a criteria resolver for story status.
         */
        return {
            criteriaResolver: function (searchString) {
                var deferred = $q.defer();
                searchString = searchString || ''; // Sanity check
                searchString = searchString.toLowerCase(); // Lowercase search

                var criteria = [];
                validStatusCriteria.forEach(function (criteriaItem) {
                    var title = criteriaItem.title.toLowerCase();

                    // If we match the title, OR someone is explicitly typing in
                    // 'status'
                    if (title.indexOf(searchString) > -1 ||
                        'status'.indexOf(searchString) === 0) {
                        criteria.push(criteriaItem);
                    }
                });
                deferred.resolve(criteria);

                return deferred.promise;
            }
        };
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */


/**
 * This criteria resolver may be injected by individual resources that accept a
 * Task Priority search parameters.
 */
angular.module('sb.services').factory('TaskPriority',
    function (Criteria, $q) {
        'use strict';

        /**
         * A list of valid story status items.
         *
         * @type {*[]}
         */
        var validPriorityCriteria = [
            Criteria.create('TaskPriority', 'high', 'High'),
            Criteria.create('TaskPriority', 'medium', 'Medium'),
            Criteria.create('TaskPriority', 'low', 'Low')
        ];

        /**
         * Return a criteria resolver for story status.
         */
        return {
            criteriaResolver: function (searchString) {
                var deferred = $q.defer();
                searchString = searchString || ''; // Sanity check
                searchString = searchString.toLowerCase(); // Lowercase search

                var criteria = [];
                validPriorityCriteria.forEach(function (criteriaItem) {
                    var title = criteriaItem.title.toLowerCase();

                    // If we match the title, OR someone is explicitly typing in
                    // 'status'
                    if (title.indexOf(searchString) > -1 ||
                        'priority'.indexOf(searchString) === 0) {
                        criteria.push(criteriaItem);
                    }
                });
                deferred.resolve(criteria);

                return deferred.promise;
            }
        };
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */


/**
 * This criteria resolver may be injected by individual resources that accept a
 * plain text search parameter.
 */
angular.module('sb.services').factory('Text',
    function (Criteria, $q) {
        'use strict';

        /**
         * Return a text search parameter constructed from the passed search
         * string.
         */
        return {
            criteriaResolver: function (searchString) {
                var deferred = $q.defer();

                deferred.resolve([Criteria.create('Text', searchString)]);

                return deferred.promise;
            }
        };
    });

/*
 * Copyright (c) 2016 Codethink Limited.
 *
 * Licensed under the Apache License, Version 2.0 (the 'License'); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Directive which displays a date/time picker.
 */
angular.module('sb.services')
    .directive('calendar', function(moment) {
        'use strict';

        return {
            restrict: 'E',
            templateUrl: 'app/services/template/calendar.html',
            scope: {
                selectedDate: '='
            },
            link: function(scope) {
                function buildWeek(date) {
                    var days = [];
                    for (var i = 0; i < 7; i++) {
                        days.push({
                            name: date.format('dd').substring(0, 1),
                            number: date.date(),
                            isToday: date.isSame(new Date(), 'day'),
                            isCurrentMonth:
                                date.month() === scope.month.month(),
                            isSelected: date.isSame(scope.selectedDate, 'day'),
                            date: date
                        });
                        date = date.clone();
                        date.add(1, 'd');
                    }
                    return days;
                }

                function buildMonth(start) {
                    scope.weeks = [];
                    var date = start.clone();
                    var monthIndex = date.month();
                    var done = false;
                    var count = 0;

                    date.date(1).isoWeekday(1);
                    while (!done) {
                        scope.weeks.push({days: buildWeek(date.clone())});
                        date.add(1, 'w');
                        monthIndex = date.month();
                        done = count++ > 2 && monthIndex !== start.month();
                    }
                }

                scope.select = function(day) {
                    scope.selectedDate = day.date;
                    if (scope.selectedDate.isBefore(scope.month, 'month')) {
                        scope.previous();
                    } else if (
                        scope.selectedDate.isAfter(scope.month, 'month')) {
                        scope.next();
                    }
                    buildMonth(scope.selectedDate.clone());
                };

                scope.previous = function() {
                    var start = scope.month.clone();
                    start.subtract(1, 'month');
                    scope.month.subtract(1, 'month');
                    buildMonth(start);
                };

                scope.next = function() {
                    var start = scope.month.clone();
                    start.add(1, 'month');
                    scope.month.add(1, 'month');
                    buildMonth(start);
                };

                scope.incrementTime = function(unit) {
                    scope.selectedDate.add(1, unit);
                    if (!scope.selectedDate.isSame(scope.month, 'month')) {
                        scope.next();
                    }
                    buildMonth(scope.selectedDate.clone());
                };

                scope.decrementTime = function(unit) {
                    scope.selectedDate.subtract(1, unit);
                    if (!scope.selectedDate.isSame(scope.month, 'month')) {
                        scope.previous();
                    }
                    buildMonth(scope.selectedDate.clone());
                };

                scope.toggleEdit = function(unit) {
                    if (unit === 'hour') {
                        if (scope.editHour) {
                            scope.selectedDate.hour(scope.editedHour);
                        } else {
                            scope.editedHour = scope.selectedDate.hour();
                        }
                        scope.editHour = !scope.editHour;
                    } else if (unit === 'minute') {
                        if (scope.editMinute) {
                            scope.selectedDate.minute(scope.editedMinute);
                        } else {
                            scope.editedMinute = scope.selectedDate.minute();
                        }
                        scope.editMinute = !scope.editMinute;
                    }
                };

                scope.selectedDate = moment(scope.selectedDate) || moment();
                scope.month = scope.selectedDate.clone();

                buildMonth(scope.selectedDate.clone());
            }
        };
    });

/*
 * Copyright (c) 2015 Codethink Limited.
 *
 * Licensed under the Apache License, Version 2.0 (the 'License'); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Service for rendering text as markdown.
 */
angular.module('sb.services')
    .directive('insertMarkdown', function($sanitize, $window) {
        'use strict';

        var md = $window.markdownit({
            breaks: true,
            html: true,
            highlight: function(code, lang) {
                if (lang && $window.hljs.getLanguage(lang)) {
                    return $window.hljs.highlight(lang, code, true).value;
                }
                return ''; // Don't highlight if no language specified
            },
            linkify: true
        });

        return {
            restrict: 'E',
            scope: {
                content: '='
            },
            link: function(scope, elem) {
                scope.$watch('content', function(newVal) {
                    var html = md.render(newVal);
                    elem.html('<div>' + $sanitize(html) + '</div>');
                }, true);
            }
        };
    });

/*
 * Copyright (c) 2015 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A directive which displays a responsive result set size :
 * (showing x to y of z)
 */
angular.module('sb.services').directive('resultSetPager',
    function () {
        'use strict';

        return {
            restrict: 'E',
            templateUrl: 'app/services/template/result_set_pager.html',
            scope: {
                total: '=',
                offset: '=',
                limit: '=',
                listType: '=',
                minimalPager: '=',
                pageSize: '&onPageSize',
                nextPage: '&onNextPage',
                previousPage: '&onPreviousPage'
            }
        };
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * An HTTP request interceptor that pays attention to POST/PUT/DELETE operations
 * on specific resources, and replaces the local cached data with the resulting
 * value.
 */
angular.module('sb.services').factory('httpCacheHandler',
    function ($q, $cacheFactory) {
        'use strict';

        var $httpDefaultCache = $cacheFactory.get('$http');

        return {
            /**
             * Handle a success response.
             */
            response: function (response) {
                var method = response.config.method;
                var url = response.config.url;
                var obj = response.data;

                // Ignore GET methods.
                switch (method) {
                    case 'POST':
                        if (obj.hasOwnProperty('id')) {
                            $httpDefaultCache.put(url + '/' + obj.id, obj);
                        }
                        break;
                    case 'PUT':
                        $httpDefaultCache.put(url, obj);
                        break;
                    case 'DELETE':
                        $httpDefaultCache.remove(url);
                        break;
                    default:
                        break;
                }

                return response;
            }
        };
    })
    // Attach the HTTP interceptor.
    .config(function ($httpProvider) {
        'use strict';
        $httpProvider.interceptors.push('httpCacheHandler');
    });

/*
 * Copyright (c) 2013 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * An HTTP request interceptor that broadcasts response status codes to the
 * rest of the application as notifications. These events are broadcast before
 * the error response itself is passed back to the receiving closure, so please
 * keep that in mind as you base your application logic on it.
 *
 * @author Michael Krotscheck
 */
angular.module('sb.services')
    // Create an HTTP Error Broadcaster that intercepts requests and lets the
    // rest of the application know about what happened.
    .factory('httpErrorBroadcaster',
    function ($q, $rootScope, Notification, Severity) {
        'use strict';

        function sendEvent(severity, response) {
            // Only send an event if a status is passed.
            if (!!response.status) {
                Notification.send('http', response.status, severity, response);
            }
        }

        return {
            /**
             * Handle a success response.
             */
            response: function (response) {
                if (!!response) {
                    sendEvent(Severity.SUCCESS, response);
                }
                return response;
            },

            /**
             * Handle a fail response.
             */
            responseError: function (response) {
                if (!!response) {
                    sendEvent(Severity.ERROR, response);
                }

                return $q.reject(response);
            }
        };
    })
    // Attach the HTTP interceptor.
    .config(function ($httpProvider) {
        'use strict';
        $httpProvider.interceptors.unshift('httpErrorBroadcaster');
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Notification interceptors for this library.
 */
angular.module('sb.services')
    .run(function (Notification, Priority) {
        'use strict';

        /**
         * Template load requests are done via $http, so we need to filter
         * those out first.
         */
        function filterTemplateRequests(message) {
            if (message.type !== 'http') {
                return false;
            }

            var request = message.cause;
            var url = request.config.url;

            if (url.substr(-5) === '.html') {
                return true;
            }
        }

        /**
         * A notification interceptor that filters successful HTTP requests.
         * It's registered at priority 999 (the lowest) so that other
         * interceptors can get access to this message first (ex: statistics).
         */
        function filterSuccessful(message) {
            var response = message.cause;
            if (message.type !== 'http' || !response) {
                return false;
            }

            // All successful requests are filtered out.
            var successful_requests = [200, 201, 202, 203, 204,
                                       205, 206, 207, 208, 226];
            if (successful_requests.indexOf(response.status) >= 0 ) {
                return true;
            }
        }

        /**
         * A notification interceptor that rewrites HTTP status codes to
         * human readable messages.
         */
        function rewriteHttpStatus(message) {

            if (message.type !== 'http') {
                // Do nothing.
                return;
            }

            var httpStatus = message.message;
            var request = message.cause;

            if (!httpStatus || !request || !request.data) {
                return;
            }
            var data = request.data;
            var method = request.config.method;
            var url = request.config.url;

            message.message = httpStatus + ': ' + method + ' ' + url + ': ';

            if (data.hasOwnProperty('faultstring')) {
                message.message += data.faultstring;
            } else if (data.hasOwnProperty('field') &&
                       data.hasOwnProperty('message')) {
                message.message += data.field + ': ' + data.message;
            } else {
                message.message += 'No error details available.';
            }
        }

        // Apply the interceptors.
        Notification.intercept(filterTemplateRequests, Priority.BEFORE);
        Notification.intercept(filterSuccessful, Priority.LAST);
        Notification.intercept(rewriteHttpStatus, Priority.AFTER);
    });

/*
 * Copyright (c) 2015 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */


/**
 * Preference service, a convenience-cashing API in front of our UserPreference
 * service, with resolving/refresh functionality.
 */
angular.module('sb.services').provider('Preference',
    function () {
        'use strict';

        /**
         * Singleton preference provider.
         */
        var preferenceInstance = null;

        /**
         * Registered default preferences.
         */
        var defaults = {};

        /**
         * Each module can manually declare its own preferences that it would
         * like to keep track of, as well as set a default. During the config()
         * phase, inject the Preference Provider and call 'addPreference()' to
         * do so. An example is available at the bottom of this file.
         */
        this.addPreference = function (name, defaultValue) {
            defaults[name] = defaultValue;
        };

        /**
         * The actual preference implementation.
         */
        function Preference($q, $log, Session, AccessToken, UserPreference,
                            SessionState) {

            /**
             * The currently loaded preferences.
             */
            var preferences = {};

            /**
             * This function resolves the user preferences. If a valid session
             * is resolvable, it will load the preferences for the user.
             * Otherwise it will resolve the default preferences.
             *
             * @returns {deferred.promise|*}
             */
            this.resolveUserPreferences = function () {
                var deferred = $q.defer();

                // First resolve the session.
                var sessionPromise = Session.resolveSessionState();
                sessionPromise.then(
                    function (state) {
                        if (state === SessionState.LOGGED_IN) {
                            UserPreference.get({id: AccessToken.getIdToken()},
                                function (prefs) {
                                    deferred.resolve(prefs);
                                }, function () {
                                    deferred.resolve(defaults);
                                });
                        } else {
                            deferred.resolve(defaults);
                        }
                    },
                    function () {
                        deferred.resolve(defaults);
                    }
                );

                return deferred.promise;
            };

            /**
             * Create a composite result of preference defaults and server
             * provided defaults.
             */
            this.getAll = function () {
                var result = {};
                for (var def_key in defaults) {
                    result[def_key] = this.get(def_key);
                }

                for (var key in preferences) {
                    result[key] = preferences[key];
                }

                return result;
            };

            /**
             * Save all the preferences in the passed hash.
             */
            this.saveAll = function (newPrefs) {
                // Update the preferences.
                for (var key in defaults) {
                    if (preferences.hasOwnProperty(key)) {
                        if (preferences[key] !== newPrefs[key]) {
                            $log.debug('Preference Change: ' + key + ' -> ' +
                                newPrefs[key]);
                            preferences[key] = newPrefs[key];
                        }
                    }
                }
                return this.save();
            };

            /**
             * Returns the value for a given preference.
             */
            this.get = function (key) {

                // Is this a valid preference?
                if (!defaults.hasOwnProperty(key)) {
                    $log.warn('Attempt to get unregistered preference: ' +
                        key);
                    return null;
                }

                // If the value is unset, and we have a default,
                // set that.
                if (!preferences.hasOwnProperty(key)) {
                    $log.warn('Setting default preference: ',
                        key, defaults[key]);
                    this.set(key, defaults[key], true);
                }

                return preferences[key];
            };

            /**
             * Save a preference and return the saving promise.
             */
            this.set = function (key, value, isDefault) {
                // Is this a valid preference?
                if (!defaults.hasOwnProperty(key)) {
                    $log.warn('Attempt to set unregistered preference: ' +
                        key);
                    return null;
                }

                // Store the preference.
                preferences[key] = value;

                return this.save(isDefault);
            };

            /**
             * Resolve the preferences.
             */
            this.refresh = function () {
                var deferred = $q.defer();

                // This should never fail, see implementation above.
                this.resolveUserPreferences().then(
                    function (newPrefs) {
                        preferences = newPrefs;
                        deferred.resolve(preferences);
                    }
                );

                return deferred.promise;
            };

            /**
             * Private save method.
             */
            this.save = function (isDefault) {
                var deferred = $q.defer();

                // Don't save default preferences, but just checking there is
                // no $save() method is insufficient. Also, if there is no
                // $save() method, don't try to save the preferences (this is
                // the case when there is no logged in user).
                if (isDefault) {
                    deferred.resolve();
                } else if (!preferences.$save) {
                    deferred.resolve();
                } else {
                    preferences.$save({id: AccessToken.getIdToken()},
                        function () {
                            deferred.resolve();
                        }, function () {
                            deferred.resolve();
                        });
                }

                return deferred.promise;
            };
        }

        /**
         * Factory getter - returns a configured instance of preference
         * provider, as needed.
         */
        this.$get =
            function ($injector) {
                if (!preferenceInstance) {
                    preferenceInstance = $injector.instantiate(Preference);
                }
                return preferenceInstance;
            };
    })
    .config(function (PreferenceProvider) {
        'use strict';

        // WARNING: In all modules OTHER than the services module, this config
        // block can appear anywhere as long as this module is listed as a
        // dependency. In the services module, the config() block must appear
        // AFTER the provider block. For more information,
        // @see https://github.com/angular/angular.js/issues/6723

        // Let our preference provider know about page_size.
        PreferenceProvider.addPreference('page_size', 10);
        PreferenceProvider.addPreference('story_detail_page_size', 10);
        PreferenceProvider.addPreference(
            'project_group_detail_projects_page_size', 10);
        PreferenceProvider.addPreference(
            'project_group_detail_stories_page_size', 10);
        PreferenceProvider.addPreference('project_detail_page_size', 10);

        // Let our preference provider know about email preferences.
        PreferenceProvider.addPreference('plugin_email_enable', 'false');

        // Let our preference provider know about notifications preferences.
        PreferenceProvider.addPreference(
            'receive_notifications_worklists', false);
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * This provider injects a sane default for the storyboardApiBase property. It
 * may be overridden simply by injecting a constant of the same name in any
 * module which lists sb.services as a dependency.
 *
 * @author Michael Krotscheck
 */
angular.module('sb.services').constant('storyboardApiBase', '/api/v1');

/*
 * Copyright (c) 2015 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A preference resolver that allows route-level preloading of preferences.
 */
angular.module('sb.services').constant('PreferenceResolver',
    {
        /**
         * Resolves all preferences.
         */
        resolvePreferences: function (Preference) {
            'use strict';
            return Preference.refresh();
        }
    });

/*
 * Copyright (c) 2015 Codethink Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The angular resource abstraction that allows us to access projects groups.
 *
 * @see ResourceFactory
 * @author Adam Coldrick
 */
angular.module('sb.services').factory('Board',
    function (ResourceFactory, Worklist, $resource, storyboardApiBase) {
        'use strict';

        var resource = ResourceFactory.build(
            '/boards/:id',
            '/boards/search',
            {id: '@id'},
            false, true
        );

        var permissionsSignature = {
            'create': {
                method: 'POST'
            },
            'get': {
                method: 'GET',
                cache: false,
                isArray: true
            },
            'update': {
                method: 'PUT'
            }
        };

        resource.Permissions = $resource(
            storyboardApiBase + '/boards/:id/permissions',
            {id: '@id'},
            permissionsSignature
        );

        ResourceFactory.applySearch(
            'Board',
            resource,
            'title',
            {
                Text: 'title',
                Story: 'story_id',
                Task: 'task_id',
                User: 'creator_id'
            }
        );

        return resource;
    });

/*
 * Copyright (c) 2016 Codethink Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The angular resource abstraction that allows us to access branches and their
 * details.
 *
 * @see ResourceFactory
 * @author Adam Coldrick
 */
angular.module('sb.services').factory('Branch',
    function (ResourceFactory) {
        'use strict';

        var resource = ResourceFactory.build(
            '/branches/:id',
            '/branches/search',
            {id: '@id'}
        );

        ResourceFactory.applySearch(
            'Branch',
            resource,
            'name',
            {
              Text: 'q',
              Project: 'project_id'
            }
        );

        return resource;
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The angular resource abstraction that allows us to access discussions that
 * are surrounding stories.
 *
 * @see storyboardApiSignature
 */
angular.module('sb.services').factory('Comment',
    function (ResourceFactory, $resource, storyboardApiBase) {
        'use strict';

        var resource = ResourceFactory.build(
            '/stories/:story_id/comments/:id',
            '/stories/0/search',
            {
                id: '@id',
                story_id: '@story_id'
            }
        );

        var historySignature = {
            'get': {
                method: 'GET',
                isArray: true
            }
        };

        resource.History = $resource(
            storyboardApiBase + '/stories/:story_id/comments/:id/history',
            {
                id: '@id',
                story_id: '@story_id'
            },
            historySignature
        );

        return resource;
    });

/*
 * Copyright (c) 2016 Codethink Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The angular resource abstraction that allows us to access due dates.
 *
 * @see ResourceFactory
 * @author Adam Coldrick
 */
angular.module('sb.services').factory('DueDate',
    function (ResourceFactory, $resource, storyboardApiBase) {
        'use strict';

        var resource = ResourceFactory.build(
            '/due_dates/:id',
            '/due_dates/search',
            {id: '@id'},
            false, true
        );

        var permissionsSignature = {
            'create': {
                method: 'POST'
            },
            'get': {
                method: 'GET',
                cache: false,
                isArray: true
            },
            'update': {
                method: 'PUT'
            }
        };

        resource.Permissions = $resource(
            storyboardApiBase + '/due_dates/:id/permissions',
            {id: '@id'},
            permissionsSignature
        );

        ResourceFactory.applySearch(
            'DueDate',
            resource,
            'name',
            {
                Text: 'q',
                User: 'owner',
                Board: 'board_id',
                Worklist: 'worklist_id'
            }
        );

        return resource;
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The angular resource abstraction that allows us to access projects and their
 * details.
 *
 * @see ResourceFactory
 * @author Michael Krotscheck
 */
angular.module('sb.services').factory('Project',
    function (ResourceFactory) {
        'use strict';

        var resource = ResourceFactory.build(
            '/projects/:id',
            '/projects/search',
            {id: '@id'}
        );

        ResourceFactory.applySearch(
            'Project',
            resource,
            'name',
            {Text: 'q'}
        );

        return resource;
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The angular resource abstraction that allows us to access projects groups.
 *
 * @see ResourceFactory
 * @author Michael Krotscheck
 */
angular.module('sb.services').factory('ProjectGroup',
    function (ResourceFactory) {
        'use strict';

        var resource = ResourceFactory.build(
            '/project_groups/:id',
            '/project_groups/search',
            {id: '@id'}
        );

        ResourceFactory.applySearch(
            'ProjectGroup',
            resource,
            'title',
            {Text: 'title'}
        );

        return resource;
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The angular resource abstraction that allows us to access children of
 * project groups.
 *
 * @author Michael Krotscheck
 */
angular.module('sb.services').factory('ProjectGroupItem',
    function ($resource, storyboardApiBase) {
        'use strict';

        return $resource(storyboardApiBase +
                '/project_groups/:projectGroupId/projects/:id',
            {
                projectGroupId: '@projectGroupId',
                id: '@id'
            },
            {
                'create': {
                    method: 'PUT',
                    transformRequest: function () {
                        // The API endpoint takes no payload.
                        return '';
                    }
                },
                'delete': {
                    method: 'DELETE'
                },
                'query': {
                    method: 'GET',
                    isArray: true,
                    responseType: 'json'
                },
                'browse': {
                    method: 'GET',
                    isArray: true,
                    responseType: 'json',
                    cache: false
                }
            }
        );
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A browse service, which wraps common resources and their typeahead
 * resolution into a single service that returns a common result format.
 * It is paired with the Criteria service to provide a consistent data
 * format to identify resources independent of their actual schema.
 */
angular.module('sb.services').factory('Search',
    function ($q, $log, Project, Story, User, Criteria) {
        'use strict';

        return {

            /**
             * Search projects by search string.
             *
             * @param searchString A string to search by.
             * @return A promise that will resolve with the search results.
             */
            project: function (searchString) {
                // Search for projects...
                var deferred = $q.defer();

                Project.search({q: searchString},
                    function (result) {
                        // Transform the results to criteria tags.
                        var projResults = [];
                        result.forEach(function (item) {
                            projResults.push(
                                Criteria.create('project', item.id, item.name)
                            );
                        });
                        deferred.resolve(projResults);
                    }, function () {
                        deferred.resolve([]);
                    }
                );

                return deferred.promise;
            },

            /**
             * Search users by search string.
             *
             * @param searchString A string to search by.
             * @return A promise that will resolve with the search results.
             */
            user: function (searchString) {

                // Search for users...
                var deferred = $q.defer();
                User.search({q: searchString},
                    function (result) {
                        // Transform the results to criteria tags.
                        var userResults = [];
                        result.forEach(function (item) {
                            userResults.push(
                                Criteria.create('user', item.id,
                                                item.full_name,
                                                item.email)
                            );
                        });
                        deferred.resolve(userResults);
                    }, function () {
                        deferred.resolve([]);
                    }
                );

                return deferred.promise;
            },

            /**
             * Search stories by search string.
             *
             * @param searchString A string to search by.
             * @return A promise that will resolve with the search results.
             */
            story: function (searchString) {

                // Search for stories...
                var deferred = $q.defer();
                Story.search({q: searchString},
                    function (result) {
                        // Transform the results to criteria tags.
                        var storyResults = [];
                        result.forEach(function (item) {
                            storyResults.push(
                                Criteria.create('story', item.id, item.title)
                            );
                        });
                        deferred.resolve(storyResults);
                    }, function () {
                        deferred.resolve([]);
                    }
                );

                return deferred.promise;
            },


            /**
             * Search all resources by a provided search string.
             *
             * @param searchString
             * @return A promise that will resolve with the search results.
             */
            all: function (searchString) {
                var deferred = $q.defer();

                // Clear the criteria
                var criteria = [];

                // Wrap everything into a collective promise
                $q.all({
                    projects: this.project(searchString),
                    stories: this.story(searchString),
                    users: this.user(searchString)
                }).then(function (results) {
                    // Add the returned projects to the results list.
                    results.projects.forEach(function (item) {
                        criteria.push(item);
                    });
                    // Add the returned stories to the results list.
                    results.stories.forEach(function (item) {
                        criteria.push(item);
                    });
                    // Add the returned stories to the results list.
                    results.users.forEach(function (item) {
                        criteria.push(item);
                    });
                    deferred.resolve(criteria);
                });

                // Return the search promise.
                return deferred.promise;
            }
        };
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The angular resource abstraction that allows us to access stories.
 *
 * @see storyboardApiSignature
 */
angular.module('sb.services').factory('Story',
    function (ResourceFactory, $resource, storyboardApiBase) {
        'use strict';

        var resource = ResourceFactory.build(
            '/stories/:id',
            '/stories/search',
            {id: '@id'}
        );

        var tags_signature = {
            'update': {
                method: 'PUT',
                //delete request body
                transformRequest: function() {
                    return '';
                }
            },
            'delete': {
                method: 'DELETE',
                //delete request body
                transformRequest: function() {
                    return '';
                }
            }
        };

        var actorSignature = {
            'create': {
                method: 'PUT'
            },
            'get': {
                method: 'GET',
                isArray: true
            },
            'delete': {
                method: 'DELETE',
                transformRequest: function() {
                    return '';
                }
            }
        };

        resource.TagsController = $resource(
                storyboardApiBase + '/stories/:id/tags', {id: '@id'},
                tags_signature);

        resource.UsersController = $resource(
            storyboardApiBase + '/stories/:story_id/users',
            {story_id: '@story_id'},
            actorSignature);
        resource.TeamsController = $resource(
            storyboardApiBase + '/stories/:story_id/teams',
            {story_id: '@story_id'},
            actorSignature);

        ResourceFactory.applySearch(
            'Story',
            resource,
            'title',
            {
                Text: 'q',
                StoryStatus: 'status',
                Tags: 'tags',
                ProjectGroup: 'project_group_id',
                Project: 'project_id',
                User: 'assignee_id'
            }
        );

        return resource;
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The angular resource for resource subscriptions.
 *
 * @see storyboardApiSignature
 */
angular.module('sb.services').factory('Subscription',
    function (ResourceFactory) {
        'use strict';

        return ResourceFactory.build(
            '/subscriptions/:id',
            '/subscriptions/search',
            {id: '@id'}
        );
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The angular resource for resource subscription events.
 */
angular.module('sb.services').factory('SubscriptionEvent',
    function (ResourceFactory) {
        'use strict';

        return ResourceFactory.build(
            '/subscription_events/:id',
            '/subscription_events/search',
            {id: '@id'}
        );
    });

/*
 * Copyright (c) 2015 Codethink Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

angular.module('sb.services').factory('SubscriptionList',
    function (Session, Subscription) {
        'use strict';

        //GET list of subscriptions for resource specified
        return {
            subsList: function (resource, user) {
                if (!Session.isLoggedIn()) {
                    return null;
                }
                else {
                    return Subscription.browse({
                        user_id: user.id,
                        target_type: resource
                    });
                }
            }
        };
    });

/*
 * Copyright (c) 2015 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A resource to our neutral system_info endpoint.
 */
angular.module('sb.services').factory('SystemInfo',
    function ($resource, storyboardApiBase) {
        'use strict';

        return $resource(storyboardApiBase + '/systeminfo', {},
            {
                'get': {
                    method: 'GET',
                    cache: true
                }
            }
        );
    }
);

/*
 * Copyright (c) 2016 Codethink Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */


/**
 * This criteria resolver may be injected by individual resources that accept a
 * Tags search parameter.
 *
 * @see ResourceFactory
 */
angular.module('sb.services').factory('Tags',
    function (ResourceFactory) {
        'use strict';

        var resource = ResourceFactory.build(
            '/tags/:id',
            '/tags/search',
            {id: '@id'}
        );

        ResourceFactory.applySearch(
            'Tags',
            resource,
            'name',
            {},
            true // Use the name field for browse criteria, instead of the ID
        );

        return resource;
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The angular resource abstraction that allows us to create and modify tasks.
 *
 * @see storyboardApiSignature
 * @author Michael Krotscheck
 */
angular.module('sb.services').factory('Task',
    function (ResourceFactory) {
        'use strict';

        var resource = ResourceFactory.build(
            '/tasks/:id',
            '/tasks/search',
            {id: '@id'}
        );

        ResourceFactory.applySearch(
            'Task',
            resource,
            'title',
            {
                Text: 'q',
                TaskStatus: 'status',
                TaskPriority: 'priority',
                Story: 'story_id',
                User: 'assignee_id',
                Project: 'project_id',
                ProjectGroup: 'project_group_id'
            }
        );

        return resource;
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The angular resource abstraction that allows us to get task statuses.
 *
 * @see storyboardApiSignature
 */
angular.module('sb.services').factory('TaskStatus',
    function (ResourceFactory) {
        'use strict';

        var resource = ResourceFactory.build(
            '/task_statuses/:id',
            '/task_statuses/search',
            {id: '@id'},
            true // Cache the search results.
        );

        ResourceFactory.applySearch(
            'TaskStatus',
            resource,
            'name',
            {
                Text: 'q'
            }
        );

        return resource;
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 * Copyright (c) 2016 Codethink Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The angular resource abstraction that allows us to access teams and their
 * details.
 *
 * @see storyboardApiSignature
 * @see ResourceFactory
 * @author Michael Krotscheck
 * @author Adam Coldrick
 */
angular.module('sb.services').factory('Team',
    function (ResourceFactory, $resource, storyboardApiBase) {
        'use strict';

        var resource = ResourceFactory.build(
            '/teams/:team_id',
            '/teams/search', // Not implemented.
            {
                team_id: '@id'
            }
        );

        var subresourceSignature = {
            'create': {
                method: 'PUT'
            },
            'get': {
                method: 'GET',
                isArray: true
            },
            'delete': {
                method: 'DELETE',
                transformRequest: function() {
                    return '';
                }
            }
        };

        resource.UsersController = $resource(
            storyboardApiBase + '/teams/:team_id/users',
            {team_id: '@team_id'},
            subresourceSignature);

        resource.ProjectsController = $resource(
            storyboardApiBase + '/teams/:team_id/projects',
            {team_id: '@team_id'},
            subresourceSignature);

        return resource;
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The angular resource abstraction that allows us to access discussions that
 * are surrounding stories.
 *
 * @see storyboardApiSignature
 */
angular.module('sb.services').factory('TimelineEvent',
    function (ResourceFactory) {
        'use strict';

        return ResourceFactory.build(
            '/stories/:story_id/events/:id',
            '/stories/:story_id/events/search', // Not implemented.
            {
                id: '@id',
                story_id: '@story_id'
            }
        );
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The angular resource abstraction that allows us to search, access, and
 * modify users.
 *
 * @see ResourceFactory
 * @author Michael Krotscheck
 */
angular.module('sb.services').factory('User',
    function (ResourceFactory, $resource, DSCacheFactory, storyboardApiBase) {
        'use strict';

        var signature = {
            'create': {
                method: 'POST'
            },
            'get': {
                method: 'GET'
            },
            'update': {
                method: 'PUT',
                interceptor: {
                    response: function(response) {
                        var user = response.resource;
                        DSCacheFactory.get('defaultCache').put(
                            storyboardApiBase + '/users/' + user.id,
                            user);
                    }
                }
            },
            'delete': {
                method: 'DELETE'
            },
            'browse': {
                method: 'GET',
                isArray: true,
                responseType: 'json',
                cache: false
            },
            'search': {
                method: 'GET',
                url: storyboardApiBase + '/users/search',
                isArray: true,
                responseType: 'json',
                cache: false
            }
        };
        var resource = $resource(
            storyboardApiBase + '/users/:id',
            {id: '@id'},
            signature
        );

        ResourceFactory.applySearch(
            'User',
            resource,
            'full_name',
            {Text: 'q'}
        );

        return resource;
    });

/*
 * Copyright (c) 2015 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The angular resource abstraction that allows us to consume preferences. This
 * resource does not adhere to our ResourceFactory pattern, as preferences are
 * treated as a single per-user object.
 */
angular.module('sb.services').factory('UserPreference',
    function ($resource, storyboardApiBase) {
        'use strict';

        return $resource(storyboardApiBase + '/users/:id/preferences',
            {
                id: '@id'
            },
            {
                'get': {
                    method: 'GET',
                    cache: true
                }
            }
        );
    }
);

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The angular resource abstraction that allows us to search, access, and
 * modify individual user's authentication tokens.
 *
 * @see ResourceFactory
 * @author Michael Krotscheck
 */
angular.module('sb.services').factory('UserToken',
    function (ResourceFactory) {
        'use strict';

        var resource = ResourceFactory.build(
            '/users/:user_id/tokens/:id',
            null,
            {user_id: '@user_id', id: '@id'}
        );

        return resource;
    });

/*
 * Copyright (c) 2015 Codethink Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The angular resource abstraction that allows us to access projects groups.
 *
 * @see ResourceFactory
 * @author Adam Coldrick
 */
angular.module('sb.services').factory('Worklist',
    function (ResourceFactory, $resource, storyboardApiBase) {
        'use strict';

        var resource = ResourceFactory.build(
            '/worklists/:id',
            '/worklists/search',
            {id: '@id'},
            false, true //turn off 'cache search results', and turn on 'disable
//                        cached GETs', respectively
        );

        var items_signature = {
            'create': {
                method: 'POST'
            },
            'get': {
                method: 'GET',
                cache: false,
                isArray: true
            },
            'update': {
                method: 'PUT'
            },
            'delete': {
                method: 'DELETE',
                transformRequest: function() {
                    return '';
                }
            }
        };

        resource.ItemsController = $resource(
            storyboardApiBase + '/worklists/:id/items/:item_id',
            {id: '@id', item_id: '@item_id'},
            items_signature);

        var permissionsSignature = {
            'create': {
                method: 'POST'
            },
            'get': {
                method: 'GET',
                cache: false,
                isArray: true
            },
            'update': {
                method: 'PUT'
            }
        };

        resource.Permissions = $resource(
            storyboardApiBase + '/worklists/:id/permissions',
            {id: '@id'},
            permissionsSignature
        );

        var filtersSignature = {
            'create': {
                method: 'POST'
            },
            'get': {
                method: 'GET',
                cache: false,
                isArray: true
            },
            'update': {
                method: 'PUT'
            },
            'delete': {
                method: 'DELETE',
                transformRequest: function() {
                    return '';
                }
            }
        };

        resource.Filters = $resource(
            storyboardApiBase + '/worklists/:id/filters/:filter_id',
            {id: '@id'},
            filtersSignature
        );

        ResourceFactory.applySearch(
            'Worklist',
            resource,
            'title',
            {
                Text: 'title',
                Story: 'story_id',
                Task: 'task_id',
                User: 'creator_id'
            }
        );

        return resource;
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the 'License'); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Factory methods that simply construction of storyboard API resources.
 *
 * @author Michael Krotscheck
 */
angular.module('sb.services')
    .service('ResourceFactory',
    function ($q, $log, $injector, Criteria, $resource, storyboardApiBase,
              Preference) {
        'use strict';

        /**
         * Construct a full API signature for a specific resource. Includes
         * CRUD, Browse, and Search. If the resource doesn't support it,
         * don't use it :).
         *
         * @param searchUrl
         * @param cacheSearchResults
         * @returns An API signature that may be used with a $resource.
         */
        function buildSignature(searchUrl, cacheSearchResults,
                                disableCachedGets) {
            // Cast to boolean.
            cacheSearchResults = !!cacheSearchResults;
            disableCachedGets = !!disableCachedGets;

            return {
                'create': {
                    method: 'POST'
                },
                'get': {
                    method: 'GET',
                    cache: !disableCachedGets
                },
                'update': {
                    method: 'PUT'
                },
                'delete': {
                    method: 'DELETE'
                },
                'browse': {
                    method: 'GET',
                    isArray: true,
                    responseType: 'json',
                    cache: cacheSearchResults
                },
                'search': {
                    method: 'GET',
                    url: searchUrl,
                    isArray: true,
                    responseType: 'json',
                    cache: cacheSearchResults
                }
            };
        }


        return {

            /**
             * Build a resource URI.
             *
             * @param restUri
             * @param searchUri
             * @param resourceParameters
             * @param cacheSearchResults
             * @returns {*}
             */
            build: function (restUri, searchUri, resourceParameters,
                             cacheSearchResults, disableCachedGets) {

                if (!restUri) {
                    $log.error('Cannot use resource factory ' +
                        'without a base REST uri.');
                    return null;
                }

                // Cast results
                cacheSearchResults = !!cacheSearchResults;

                var signature = buildSignature(storyboardApiBase + searchUri,
                    cacheSearchResults, disableCachedGets);
                return $resource(storyboardApiBase + restUri,
                    resourceParameters, signature);
            },

            /**
             * This method takes an already configured resource, and applies
             * the static methods necessary to support the criteria search API.
             * Browse parameters should be formatted as an object containing
             * 'injector name': 'param'. For example, {'Project': 'project_id'}.
             *
             * @param resourceName The explicit resource name of this resource
             * within the injection scope.
             * @param resource The configured resource.
             * @param nameField The name field to use while browsing criteria.
             * @param searchParameters The search parameters to apply.
             */
            applySearch: function (resourceName, resource, nameField,
                                   searchParameters, useNameField) {

                // List of criteria resolvers which we're building.
                var criteriaResolvers = [];

                for (var type in searchParameters) {

                    // If the requested type exists and has a criteriaResolver
                    // method, add it to the list of resolvable browse criteria.
                    var typeResource = $injector.get(type);
                    if (!!typeResource &&
                        typeResource.hasOwnProperty('criteriaResolver')) {
                        criteriaResolvers.push(typeResource.criteriaResolver);
                    }
                }

                /**
                 * Return a list of promise-returning methods that, given a
                 * browse string, will provide a list of search criteria.
                 *
                 * @returns {*[]}
                 */
                resource.criteriaResolvers = function () {
                    return criteriaResolvers;
                };


                // If we found a browse parameter, add the ability to use
                // this resource as a source of criteria.
                if (!!nameField) {
                    /**
                     * Add the criteria resolver method.
                     */
                    resource.criteriaResolver =
                        function (searchString, pageSize) {
                            pageSize = pageSize || Preference.get('page_size');

                            var deferred = $q.defer();

                            // build the query parameters.
                            var queryParams = {};
                            queryParams[nameField] = searchString;
                            queryParams.limit = pageSize;

                            resource.browse(queryParams,
                                function (result) {
                                    // Transform the results to criteria tags.
                                    var criteriaResults = [];
                                    result.forEach(function (item) {
                                        if (useNameField) {
                                            criteriaResults.push(
                                                Criteria.create(resourceName,
                                                    item[nameField],
                                                    item[nameField])
                                            );
                                        } else if (item.hasOwnProperty('id')) {
                                            criteriaResults.push(
                                                Criteria.create(resourceName,
                                                    item.id,
                                                    item[nameField])
                                            );
                                        } else {
                                            criteriaResults.push(
                                                Criteria.create(resourceName,
                                                    item.key,
                                                    item[nameField])
                                            );
                                        }
                                    });
                                    deferred.resolve(criteriaResults);
                                }, function () {
                                    deferred.resolve([]);
                                }
                            );

                            return deferred.promise;
                        };
                }


                /**
                 * The criteria filter.
                 */
                resource.criteriaFilter = Criteria
                    .buildCriteriaFilter(searchParameters);

                /**
                 * The criteria map.
                 */
                resource.criteriaMap = Criteria
                    .buildCriteriaMap(searchParameters);

            }
        };
    });

/*
 * Copyright (c) 2016 Codethink Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

angular.module('sb.story').controller('CommentHistoryController',
    function ($scope, $modalInstance, history, comment) {
        'use strict';

        $scope.history = history;
        $scope.comment = comment;
    })
;

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the 'License'); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Story detail &  manipulation controller.
 */
angular.module('sb.story').controller('StoryDeleteController',
    function ($log, $scope, $state, story, $modalInstance) {
        'use strict';

        $scope.story = story;

        // Set our progress flags and clear previous error conditions.
        $scope.isUpdating = true;
        $scope.error = {};

        $scope.remove = function () {
            $scope.story.$delete(
                function () {
                    $modalInstance.dismiss('success');
                    $state.go('sb.project.list');
                }
            );
        };

        $scope.close = function () {
            $modalInstance.dismiss('cancel');
        };
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 * Copyright (c) 2016 Codethink Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the 'License'); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Story detail &  manipulation controller.
 */
angular.module('sb.story').controller('StoryDetailController',
    function ($log, $rootScope, $scope, $state, $stateParams, $modal, Session,
              Preference, TimelineEvent, Comment, TimelineEventTypes, story,
              Story, Project, Branch, creator, tasks, Task, DSCacheFactory,
              User, $q, storyboardApiBase, SessionModalService, moment,
              $document, $anchorScroll, $timeout, $location, currentUser,
              enableEditableComments, Tags, worklists, Team, StoryHelper) {
        'use strict';

        var pageSize = Preference.get('story_detail_page_size');
        var firstLoad = true;

        $scope.enableEditableComments = enableEditableComments;

        // Set the yOffset to 50 because the fixed bootstrap navbar
        // is 50px high.
        $anchorScroll.yOffset = 50;
        $scope.filterMode = 'advanced';

        /**
         * The story, resolved in the state.
         *
         * @type {Story}
         */
        $scope.story = story;

        /**
         * The user record for the author, resolved in the state.
         *
         * @type {User}
         */
        $scope.creator = creator;

        /**
         * All tasks associated with this story, resolved in the state.
         *
         * @type {[Task]}
         */
        $scope.projectNames = [];
        $scope.projects = {};
        $scope.tasks = tasks;

        /**
         * Allowing sorting tasks in search results by certain fields.
         */
        $scope.tasks_sort_field = {
            label: 'Id',
            value: 'id'
        };
        $scope.changeOrder = function(orderBy){
            $scope.tasks_sort_field.label = orderBy.toString();
            switch(orderBy) {
                case 'Id':
                    $scope.tasks_sort_field.value = 'id';
                    break;
                case 'Status':
                    $scope.tasks_sort_field.value = 'status';
                    break;
                case 'Title':
                    $scope.tasks_sort_field.value = 'title';
                    break;
            }
        };


        function mapTaskToProject(task) {
            Project.get({id: task.project_id}).$promise.then(function(project) {
                var idx = $scope.projectNames.indexOf(project.name);
                if (idx < 0) {
                    $scope.projectNames.push(project.name);
                    $scope.projects[project.name] = project;
                    $scope.projects[project.name].branchNames = [];
                    $scope.projects[project.name].branches = {};
                }
                Branch.get({id: task.branch_id}).$promise.then(
                    function(branch) {
                        var branchIdx = $scope.projects[project.name]
                            .branchNames.indexOf(branch.name);
                        if (branchIdx > -1) {
                            $scope.projects[project.name].branches[branch.name]
                                .tasks.push(task);
                        } else {
                            $scope.projects[project.name]
                                .branches[branch.name] = branch;
                            $scope.projects[project.name]
                                .branches[branch.name].tasks = [task];
                            $scope.projects[project.name]
                                .branches[branch.name].newTask = new Task({
                                    story_id: $scope.story.id,
                                    branch_id: branch.id,
                                    project_id: project.id,
                                    status: 'todo'
                                });
                            $scope.projects[project.name]
                                .branchNames.push(branch.name);
                        }
                    });
            });
        }

        angular.forEach(tasks, mapTaskToProject);

        /**
         * All worklists containing this story or tasks within it, with
         * information about which task is relevant added.
         *
         * @type {[Worklist]}
         */
        function setWorklists() {
            function isNotArchived(card) {
                return !card.archived;
            }

            var taskIds = $scope.tasks.map(function(task) {
                return task.id;
            });
            for (var i = 0; i < worklists.length; i++) {
                var worklist = worklists[i];
                worklist.relatedItems = [];
                worklist.items = worklist.items.filter(isNotArchived);
                for (var j = 0; j < worklist.items.length; j++) {
                    var item = worklist.items[j];
                    if (item.item_type === 'story') {
                        if (item.item_id === story.id) {
                            worklist.relatedItems.push(item);
                        }
                    } else if (item.item_type === 'task') {
                        if (taskIds.indexOf(item.item_id) > -1) {
                            worklist.relatedItems.push(item);
                        }
                    }
                }
            }
            $scope.worklists = worklists.map(function(list) {
                if (list.relatedItems.length > 0) {
                    return list;
                }
            }).filter(function(list) { return list; });
        }

        setWorklists();

        $scope.showWorklistsModal = function() {
            var modalInstance = $modal.open({
                templateUrl: 'app/stories/template/worklists.html',
                backdrop: 'static',
                controller: 'StoryWorklistsController',
                resolve: {
                    worklists: function () {
                        return $scope.worklists;
                    }
                }
            });

            // Return the modal's promise.
            return modalInstance.result;
        };

        // Load the preference for each display event.
        function reloadPagePreferences() {
            TimelineEventTypes.forEach(function (type) {
                // Prefs are stored as strings, UI tests on booleans, so we
                // convert here.
                var pref_name = 'display_events_' + type;
                $scope[pref_name] = Preference.get(pref_name) === 'true';
            });
            pageSize = Preference.get('story_detail_page_size');
            $scope.loadEvents();
        }

        $scope.filterComments = function() {
            $scope.filterMode = 'comments';
            angular.forEach(TimelineEventTypes, function(type) {
                var pref_name = 'display_events_' + type;
                $scope[pref_name] = false;
            });
            $scope.display_events_user_comment = true;
            $scope.display_events_story_created = true;
            $scope.loadEvents();
        };

        $scope.filterAll = function() {
            $scope.filterMode = 'all';
            angular.forEach(TimelineEventTypes, function(type) {
                var pref_name = 'display_events_' + type;
                $scope[pref_name] = true;
            });
            $scope.loadEvents();
        };

        $scope.isSearching = false;

        /**
         * Load TimelineEvents related to the story.
         */
        $scope.loadEvents = function () {
            $scope.isSearching = true;
            var params = {};
            params.sort_field = 'id';
            params.sort_dir = 'asc';
            params.story_id = $scope.story.id;
            params.event_type = TimelineEventTypes.map(function(type) {
                var pref_name = 'display_events_' + type;
                if ($scope[pref_name]) {
                    return type;
                }
            }).filter(function(item) { return item; });

            TimelineEvent.browse(params,
                function (result) {
                    var eventResults = [];
                    result.forEach(function (item) {
                        item.author = User.get({id: item.author_id});
                        item.event_info = JSON.parse(item.event_info);

                        eventResults.push(item);
                    });
                    $scope.events = eventResults;
                    $scope.isSearching = false;
                    if (firstLoad) {
                        firstLoad = false;
                        // Wrap this in a timeout to make sure we don't
                        // try to scroll before the timeline is rendered.
                        $timeout(function() {
                            $anchorScroll();
                        }, 0);
                    }
                },
                function () {
                    $scope.isSearching = false;
                }
            );
        };
        reloadPagePreferences();

        /**
         * Next page of the results.
         */
        $scope.nextPage = function () {
            $scope.searchOffset += pageSize;
            $scope.loadEvents();
        };

        /**
         * Previous page of the results.
         */
        $scope.previousPage = function () {
            $scope.searchOffset -= pageSize;
            if ($scope.searchOffset < 0) {
                $scope.searchOffset = 0;
            }
            $scope.loadEvents();
        };

        /**
         * Update the page size preference and re-search.
         */
        $scope.updatePageSize = function (value) {
            Preference.set('story_detail_page_size', value).then(
                function () {
                    pageSize = value;
                    $scope.loadEvents();
                }
            );
        };

        /**
         * The new comment backing the input form.
         */
        $scope.newComment = new Comment({});

        /**
         * Generic service error handler. Assigns errors to the view's scope,
         * and unsets our flags.
         */
        function handleServiceError(error) {
            // We've encountered an error.
            $scope.error = error;
            $scope.isLoading = false;
            $scope.isUpdating = false;
        }

        /**
         * Resets our loading flags.
         */
        function handleServiceSuccess() {
            $scope.isLoading = false;
            $scope.isUpdating = false;
        }

        /**
         * Toggle/display the edit form
         */
        $scope.toggleEditMode = function () {
            if (Session.isLoggedIn()) {
                $scope.showEditForm = !$scope.showEditForm;

                // Deferred timeout request for a re-rendering of elastic
                // text fields, since the size calculation breaks when
                // visible: false
                setTimeout(function () {
                    $rootScope.$broadcast('elastic:adjust');
                }, 1);
            } else {
                $scope.showEditForm = false;
            }
        };

        /**
         * UI Flag for when we're in edit mode.
         */
        $scope.showEditForm = false;

        /**
         * UI flag for when we're initially loading the view.
         *
         * @type {boolean}
         */
        $scope.isLoading = true;

        /**
         * UI view for when a change is round-tripping to the server.
         *
         * @type {boolean}
         */
        $scope.isUpdating = false;

        /**
         * UI view for when we're trying to save a comment.
         *
         * @type {boolean}
         */
        $scope.isSavingComment = false;

        /**
         * Any error objects returned from the services.
         *
         * @type {{}}
         */
        $scope.error = {};

        /**
         * Scope method, invoke this when you want to update the story.
         */
        $scope.update = function () {
            // Set our progress flags and clear previous error conditions.
            $scope.isUpdating = true;
            $scope.error = {};

            // Invoke the save method and wait for results.
            $scope.story.$update(
                function () {
                    $scope.showEditForm = false;
                    $scope.previewStory = false;
                    handleServiceSuccess();
                },
                handleServiceError
            );
        };

        /**
         * Resets any changes and toggles the form back.
         */
        $scope.cancel = function () {
            $scope.showEditForm = false;
        };

        $scope.privacyLocked = false;

        /**
         * Handle any change to whether or not the story is security-related
         */
        $scope.updateSecurity = function(forcePrivate, update) {
            $scope.privacyLocked = StoryHelper.updateSecurity(
                forcePrivate, update, $scope.story, $scope.tasks);
        };

        /**
         * Delete method.
         */
        $scope.remove = function () {
            var modalInstance = $modal.open({
                templateUrl: 'app/stories/template/delete.html',
                backdrop: 'static',
                controller: 'StoryDeleteController',
                resolve: {
                    story: function () {
                        return $scope.story;
                    }
                }
            });

            // Return the modal's promise.
            return modalInstance.result;
        };

        $scope.updateFilter = function () {
            var modalInstance = $modal.open({
                templateUrl: 'app/stories/template/update_filter.html',
                backdrop: 'static',
                controller: 'TimelineFilterController'
            });

            $scope.filterMode = 'advanced';
            modalInstance.result.then(reloadPagePreferences);
            $scope.searchLimit = Preference.get('story_detail_page_size');
        };

        $scope.previewingComment = false;
        $scope.togglePreview = function(val) {
            $scope.previewingComment = !$scope.previewingComment;
            if (val === true || val === false) {
                $scope.previewingComment = val;
            }
        };

        /**
         * Quote a comment
         */
        $scope.quote = function(event) {
            var timestamp = moment(event.created_at);
            var reference = '<footer>'
                            + event.author.full_name
                            + ' on '
                            + timestamp.format('YYYY-MM-DD [at] HH:mm:ss')
                            + '</footer>';
            var lines = event.comment.content.split('\n');
            for (var i = 0; i < lines.length; i++) {
                lines[i] = '> ' + lines[i];
            }
            lines.push('> ' + reference);
            var quoted = lines.join('\n');
            if ($scope.newComment.content) {
                $scope.newComment.content += '\n\n' + quoted;
            } else {
                $scope.newComment.content = quoted;
            }
            $document[0].getElementById('comment').focus();
        };

        /**
         * Determine if a comment is currently being permalinked
         */
        $scope.isLinked = function(event) {
            if ($location.hash() === 'comment-' + event.comment.id) {
                return true;
            }
            return false;
        };

        /**
         * Determine if the current user is the author of a comment.
         */
        $scope.isAuthor = function(event) {
            if (currentUser.id === event.author_id) {
                return true;
            }
            return false;
        };

        /**
         * Add a comment
         */
        $scope.addComment = function () {

            function resetSavingFlag() {
                $scope.isSavingComment = false;
            }

            // Do nothing if the comment is empty
            if (!$scope.newComment.content) {
                $log.warn('No content in comment, discarding submission');
                return;
            }

            $scope.isSavingComment = true;
            $scope.togglePreview(false);

            // Author ID will be automatically attached by the service, so
            // don't inject it into the conversation until it comes back.
            $scope.newComment.$create({story_id: $scope.story.id},
                function () {
                    $scope.newComment = new Comment({});
                    resetSavingFlag();
                    $scope.loadEvents();
                }
            );
        };

        $scope.edit = function(event) {
            event.editing = true;
            event.comment.edited = angular.copy(event.comment.content);
        };

        $scope.editComment = function(event) {
            event.isUpdating = true;
            Comment.update(
                {
                    id: story.id,
                    comment_id: event.comment.id,
                    content: event.comment.edited
                },
                function(result) {
                    event.comment.content = result.content;
                    event.isUpdating = false;
                    event.editing = false;
                },
                function() {
                    event.isUpdating = false;
                    event.editing = false;
                }
            );
        };

        /**
         * View comment history
         */
        $scope.showHistory = function (event) {
            var modalInstance = $modal.open({
                size: 'lg',
                templateUrl: 'app/stories/template/comments/history.html',
                backdrop: 'static',
                controller: 'CommentHistoryController',
                resolve: {
                    history: function() {
                        if (event.comment && event.comment.updated_at) {
                            return Comment.History.get({
                                story_id: story.id,
                                id: event.comment.id
                            });
                        }
                        return [];
                    },
                    comment: function() {
                        return event.comment;
                    }
                }
            });

            // Return the modal's promise.
            return modalInstance.result;
        };

        /**
         * Show modal informing the user login is required.
         */
        $scope.showLoginRequiredModal = function() {
            SessionModalService.showLoginRequiredModal();
        };

        /**
         * User/team typeahead search method.
         */
        $scope.searchActors = function (value, users, teams) {
            var userIds = users.map(function(user){return user.id;});
            var teamIds = teams.map(function(team){return team.id;});
            var deferred = $q.defer();
            var usersDeferred = $q.defer();
            var teamsDeferred = $q.defer();

            User.browse({full_name: value, limit: 10},
                function(searchResults) {
                    var results = [];
                    angular.forEach(searchResults, function(result) {
                        if (userIds.indexOf(result.id) === -1) {
                            result.name = result.full_name;
                            result.type = 'user';
                            results.push(result);
                        }
                    });
                    usersDeferred.resolve(results);
                }
            );
            Team.browse({name: value, limit: 10},
                function(searchResults) {
                    var results = [];
                    angular.forEach(searchResults, function(result) {
                        if (teamIds.indexOf(result.id) === -1) {
                            result.type = 'team';
                            results.push(result);
                        }
                    });
                    teamsDeferred.resolve(results);
                }
            );

            var searches = [teamsDeferred.promise, usersDeferred.promise];
            $q.all(searches).then(function(searchResults) {
                var results = [];
                angular.forEach(searchResults, function(promise) {
                    angular.forEach(promise, function(result) {
                        results.push(result);
                    });
                });
                deferred.resolve(results);
            });

            return deferred.promise;
        };

        /**
         * Add a new user or team to one of the permission levels.
         */
        $scope.addActor = function (model) {
            if (model.type === 'user') {
                $scope.story.users.push(model);
            } else if (model.type === 'team') {
                $scope.story.teams.push(model);
            }
        };

        /**
         * Remove a user from the permissions.
         */
        $scope.removeUser = function (model) {
            var idx = $scope.story.users.indexOf(model);
            $scope.story.users.splice(idx, 1);
        };

        /**
         * Remove a team from the permissions.
         */
        $scope.removeTeam = function(model) {
            var idx = $scope.story.teams.indexOf(model);
            $scope.story.teams.splice(idx, 1);
        };

        // ###################################################################
        // Task Management
        // ###################################################################

        /**
         * The new task for the task form.
         */
        $scope.newTask = new Task({
            story_id: $scope.story.id,
            status: 'todo'
        });

        /**
         * Adds a task.
         */
        $scope.createTask = function (task, branch) {
            // Make a copy to save, so that the next task retains the
            // old information (for easier continuous editing).
            var savingTask = new Task(angular.copy(task));
            savingTask.$save(function (savedTask) {
                $scope.tasks.push(savedTask);
                if (branch) {
                    branch.tasks.push(savedTask);
                } else {
                    mapTaskToProject(savedTask);
                    $scope.updateSecurity(false, true);
                }
                $scope.loadEvents();
                task.title = '';
            });
        };

        /**
         * Cleans up the project/branch/task tree.
         *
         * If there are no tasks remaining in the given branch in the given
         * project, then remove that branch from the project's list of
         * branches to display tasks for.
         *
         * If there are then no branches left in the project, remove the
         * project from the list of projects to display tasks for.
         */
        function cleanBranchAndProject(projectName, branchName) {
            var project = $scope.projects[projectName];
            var branch = project.branches[branchName];
            var nameIdx = -1;

            if (branch.tasks.length === 0) {
                nameIdx = project.branchNames.indexOf(branchName);
                if (nameIdx > -1) {
                    project.branchNames.splice(nameIdx, 1);
                }
                delete project.branches[branchName];
            }
            if (project.branchNames.length === 0) {
                nameIdx = $scope.projectNames.indexOf(projectName);
                if (nameIdx > -1) {
                    $scope.projectNames.splice(nameIdx, 1);
                }
                delete $scope.projects[projectName];
            }
        }

        /**
         * Updates the task list.
         */
        $scope.updateTask = function (task, fieldName, value, projectName,
                                      branchName) {
            var params = {id: task.id};
            params[fieldName] = value;
            task[fieldName] = value;
            if(!!task.id) {
                Task.update(params, function() {
                    $scope.showTaskEditForm = false;
                    $scope.loadEvents();
                }).$promise.then(function(updated) {
                    if (fieldName === 'project_id') {
                        var project = $scope.projects[projectName];
                        var branch = project.branches[branchName];

                        var branchTaskIndex = branch.tasks.indexOf(task);
                        if (branchTaskIndex > -1) {
                            branch.tasks.splice(branchTaskIndex, 1);
                        }

                        cleanBranchAndProject(projectName, branchName);
                        mapTaskToProject(updated);
                        $scope.updateSecurity(false, true);
                    }
                });
            }
        };

        $scope.editNotes = function(task) {
            task.tempNotes = task.link;
            task.editing = true;
        };

        $scope.cancelEditNotes = function(task) {
            task.tempNotes = '';
            task.editing = false;
        };

        $scope.showAddWorklist = function(task) {
            $modal.open({
                templateUrl: 'app/stories/template/add_task_to_worklist.html',
                backdrop: 'static',
                controller: 'StoryTaskAddWorklistController',
                resolve: {
                    task: function() {
                        return task;
                    }
                }
            });
        };

        /**
         * Removes this task
         */
        $scope.removeTask = function (task, projectName, branchName) {
            var modalInstance = $modal.open({
                templateUrl: 'app/stories/template/delete_task.html',
                backdrop: 'static',
                controller: 'StoryTaskDeleteController',
                resolve: {
                    task: function () {
                        return task;
                    },
                    params: function () {
                        return {
                            lastTask: ($scope.tasks.length === 1)
                        };
                    }
                }
            });

            modalInstance.result.then(
                function () {
                    var taskIndex = $scope.tasks.indexOf(task);
                    if (taskIndex > -1) {
                        $scope.tasks.splice(taskIndex, 1);
                    }

                    var project = $scope.projects[projectName];
                    var branch = project.branches[branchName];
                    var branchTaskIndex = branch.tasks.indexOf(task);
                    if (branchTaskIndex > -1) {
                        branch.tasks.splice(branchTaskIndex, 1);
                    }
                    cleanBranchAndProject(projectName, branchName);
                    $scope.loadEvents();
                }
            );
        };

        // ###################################################################
        // Tags Management
        // ###################################################################

        /**
         * The controller to add/delete tags from a story.
         *
         * @type {TagsController}
         */
        $scope.TagsController = new Story.TagsController({'id': story.id});

        /**
         * Show an input for a new tag
         *
         * @type {boolean}
         */
        $scope.showAddTag = false;

        $scope.toggleAddTag = function() {
            $scope.newTag.name = null;
            $scope.showAddTag = !$scope.showAddTag;
        };

        $scope.addTag = function (tag_name) {
            if(!!tag_name) {
                $scope.TagsController.$update({tags: [tag_name]},
                    function (updatedStory) {
                        DSCacheFactory.get('defaultCache').put(
                            storyboardApiBase + '/stories/' + story.id,
                            updatedStory);
                        $scope.story.tags.push(tag_name);
                        $scope.loadEvents();
                    },
                    handleServiceError);
                $scope.newTag.name = null;
            }
        };

        $scope.removeTag = function (tag_name) {
            $scope.TagsController.$delete({tags: [tag_name]},
                function() {
                    var tagIndex = $scope.story.tags.indexOf(tag_name);
                    DSCacheFactory.get('defaultCache').remove(
                            storyboardApiBase + '/stories/' + story.id);
                    if (tagIndex > -1) {
                        $scope.story.tags.splice(tagIndex, 1);
                    }
                    $scope.loadEvents();
                },
                handleServiceError);
        };

        $scope.searchTags = function (value) {
            return Tags.browse({name: value, limit: 10}).$promise;
        };

        $scope.updateViewValue = function (value) {
            $scope.newTag.name = value;
        };

        $scope.newTag = {};
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 * Copyright (c) 2016 Codethink Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Controller for our story list.
 */
angular.module('sb.story').controller('StoryListController',
    function ($scope, $state, Criteria, NewStoryService, SubscriptionList,
              CurrentUser, $stateParams, SearchHelper) {
        'use strict';

        // search results must be of type "story"
        $scope.resourceTypes = ['Story'];

        // Search result criteria default must be "active"
        $scope.defaultCriteria = SearchHelper.parseParameters($stateParams);

        /**
         * Creates a new story.
         */
        $scope.newStory = function () {
            NewStoryService.showNewStoryModal()
                .then(function (story) {
                    // On success, go to the story detail.
                    $state.go('sb.story.detail', {storyId: story.id});
                }
            );
        };

        //GET list of subscriptions
        var cuPromise = CurrentUser.resolve();

        cuPromise.then(function(user){
            $scope.storySubscriptions = SubscriptionList.subsList(
                'story', user);
        });
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Controller for items in the story list.
 */
angular.module('sb.story').controller('StoryListItemController',
    function ($scope, TaskStatus) {
        'use strict';

        /**
         * Gets the status for the task count texts
         */
        function getStatusClass(status) {
            var className = '';
            switch(status) {
                case 'inprogress':
                    className = 'text-info';
                    break;
                case 'review':
                    className = 'text-warning';
                    break;
                case 'merged':
                    className = 'text-success';
                    break;
                case 'invalid':
                    className = 'muted';
                    break;
                default:
                    className = '';
                    break;
            }
            return className;
        }

        /**
         * UI toggle, show row details?
         */
        $scope.expandRow = false;

        $scope.status_texts = [];
        $scope.status_classes = [];
        TaskStatus.query({}, function (items) {
            for (var i = 0;i < items.length; i++) {
                $scope.status_texts[items[i].key] = items[i].name;
                $scope.status_classes[items[i].key] = getStatusClass(
                    items[i].key);
            }
        });

        /**
         * Figure out what color we're labeling ourselves as...
         */
        switch ($scope.story.status) {
            case 'active':
                $scope.statusLabelStyle = 'label-info';
                break;
            case 'merged':
                $scope.statusLabelStyle = 'label-success';
                break;
            case 'invalid':
                $scope.statusLabelStyle = 'label-default';
                break;
            default:
                $scope.statusLabelStyle = 'label-default';
                break;
        }
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Controller for the "new story" modal popup.
 */
angular.module('sb.story').controller('StoryModalController',
    function ($scope, $modalInstance, params, Project, Story, Task, User,
              Team, $q, CurrentUser, StoryHelper) {
        'use strict';

        var currentUser = CurrentUser.resolve();

        $scope.projects = Project.browse({});

        currentUser.then(function(user) {
            $scope.story = new Story({
                title: '',
                users: [user],
                teams: []
            });
        });

        $scope.tasks = [new Task({
            title: ''
        })];


        /**
         * Handle any change to whether or not the story is security-related
         */
        $scope.updateSecurity = function(forcePrivate, update) {
            $scope.privacyLocked = StoryHelper.updateSecurity(
                forcePrivate, update, $scope.story, $scope.tasks);
        };

        // Preload the project
        if (params.projectId) {
            Project.get({
                id: params.projectId
            }, function (project) {
                $scope.asyncProject = project;
                $scope.tasks = [new Task({
                    title: '',
                    project_id: project.id
                })];
            });
        }

        var lastTitle = '', trackingStoryTitle = true;
        $scope.$on('$destroy', $scope.$watch(
            function () {
                return $scope.story.title;
            },
            function (newTitle) {

                // Exit if we've set a custom title.
                if (!trackingStoryTitle) {
                    return;
                }

                // Get the first task in the list.
                var task = $scope.tasks[0];
                if (trackingStoryTitle && task.title === lastTitle) {
                    task.title = newTitle;
                    lastTitle = newTitle;
                } else {
                    trackingStoryTitle = false;
                }
            }
        ));

        /**
         * Saves the story, then saves all the tasks associated with that
         * story.
         */
        $scope.save = function () {
            $scope.story.$create(function (story) {

                var resolvingTasks = $scope.tasks.length;

                // Now that we've created the task, save all the tasks.
                $scope.tasks.forEach(function (task) {
                    task.story_id = story.id;
                    task.$create(function () {
                        resolvingTasks--;
                        if (resolvingTasks === 0) {
                            $modalInstance.close(story);
                        }
                    });
                });
            });
        };

        /**
         * Close this modal without saving.
         */
        $scope.close = function () {
            $modalInstance.dismiss('cancel');
        };

        /**
         * Add another task.
         */
        $scope.addTask = function () {
            // When adding a new modal, grab the project ID from the last
            // item in the list.

            var lastTask = $scope.tasks[$scope.tasks.length - 1];
            var project_id = lastTask.project_id;

            if (!project_id) {
                // if we are in the scope of a project, grab the project ID
                // from that scope.
                project_id = params.projectId || null;
            }

            var current_task = new Task({project_id: project_id});
            if (project_id) {
                // Preload the project
                Project.get({
                    id: project_id
                }, function (project) {
                    $scope.asyncProject = project;
                });
            }
            $scope.tasks.push(current_task);
            $scope.updateSecurity(true, false);
        };

        /**
         * Remove a task from the task list, but only if we have
         * more than one task.
         */
        $scope.removeTask = function (task) {
            if ($scope.tasks.length < 2) {
                return;
            }
            var idx = $scope.tasks.indexOf(task);
            $scope.tasks.splice(idx, 1);
        };

        /**
         * Project typeahead search method.
         */
        $scope.searchProjects = function (value) {
            return Project.browse({name: value, limit: 10}).$promise;
        };

        /**
         * Formats the project name.
         */
        $scope.formatProjectName = function (model) {
            if (!!model) {
                return model.name;
            }
            return '';
        };

        /**
         * Select a new project.
         */
        $scope.selectNewProject = function (model, task) {
            task.project_id = model.id;
            $scope.updateSecurity(true, false);
        };

        /**
         * User/team typeahead search method.
         */
        $scope.searchActors = function (value, users, teams) {
            var userIds = users.map(function(user){return user.id;});
            var teamIds = teams.map(function(team){return team.id;});
            var deferred = $q.defer();
            var usersDeferred = $q.defer();
            var teamsDeferred = $q.defer();

            User.browse({full_name: value, limit: 10},
                function(searchResults) {
                    var results = [];
                    angular.forEach(searchResults, function(result) {
                        if (userIds.indexOf(result.id) === -1) {
                            result.name = result.full_name;
                            result.type = 'user';
                            results.push(result);
                        }
                    });
                    usersDeferred.resolve(results);
                }
            );
            Team.browse({name: value, limit: 10},
                function(searchResults) {
                    var results = [];
                    angular.forEach(searchResults, function(result) {
                        if (teamIds.indexOf(result.id) === -1) {
                            result.type = 'team';
                            results.push(result);
                        }
                    });
                    teamsDeferred.resolve(results);
                }
            );

            var searches = [teamsDeferred.promise, usersDeferred.promise];
            $q.all(searches).then(function(searchResults) {
                var results = [];
                angular.forEach(searchResults, function(promise) {
                    angular.forEach(promise, function(result) {
                        results.push(result);
                    });
                });
                deferred.resolve(results);
            });

            return deferred.promise;
        };

        /**
         * Add a new user or team to one of the permission levels.
         */
        $scope.addActor = function (model) {
            if (model.type === 'user') {
                $scope.story.users.push(model);
            } else if (model.type === 'team') {
                $scope.story.teams.push(model);
            }
        };

        /**
         * Remove a user from the permissions.
         */
        $scope.removeUser = function (model) {
            var idx = $scope.story.users.indexOf(model);
            $scope.story.users.splice(idx, 1);
        };

        /**
         * Remove a team from the permissions.
         */
        $scope.removeTeam = function(model) {
            var idx = $scope.story.teams.indexOf(model);
            $scope.story.teams.splice(idx, 1);
        };
    })
;

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 * Copyright (c) 2016 Codethink Ltd.
 * Copyright (c) 2017 Adam Coldrick
 *
 * Licensed under the Apache License, Version 2.0 (the 'License'); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Controller for the New Story view.
 *
 * This view is accessible from /#!/story/new, and can understand the
 * following parameters in the URL:
 *
 * - title: The initial story title
 * - description: The initial content of the story description
 * - force_private: If truthy, the story will be forced to be private
 *                  (i.e. private with the option to change privacy hidden).
 * - private: If truthy, the story will begin set to private.
 *            Unlike force_private, this allows the user to change to public.
 * - security: If truthy, the story will begin set as security-related.
 * - tags: Tags to set on the story. Can be given multiple times.
 * - team_id: A team ID to grant permissions for this story to. Can be
 *            given multiple times.
 * - user_id: A user ID to grant permissions for this story to. Can be
 *            given multiple times.
 */
angular.module('sb.story').controller('StoryNewController',
    function ($scope, $state, $stateParams, Story, Project, Branch, Tags,
              Task, Team, User, StoryHelper, $q, storyboardApiBase,
              currentUser) {
        'use strict';

        /**
         * Handle any change to whether or not the story is security-related
         */
        $scope.updateSecurity = function(forcePrivate, update) {
            $scope.privacyLocked = StoryHelper.updateSecurity(
                forcePrivate, update, $scope.story, $scope.tasks);
        };

        var story = new Story({
            title: $stateParams.title,
            description: $stateParams.description,
            private: (!!$stateParams.private ||
                      !!$stateParams.force_private ||
                      !!$stateParams.security),
            security: !!$stateParams.security,
            users: [currentUser],
            teams: []
        });

        // Convert the user_id and team_id parameters to arrays if needed
        var stateTeamIds = [];
        if (!!$stateParams.team_id) {
            if ($stateParams.team_id.constructor !== Array) {
                stateTeamIds.push($stateParams.team_id);
            } else {
                stateTeamIds.push.apply(stateTeamIds, $stateParams.team_id);
            }
        }
        var stateUserIds = [];
        if (!!$stateParams.user_id) {
            if ($stateParams.user_id.constructor !== Array) {
                stateUserIds.push($stateParams.user_id);
            } else {
                stateUserIds.push.apply(stateUserIds, $stateParams.user_id);
            }
        }
        // Populate the story's permission lists as requested
        angular.forEach(stateTeamIds, function(team_id) {
            Team.get({team_id: team_id}).$promise.then(function(team) {
                story.teams.push(team);
            });
        });
        angular.forEach(stateUserIds, function(user_id) {
            User.get({id: user_id}).$promise.then(function(user) {
                story.users.push(user);
            });
        });

        $scope.story = story;

        // Convert the tags parameter into an array if it isn't already
        if (!!$stateParams.tags && $stateParams.tags.constructor !== Array) {
            $stateParams.tags = [$stateParams.tags];
        }
        // List of tags to give the story
        $scope.tags = $stateParams.tags || [];

        /**
         * If the force_private query parameter is set, then enforce story
         * privacy and hide the option from the user.
         **/
        $scope.forcePrivate = !!$stateParams.force_private;

        /**
         * All tasks associated with this story, resolved in the state.
         *
         * @type {[Task]}
         */
        $scope.projectNames = [];
        $scope.projects = {};
        $scope.tasks = [];
        $scope.updateSecurity(true, false);

        /**
         * UI flag for when we're initially loading the view.
         *
         * @type {boolean}
         */
        $scope.isLoading = true;

        /**
         * UI view for when a change is round-tripping to the server.
         *
         * @type {boolean}
         */
        $scope.isUpdating = false;

        $scope.isValid = function() {
            return !!$scope.tasks.length && !!$scope.story.title;
        };

        /**
         * Create the tasks for the story.
         */
        function createTasks(createdStory) {
            var resolvingTasks = $scope.tasks.length;
            angular.forEach($scope.tasks, function(task) {
                task.story_id = createdStory.id;
                task.$create(function() {
                    resolvingTasks--;
                    if (resolvingTasks === 0) {
                        // Finished, navigate to the new story
                        $state.go('sb.story.detail',
                                  {storyId: createdStory.id});
                    }
                });
            });
        }

        /**
         * Create the story.
         */
        $scope.createStory = function() {
            $scope.isUpdating = true;
            // Ensure story is private if force_private is set.
            if ($scope.forcePrivate) {
                $scope.story.private = true;
            }
            $scope.story.$create(function(createdStory) {
                var tagsController = new Story.TagsController({
                    id: createdStory.id
                });
                if ($scope.tags.length > 0) {
                    // Create tags for the story, then tasks
                    tagsController.$update({tags: $scope.tags}, createTasks);
                } else {
                    // Create tasks for the story
                    createTasks(createdStory);
                }
            }, function() {
                $scope.isUpdating = false;
            });
        };

        /**
         * User/team typeahead search method.
         */
        $scope.searchActors = function (value, users, teams) {
            var userIds = users.map(function(user){return user.id;});
            var teamIds = teams.map(function(team){return team.id;});
            var deferred = $q.defer();
            var usersDeferred = $q.defer();
            var teamsDeferred = $q.defer();

            User.browse({full_name: value, limit: 10},
                function(searchResults) {
                    var results = [];
                    angular.forEach(searchResults, function(result) {
                        if (userIds.indexOf(result.id) === -1) {
                            result.name = result.full_name;
                            result.type = 'user';
                            results.push(result);
                        }
                    });
                    usersDeferred.resolve(results);
                }
            );
            Team.browse({name: value, limit: 10},
                function(searchResults) {
                    var results = [];
                    angular.forEach(searchResults, function(result) {
                        if (teamIds.indexOf(result.id) === -1) {
                            result.type = 'team';
                            results.push(result);
                        }
                    });
                    teamsDeferred.resolve(results);
                }
            );

            var searches = [teamsDeferred.promise, usersDeferred.promise];
            $q.all(searches).then(function(searchResults) {
                var results = [];
                angular.forEach(searchResults, function(promise) {
                    angular.forEach(promise, function(result) {
                        results.push(result);
                    });
                });
                deferred.resolve(results);
            });

            return deferred.promise;
        };

        /**
         * Add a new user or team to one of the permission levels.
         */
        $scope.addActor = function (model) {
            if (model.type === 'user') {
                $scope.story.users.push(model);
            } else if (model.type === 'team') {
                $scope.story.teams.push(model);
            }
        };

        /**
         * Remove a user from the permissions.
         */
        $scope.removeUser = function (model) {
            var idx = $scope.story.users.indexOf(model);
            $scope.story.users.splice(idx, 1);
        };

        /**
         * Remove a team from the permissions.
         */
        $scope.removeTeam = function(model) {
            var idx = $scope.story.teams.indexOf(model);
            $scope.story.teams.splice(idx, 1);
        };

        // ###################################################################
        // Task Management
        // ###################################################################

        /**
         * The new task for the task form.
         */
        $scope.newTask = new Task({
            project_id: $stateParams.project_id,
            show: true,
            status: 'todo',
            title: $stateParams.title
        });

        function mapTaskToProject(task) {
            Project.get({id: task.project_id}).$promise.then(function(project) {
                var idx = $scope.projectNames.indexOf(project.name);
                if (idx < 0) {
                    $scope.projectNames.push(project.name);
                    $scope.projects[project.name] = project;
                    $scope.projects[project.name].branchNames = [];
                    $scope.projects[project.name].branches = {};
                }
                Branch.get({id: task.branch_id}).$promise.then(
                    function(branch) {
                        var branchIdx = $scope.projects[project.name]
                            .branchNames.indexOf(branch.name);
                        if (branchIdx > -1) {
                            $scope.projects[project.name].branches[branch.name]
                                .tasks.push(task);
                        } else {
                            $scope.projects[project.name]
                                .branches[branch.name] = branch;
                            $scope.projects[project.name]
                                .branches[branch.name].tasks = [task];
                            $scope.projects[project.name]
                                .branches[branch.name].newTask = new Task({
                                    story_id: $scope.story.id,
                                    branch_id: branch.id,
                                    project_id: project.id,
                                    status: 'todo'
                                });
                            $scope.projects[project.name]
                                .branchNames.push(branch.name);
                        }
                    });
            });
        }

        $scope.validTask = function(task) {
            return !isNaN(task.project_id) && !!task.title;
        };

        /**
         * Adds a task.
         */
        $scope.createTask = function(task, branch) {
            // Make a copy to save, so that the next task retains the
            // old information (for easier continuous editing).
            var savedTask = new Task(angular.copy(task));
            if (branch) {
                branch.tasks.push(savedTask);
            } else {
                var params = {project_id: task.project_id, name: 'master'};
                Branch.browse(params).$promise.then(function(result) {
                    if (result) {
                        savedTask.branch_id = result[0].id;
                        mapTaskToProject(savedTask);
                    }
                });
            }
            $scope.tasks.push(savedTask);
            $scope.updateSecurity(true, false);
            task.title = '';
        };

        /**
         * Cleans up the project/branch/task tree.
         *
         * If there are no tasks remaining in the given branch in the given
         * project, then remove that branch from the project's list of
         * branches to display tasks for.
         *
         * If there are then no branches left in the project, remove the
         * project from the list of projects to display tasks for.
         */
        function cleanBranchAndProject(projectName, branchName) {
            var project = $scope.projects[projectName];
            var branch = project.branches[branchName];
            var nameIdx = -1;

            if (branch.tasks.length === 0) {
                nameIdx = project.branchNames.indexOf(branchName);
                if (nameIdx > -1) {
                    project.branchNames.splice(nameIdx, 1);
                }
                delete project.branches[branchName];
            }
            if (project.branchNames.length === 0) {
                nameIdx = $scope.projectNames.indexOf(projectName);
                if (nameIdx > -1) {
                    $scope.projectNames.splice(nameIdx, 1);
                }
                delete $scope.projects[projectName];
            }
        }

        /**
         * Updates the task list.
         */
        $scope.updateTask = function (task, fieldName, value, projectName,
                                      branchName) {
            var params = {id: task.id};
            params[fieldName] = value;
            task[fieldName] = value;
            if (fieldName === 'project_id') {
                var project = $scope.projects[projectName];
                var branch = project.branches[branchName];

                var branchTaskIndex = branch.tasks.indexOf(task);
                if (branchTaskIndex > -1) {
                    branch.tasks.splice(branchTaskIndex, 1);
                }

                cleanBranchAndProject(projectName, branchName);
                mapTaskToProject(task);
                $scope.updateSecurity(true, false);
            }
        };

        $scope.editNotes = function(task) {
            task.tempNotes = task.link;
            task.editing = true;
        };

        $scope.cancelEditNotes = function(task) {
            task.tempNotes = '';
            task.editing = false;
        };

        /**
         * Removes this task
         */
        $scope.removeTask = function (task, projectName, branchName) {
            var taskIndex = $scope.tasks.indexOf(task);
            if (taskIndex > -1) {
                $scope.tasks.splice(taskIndex, 1);
            }

            var project = $scope.projects[projectName];
            var branch = project.branches[branchName];
            var branchTaskIndex = branch.tasks.indexOf(task);
            if (branchTaskIndex > -1) {
                branch.tasks.splice(branchTaskIndex, 1);
            }
            cleanBranchAndProject(projectName, branchName);
        };

        // ###################################################################
        // Tags Management
        // ###################################################################

        /**
         * Show an input for a new tag
         *
         * @type {boolean}
         */
        $scope.showAddTag = false;

        $scope.toggleAddTag = function() {
            $scope.showAddTag = !$scope.showAddTag;
        };

        $scope.addTag = function (tag_name) {
            if(!!tag_name) {
                $scope.showAddTag = false;
                $scope.tags.push(tag_name);
            }
        };

        $scope.removeTag = function (tag_name) {
            var tagIndex = $scope.tags.indexOf(tag_name);
            if (tagIndex > -1) {
                $scope.tags.splice(tagIndex, 1);
            }
        };

        $scope.searchTags = function (value) {
            return Tags.browse({name: value, limit: 10}).$promise;
        };

        $scope.updateViewValue = function (value) {
            $scope.newTag.name = value;
        };

        $scope.newTag = {};
    });

/*
 * Copyright (c) 2016 Codethink Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the 'License'); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Controller for "Add task to worklist" modal.
 */
angular.module('sb.story').controller('StoryTaskAddWorklistController',
    function ($log, $scope, $state, task, $modalInstance, Worklist) {
        'use strict';

        $scope.task = task;
        $scope.defaultCriteria = [];

        $scope.selected = 'none';

        $scope.select = function(worklist) {
            $scope.selected = worklist;
        };

        $scope.add = function() {
            var params = {
                item_id: task.id,
                id: $scope.selected.id,
                list_position: $scope.selected.items.length,
                item_type: 'task'
            };

            Worklist.ItemsController.create(params, function(item) {
                $modalInstance.dismiss(item);
            });
        };

        $scope.close = function () {
            $modalInstance.dismiss('cancel');
        };
    }
);

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the 'License'); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Story detail &  manipulation controller.
 */
angular.module('sb.story').controller('StoryTaskDeleteController',
    function ($log, $scope, $state, task, $modalInstance, params) {
        'use strict';

        $scope.task = task;
        $scope.params = params;

        // Set our progress flags and clear previous error conditions.
        $scope.isUpdating = true;
        $scope.error = {};

        $scope.remove = function () {
            $scope.task.$delete(
                function () {
                    $modalInstance.close('success');
                }
            );
        };

        $scope.close = function () {
            $modalInstance.dismiss('cancel');
        };
    });

/*
 * Copyright (c) 2016 Codethink Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the 'License'); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Modal to display worklists related to stories
 */
angular.module('sb.story').controller('StoryWorklistsController',
    function($scope, $modalInstance, worklists) {
        'use strict';

        /**
         * Close this modal.
         */
        $scope.close = function () {
            $modalInstance.dismiss('cancel');
        };

        $scope.worklists = worklists;
    }
);

/*
 * Copyright (c) 2014 Mirantis Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

angular.module('sb.story').controller('TimelineFilterController',
    function ($scope, $modalInstance, Preference, TimelineEventTypes) {
        'use strict';

        function init() {
            TimelineEventTypes.forEach(function (type) {
                var pref_name = 'display_events_' + type;
                $scope[pref_name] = Preference.get(pref_name);
            });
        }

        $scope.close = function () {
            $modalInstance.dismiss('cancel');
        };

        $scope.save = function () {

            TimelineEventTypes.forEach(function (type) {
                var pref_name = 'display_events_' + type;
                var old_value = Preference.get(pref_name);
                var new_value = $scope[pref_name];

                if (old_value !== new_value) {
                    Preference.set(pref_name, new_value);
                }
            });

            return $modalInstance.close();
        };

        init();

    })
;

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */


angular.module('sb.story').factory('NewStoryService',
    function ($modal, $log, Session, SessionModalService) {
        'use strict';

        return {
            showNewStoryModal: function (projectId) {
                if (!Session.isLoggedIn()) {
                    return SessionModalService.showLoginRequiredModal();
                } else {
                    var modalInstance = $modal.open(
                        {
                            size: 'lg',
                            templateUrl: 'app/stories/template/new.html',
                            backdrop: 'static',
                            controller: 'StoryModalController',
                            resolve: {
                                params: function () {
                                    return {
                                        projectId: projectId || null
                                    };
                                }
                            }
                        }
                    );

                    // Return the modal's promise.
                    return modalInstance.result;
                }
            }
        };
    }
);

/*
 * Copyright (c) 2019 Codethink Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the 'License'); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

angular.module('sb.story').factory('StoryHelper',
    function(Story, Team) {
        'use strict';

        function updateSecurity(forcePrivate, update, story, tasks) {
            var privacyLocked;
            if (story.security) {
                if (forcePrivate) {
                    story.private = true;
                    privacyLocked = true;
                }

                // Add security teams for affected projects
                var projects = tasks.map(function(task) {
                    return task.project_id;
                }).filter(function(value) {
                    // Remove any unset project_ids we've somehow got.
                    // Otherwise, the browse will return all teams, rather
                    // than only relevant teams.
                    return !isNaN(value);
                });
                angular.forEach(projects, function(project_id) {
                    Team.browse({project_id: project_id}, function(teams) {
                        var teamIds = story.teams.map(function(team) {
                            return team.id;
                        });
                        teams = teams.filter(function(team) {
                            return ((teamIds.indexOf(team.id) === -1)
                                    && team.security);
                        });
                        angular.forEach(teams, function(team) {
                            story.teams.push(team);
                            if (update) {
                                Story.TeamsController.create({
                                    story_id: story.id,
                                    team_id: team.id
                                });
                            }
                        });
                    });
                });
            } else {
                if (forcePrivate) {
                    privacyLocked = false;
                }
            }
            return privacyLocked;
        }

        return {
            updateSecurity: updateSecurity
        };
    }
);

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 * Copyright (c) 2016 Codethink Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/*
 * Automatic configuration loader. Checks the /config endpoint, and if
 * it receives a JSON hash will load all the found values into the application
 * module.
 */
angular.element(document)
    .ready(function () {
        'use strict';

        var initInjector = angular.injector(['ng']);
        var $http = initInjector.get('$http');
        var $log = initInjector.get('$log');

        function initializeApplication(config) {
            var defaults = {
                enableEditableComments: false
            };

            // Set default config values
            for (var key in defaults) {
                if (!config.hasOwnProperty(key)) {
                    config[key] = defaults[key];
                }
            }

            // Load everything we got into our module.
            for (key in config) {
                $log.debug('Configuration: ' + key + ' -> ' + config[key]);
                angular.module('storyboard').constant(key, config[key]);
            }
            angular.bootstrap(document, ['storyboard']);
        }

        $log.info('Attempting to load parameters from ./config.json');
        $http.get('./config.json').then(
            function (response) {
                initializeApplication(response.data);
            },
            function () {
                $log.warn('Cannot load ./config.json, using defaults.');
                initializeApplication({});
            }
        );
    }
);

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Application controller, which keeps track of our active state chain and
 * sets global scope variable accordingly.
 */
angular.module('storyboard').controller('ApplicationController',
    function ($scope, $state, $rootScope, $transitions) {
        'use strict';

        /**
         * This method traverses the current active state tree to see if the
         * current state, or any of its parents, have a submenu declared.
         *
         * @param state
         */
        function hasSubmenu(state) {
            for (var stateName in state.views) {
                if (!!stateName.match(/^submenu@/)) {
                    return true;
                }
            }

            if (!!state.parent) {
                return hasSubmenu(state.parent);
            } else {
                return false;
            }
        }

        /**
         * Listen to changes in the state machine to trigger our submenu
         * scan.
         */
        $transitions.onSuccess({},
            function () {
                $scope.hasSubmenu = hasSubmenu($state.$current);
            });

        // Set a sane default.
        $scope.hasSubmenu = false;

    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Controller for our application header. Includes a typeahead-style quicknav
 * and search box.
 */
angular.module('storyboard').controller('HeaderController',
    function ($q, $scope, $rootScope, $state, $modal, NewStoryService,
              Session, SessionState, CurrentUser, Criteria, Notification,
              Priority, Project, Story, ProjectGroup, NewWorklistService,
              NewBoardService, SessionModalService, Severity, Task) {
        'use strict';

        function resolveCurrentUser() {
            CurrentUser.resolve().then(
                function (user) {
                    $scope.currentUser = user;
                },
                function () {
                    $scope.currentUser = null;
                }
            );
        }

        resolveCurrentUser();

        /**
         * Load and maintain the current user.
         */
        $scope.currentUser = null;

        /**
         * Create a new story.
         */
        $scope.newStory = function () {
            var projectId = null;
            if ($state.current.name === 'sb.project.detail') {
                projectId = $state.params.id;
            }
            NewStoryService.showNewStoryModal(projectId)
                .then(function (story) {
                    // On success, go to the story detail.
                    $scope.showMobileNewMenu = false;
                    $state.go('sb.story.detail', {storyId: story.id});
                }
            );
        };

        /**
         * Create a new worklist.
         */
        $scope.newWorklist = function () {
            NewWorklistService.showNewWorklistModal()
                .then(function (worklist) {
                    // On success, go to the worklist detail.
                    $scope.showMobileNewMenu = false;
                    $state.go('sb.worklist.detail',
                              {worklistID: worklist.id});
                }
            );
        };

        /**
         * Create a new board.
         */
        $scope.newBoard = function () {
            NewBoardService.showNewBoardModal()
                .then(function (board) {
                    // On success, go to the board detail.
                    $scope.showMobileNewMenu = false;
                    $state.go('sb.board.detail', {boardID: board.id});
                }
            );
        };

        /**
         * Create a new project.
         */
        $scope.newProject = function () {
            $scope.modalInstance = $modal.open({
                size: 'lg',
                templateUrl: 'app/projects/template/new.html',
                backdrop: 'static',
                controller: 'ProjectNewController'
            });
        };

        /**
         * Create a new project-group.
         */
        $scope.newProjectGroup = function () {
            $scope.modalInstance = $modal.open(
                {
                    templateUrl: 'app/project_group/template/new.html',
                    backdrop: 'static',
                    controller: 'ProjectGroupNewController'
                });

            $scope.modalInstance.result.then(function (projectGroup) {
                    // On success, go to the project group detail.
                    $scope.showMobileNewMenu = false;
                    $state.go(
                        'sb.project_group.detail',
                        {id: projectGroup.id}
                    );
                });
        };

        /**
         * Show modal informing the user login is required.
         */
        $scope.showLoginRequiredModal = function() {
            SessionModalService.showLoginRequiredModal();
        };

        /**
         * Log out the user.
         */
        $scope.logout = function () {
            Session.destroySession();
        };

        /**
         * Initialize the search string.
         */
        $scope.searchString = '';

        /**
         * Send the user to search and clear the header search string.
         */
        $scope.search = function (criteria) {

            switch (criteria.type) {
                case 'Text':
                    $state.go('sb.search', {q: criteria.value});
                    break;
                case 'ProjectGroup':
                    $state.go('sb.project_group.detail', {id: criteria.value});
                    break;
                case 'Project':
                    $state.go('sb.project.detail', {id: criteria.value});
                    break;
                case 'Story':
                    $state.go('sb.story.detail', {storyId: criteria.value});
                    break;
                case 'Task':
                    $state.go('sb.task', {taskId: criteria.value});
                    break;
            }

            $scope.searchString = '';
        };

        /**
         * Filter down the search string to actual resources that we can
         * browse to directly (Explicitly not including users here). If the
         * search string is entirely numeric, we'll instead do a
         * straightforward GET :id.
         */
        $scope.quickSearch = function (searchString) {
            var deferred = $q.defer();

            searchString = searchString || '';

            var searches = [];
            var headerGET = false;

            Notification.intercept(function(message) {
                if (message.type === 'http' &&
                    message.severity === Severity.ERROR &&
                    message.message === 404 &&
                    headerGET
                ) {
                    return true;
                }
            });

            if (searchString.match(/^[0-9]+$/)) {
                var getProjectGroupDeferred = $q.defer();
                var getProjectDeferred = $q.defer();
                var getStoryDeferred = $q.defer();
                var getTaskDeferred = $q.defer();
                headerGET = true;

                ProjectGroup.get({id: searchString},
                    function (result) {
                        getProjectGroupDeferred.resolve(Criteria.create(
                            'ProjectGroup', result.id, result.name
                        ));
                    }, function () {
                        getProjectGroupDeferred.resolve(null);
                    });
                Project.get({id: searchString},
                    function (result) {
                        getProjectDeferred.resolve(Criteria.create(
                            'Project', result.id, result.name
                        ));
                    }, function () {
                        getProjectDeferred.resolve(null);
                    });
                Story.get({id: searchString},
                    function (result) {
                        getStoryDeferred.resolve(Criteria.create(
                            'Story', result.id, result.title
                        ));
                    }, function () {
                        getStoryDeferred.resolve(null);
                    });
                Task.get({id: searchString},
                    function (result) {
                        getTaskDeferred.resolve(Criteria.create(
                            'Task', result.id, result.title
                        ));
                    }, function () {
                        getTaskDeferred.resolve(null);
                    });

                // If the search string is entirely numeric, do a GET.
                searches.push(getProjectGroupDeferred.promise);
                searches.push(getProjectDeferred.promise);
                searches.push(getStoryDeferred.promise);
                searches.push(getTaskDeferred.promise);

            } else {
                searches.push(ProjectGroup.criteriaResolver(searchString, 5));
                searches.push(Project.criteriaResolver(searchString, 5));
                searches.push(Story.criteriaResolver(searchString, 5));
            }
            $q.all(searches).then(function (searchResults) {
                headerGET = false;
                var criteria = [
                    Criteria.create('Text', searchString)
                ];


                /**
                 * Add a result to the returned criteria.
                 */
                var addResult = function (item) {
                    criteria.push(item);
                };

                for (var i = 0; i < searchResults.length; i++) {
                    var results = searchResults[i];

                    if (!results) {
                        continue;
                    }

                    if (!!results.forEach) {

                        // If it's iterable, do that. Otherwise just add it.
                        results.forEach(addResult);
                    } else {
                        addResult(results);
                    }
                }

                deferred.resolve(criteria);
            });

            // Return the search promise.
            return deferred.promise;
        };

        // Watch for changes to the session state.
        Notification.intercept(function (message) {
            switch (message.type) {
                case SessionState.LOGGED_IN:
                    resolveCurrentUser();
                    break;
                case SessionState.LOGGED_OUT:
                    $scope.currentUser = null;
                    break;
                default:
                    break;
            }
        }, Priority.LAST);
    });

/*
 * Copyright (c) 2013 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A directive which checks, enables, and disables subscriptions by resource.
 *
 * @author Michael Krotscheck
 */
angular.module('sb.util').directive('subscribe',
    function (CurrentUser, Notification, Priority, Session, SessionState,
              Subscription) {
        'use strict';

        return {
            restrict: 'E',
            scope: {
                resource: '@',
                resourceId: '=',
                subscriptions: '='
            },
            templateUrl: 'app/subscription/template/subscribe.html',
            link: function ($scope) {

                /**
                 * When we start, create a promise for the current user.
                 */
                var cuPromise = CurrentUser.resolve();

                /**
                 * Is this control currently enabled?
                 *
                 * @type {boolean}
                 */
                $scope.enabled = Session.isLoggedIn();

                /**
                 * Is this user subscribed to this resource?
                 *
                 * @type {boolean}
                 */
                $scope.subscribed = false;

                /**
                 * Is the control currently trying to resolve the user's
                 * subscription?
                 *
                 * @type {boolean}
                 */
                $scope.resolving = false;

                /**
                 * The loaded subscription resource
                 *
                 * @type {Object}
                 */
                $scope.subscription = null;

                /**
                 * The current user.
                 *
                 * @param currentUser
                 */
                $scope.currentUser = null;
                cuPromise.then(function (user) {
                    $scope.currentUser = user;
                });

                /**
                 * Set or clear the subscription.
                 */
                function setSubscription(subscription) {
                    $scope.subscription = subscription || null;
                    $scope.subscribed = !!$scope.subscription;
                }

                /**
                 * Resolve whether the current user already has a subscription
                 * to this resource.
                 */
                function resolveSubscription() {

                    if (!Session.isLoggedIn()) {
                        setSubscription();
                        return;
                    }

                    $scope.resolving = true;

                    cuPromise.then(
                        function (user) {
                            Subscription.browse({
                                    user_id: user.id,
                                    target_type: $scope.resource,
                                    target_id: $scope.resourceId
                                },
                                function (results) {
                                    setSubscription(results[0]);
                                    $scope.resolving = false;
                                },
                                function () {
                                    setSubscription();
                                    $scope.resolving = false;
                                }
                            );

                        }
                    );
                }

                // Subscribe to login/logout events for enable/disable/resolve.
                var removeNotifier = Notification.intercept(function (message) {
                    switch (message.type) {
                        case SessionState.LOGGED_IN:
                            $scope.enabled = true;
                            resolveSubscription();
                            break;
                        case SessionState.LOGGED_OUT:
                            $scope.enabled = false;
                            $scope.subscribed = false;
                            break;
                    }

                }, Priority.LAST);

                // Remove the notifier when this scope is destroyed.
                $scope.$on('$destroy', removeNotifier);

                /**
                 * When the user clicks on this control, activate/deactivate the
                 * subscription.
                 */
                $scope.toggleSubscribe = function () {
                    if ($scope.resolving) {
                        return;
                    }

                    $scope.resolving = true;

                    if (!!$scope.subscription) {
                        $scope.subscription.$delete(function () {
                            setSubscription();
                            $scope.resolving = false;
                        }, function () {
                            $scope.resolving = false;
                        });
                    } else {

                        cuPromise.then(
                            function (user) {
                                var sub = new Subscription({
                                    user_id: user.id,
                                    target_type: $scope.resource,
                                    target_id: $scope.resourceId
                                });
                                sub.$create(function (result) {
                                    setSubscription(result);
                                    $scope.resolving = false;
                                }, function () {
                                    $scope.resolving = false;
                                });
                            }
                        );
                    }
                };

                /** Set subscriptions based on list passed from
                * stories/projects/project_groups controller(s)
                */
                function resolveSubsList() {

                    if (!Session.isLoggedIn()) {
                        setSubscription();
                        return;
                    }

                    $scope.resolving = true;

                    cuPromise.then(function () {
                        angular.forEach($scope.subscriptions, function(sub) {
                            if ($scope.resourceId === sub.target_id) {
                                setSubscription(sub);
                            }
                        });
                        $scope.resolving = false;
                    });
                }

                // On initialization, resolve.
                if (!!$scope.subscriptions) {
                    $scope.subscriptions.$promise.then(resolveSubsList);
                } else {
                    resolveSubscription();
                }
            }
        };
    });

/*
 * Copyright (c) 2014 Mirantis Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

angular.module('sb.util').constant('TimelineEventTypes',
    [
        'story_created',
        'story_details_changed',
        'tags_added',
        'tags_deleted',
        'task_created',
        'task_assignee_changed',
        'task_status_changed',
        'task_priority_changed',
        'task_details_changed',
        'task_deleted',
        'user_comment'
    ]
);

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The validation regex used by our title forms, injected into the rootScope
 * as a constant.
 */
angular.module('sb.util').run(function ($rootScope) {
    'use strict';
    var regex = '^[a-zA-Z0-9]+([\\-\\./]?[a-zA-Z0-9]+)*$';
    $rootScope.PROJECT_NAME_REGEX = new RegExp(regex);
});

/*
 * Copyright (c) 2013 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A tag-input control that autocompletes based on a fixed (non-autoloading)
 * set of source data, and restricts inputs to items in that list.
 */
angular.module('sb.util').directive('tagComplete',
    function ($q, $parse, $rootScope, $position, typeaheadParser, $timeout) {
        'use strict';

        var HOT_KEYS = [9, 13, 27, 38, 40];

        return {
            restrict: 'EA',
            replace: true,
            scope: {
                tagCompleteTags: '=',
                tagCompleteLabelField: '@',
                tagCompleteTagTemplateUrl: '=',
                tagCompleteOptionTemplateUrl: '=',
                tagCompleteVerify: '&',
                tagCompleteOnSelect: '&',
                tagCompleteLoading: '&',
                tagRemoveCallback: '&',
                maxTags: '=',
                placeholder: '@'
            },
            templateUrl: 'app/util/template/tag_complete.html',
            link: function ($scope, $element, attrs) {
                /**
                 * Grab our input.
                 */
                var $input = $element.find('input');

                /**
                 * Set the input's placeholder text.
                 */
                $scope.$watch(function () {
                    return $scope.tagCompleteTags.length;
                }, function(tags) {
                    if (tags > 0) {
                        $input[0].placeholder = '';
                    } else {
                        $input[0].placeholder = $scope.placeholder || '';
                    }
                });

                /**
                 * Override the element's focus method, use it to focus our
                 * input instead.
                 */
                $element[0].focus = function () {
                    $input[0].focus();
                };

                /**
                 * The typeahead query parser, used for our selection syntax.
                 */
                var parserResult = typeaheadParser.parse(attrs.tagComplete);

                /**
                 * A setter that simplifies setting a property in the original
                 * scope when an async load is kicked off.
                 *
                 * @type {Function}
                 */
                var isLoadingSetter = function (isLoading) {
                    if (!!$scope.tagCompleteLoading) {
                        $scope.tagCompleteLoading({isLoading: isLoading});
                    }
                };

                /**
                 * URL for the template to use for the dropdown rendering.
                 *
                 * @type {String}
                 */
                $scope.tagCompleteTemplateUrl =
                    $parse(attrs.tagCompleteTemplateUrl);

                /**
                 * Are we focused?
                 * @type {Boolean}
                 */
                $scope.hasFocus = false;

                /**
                 * Reset the matches on the scope.
                 */
                var resetMatches = function () {
                    $scope.matches = [];
                    $scope.activeIdx = -1;
                };

                /**
                 * Search for matches...
                 *
                 * @param inputValue
                 */
                var getMatchesAsync = function (inputValue) {

                    if (!$scope.hasFocus || !inputValue) {
                        return;
                    }

                    var locals = {$viewValue: inputValue};
                    isLoadingSetter(true);

                    $q.when(parserResult.source($scope.$parent, locals))
                        .then(function (matches) {


                            // Make sure that the returned query equals what's
                            // currently being searched for: It could be that
                            // we have multiple queries in flight...
                            if (inputValue === $scope.newTagName &&
                                $scope.hasFocus) {

                                // Transform Matches
                                $scope.matches = [];
                                for (var i = 0; i < matches.length; i++) {
                                    $scope.matches.push({
                                        label: parserResult
                                            .viewMapper($scope.$parent, locals),
                                        model: matches[i]
                                    });
                                }

                                if (matches.length > 0) {
                                    $scope.activeIdx = 0;
                                    $scope.query = inputValue;
                                } else {
                                    resetMatches();
                                }
                                isLoadingSetter(false);

                                // Position pop-up with matches - we need to
                                // re-calculate its position each time we are
                                // opening a window with matches due to other
                                // elements being rendered
                                $scope.position = $position.position($element);
                                $scope.position.top = $scope.position.top +
                                    $element.prop('offsetHeight');
                            }
                        }, function () {
                            resetMatches();
                            isLoadingSetter(false);
                        });
                };

                // Watch the model and trigger searches when the value changes.
                $scope.$watch(function () {
                    return $scope.newTagName;
                }, getMatchesAsync);

                /**
                 * Focus when the input gets focus.
                 */
                $input.on('focus', function () {
                    $scope.hasFocus = true;
                    if (!$rootScope.$$phase) {
                        $scope.$digest();
                    }
                });

                /**
                 * Blur when the input gets blurred.
                 */
                $input.on('blur', function () {
                    $timeout(function() {
                        resetMatches();
                        $scope.newTagName = '';
                        $scope.hasFocus = false;
                        if (!$rootScope.$$phase) {
                            $scope.$digest();
                        }
                    }, 200);
                });

                /**
                 * Bind to arrow controls, escape, and return.
                 */
                $input.on('keydown', function (evt) {

                    // Make sure we have something to react to.
                    if ($scope.matches.length === 0 ||
                        HOT_KEYS.indexOf(evt.which) === -1) {
                        return;
                    }

                    evt.preventDefault();

                    if (evt.which === 40) {
                        $scope.activeIdx = ($scope.activeIdx + 1) %
                            $scope.matches.length;
                        $scope.$digest();
                    } else if (evt.which === 38) {
                        $scope.activeIdx = ($scope.activeIdx ? $scope.activeIdx
                            : $scope.matches.length) - 1;
                        $scope.$digest();
                    } else if (evt.which === 13 || evt.which === 9) {
                        $scope.$apply(function () {
                            $scope.select($scope.activeIdx);
                        });
                    } else if (evt.which === 27) {
                        evt.stopPropagation();

                        resetMatches();
                        $scope.$digest();
                    }
                });

                /**
                 * Event handler when delete is pressed inside the input field.
                 * Pops the last item off the selected list.
                 */
                $scope.deletePressed = function () {
                    var selectedTags = $scope.tagCompleteTags || [];

                    if (selectedTags.length > 0 && !$scope.newTagName) {
                        $scope.tagRemoveCallback(selectedTags.pop());
                        return true;
                    }
                };

                /**
                 * When a user clicks on the background of this control, we want
                 * to focus the text input field.
                 */
                $scope.focus = function () {

                    $input[0].focus();
                };

                /**
                 * When a user clicks on the actual tag, we need to intercept
                 * the mouse event so it doesn't refocus the cursor.
                 */
                $scope.noFocus = function (event) {
                    event.stopImmediatePropagation();
                };

                /**
                 * Called when something's selected
                 */
                $scope.select = function (idx) {
                    var item = $scope.matches[idx].model;

                    $scope.tagCompleteTags.push(item);

                    $scope.newTagName = '';
                    resetMatches();

                    if (!!$scope.tagCompleteOnSelect) {
                        $scope.tagCompleteOnSelect({tag: item});
                    }
                };

                /**
                 * Removes a tag.
                 */
                $scope.removeTag = function (tag) {
                    if (!$scope.tagCompleteTags) {
                        return;
                    }
                    var idx = $scope.tagCompleteTags.indexOf(tag);
                    if (idx > -1) {
                        $scope.tagCompleteTags.splice(idx, 1);
                    }
                    $scope.tagRemoveCallback(tag);
                };

                resetMatches();
            }
        };
    })
    .directive('tagCompleteTag',
    function ($http, $templateCache, $compile, $parse) {
        'use strict';

        return {
            restrict: 'EA',
            scope: {
                index: '=',
                tag: '=',
                labelField: '=',
                removeTag: '&'
            },
            link: function (scope, element, attrs) {
                var tplUrl = $parse(attrs.templateUrl)(scope.$parent) ||
                    '/tag_complete/default_tag_template.html';

                $http.get(tplUrl, {cache: $templateCache})
                    .success(function (tplContent) {
                        element.replaceWith($compile(tplContent.trim())(scope));
                    });
            }
        };
    });

/*
 * Copyright (c) 2013 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A tag-input control that allows free form input of tags.
 */
angular.module('sb.util').directive('tagInput',
    function () {
        'use strict';

        return {
            restrict: 'E',
            replace: true,
            scope: {
                'selectedTags': '=ngModel'
            },
            require: 'ngModel',
            controller: 'TagInputController',
            templateUrl: 'app/util/template/tag_input.html'
        };
    });

/**
 * The controller for the above referenced Tag Input control.
 */
angular.module('sb.util').controller('TagInputController',
    function ($element, $scope) {
        'use strict';

        /**
         * Variable for the input field that triggers our filter.
         */
        $scope.newTagName = '';

        /**
         * Event handler when delete is pressed inside the input field. Pops
         * the last item off the selected list.
         */
        $scope.deletePressed = function () {
            if ($scope.newTagName.length === 0) {
                $scope.selectedTags.pop();
                return true;
            }
            return false;
        };

        /**
         * Adds a tag.
         */
        $scope.addTag = function () {
            // Do we have a model?
            if (!$scope.selectedTags) {
                $scope.selectedTags = [];
            }
            if ($scope.newTagName.length > 0 &&
                $scope.selectedTags.indexOf($scope.newTagName) === -1) {

                $scope.selectedTags.push($scope.newTagName);
                $scope.newTagName = '';
            }
        };

        /**
         * When a user clicks on the background of this control, we want to
         * focus the text input field.
         */
        $scope.focus = function () {
            $element.find('input[name=tagInputField]').focus();
        };

        /**
         * When a user clicks on the actual tag, we need to intercept the
         * mouse event so it doesn't refocus the cursor.
         */
        $scope.noFocus = function (event) {
            event.stopImmediatePropagation();
        };
    }
);

/*
 * Copyright (c) 2013 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * This directive requires ui-router, as it listens for events dispatched as
 * a user navigates through the application, and adds the 'active' class to
 * the bound element if the user's selected path matches the one configured.
 *
 * @author Michael Krotscheck
 */
angular.module('sb.util').directive('activePath',
    function ($location, $rootScope, $transitions) {
        'use strict';

        return {
            link: function ($scope, element, attrs) {
                var activePath = attrs.activePath;

                function setActivePath() {
                    var path = $location.path();
                    var isMatchedPath = path.match(activePath) !== null;

                    element.toggleClass('active', isMatchedPath);
                }

                // This is angularjs magic, the return method from any $on
                // binding will return a function that will disconnect
                // that binding.
                var disconnectBinding =
                    $transitions.onSuccess({}, function setActivePath() {
                    var path = $location.path();
                    var isMatchedPath = path.match(activePath) !== null;

                    element.toggleClass('active', isMatchedPath);
                });
                $scope.$on('$destroy', disconnectBinding);

                // INIT
                setActivePath();
            }
        };
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * This directive, when attached to an input field, will detect
 * keystrokes and automatically resize the width of the control.
 *
 * Due to browser differences, we do not have access to a consistent text
 * measurement API. Therefore, we have to maintain a 'shadow' DOM element
 * that mirrors our existing one, dump our current text into that one, and then
 * measure the results.
 */
angular.module('sb.util')
    .directive('autoresizeWidth', function ($document) {
        'use strict';

        return {
            link: function ($scope, $element) {
                // Clone the created element and attach it to our DOM.
                var shadow = angular.element('<span></span>');
                shadow.attr('class', $element.attr('class'));
                shadow.css({
                    display: 'none',
                    'white-space': 'pre',
                    width: 'auto',
                    visibility: 'hidden'
                });

                // Sane attach/detach
                $document.find('body').append(shadow);
                $scope.$on('$destroy', function () {
                    shadow.remove();
                });

                /**
                 * Recalculate size. We're binding this both to keypress and
                 * keyup, because the former allows us to trap regular
                 * keystrokes, while the latter allows us to capture copy,
                 * paste, etc. Note that this isn't perfect.
                 */
                function recalculateSize(event) {

                    var value = $element.val() || 'M'; // At least one
                    if (event && event.type === 'keypress') {
                        value += String.fromCharCode(event.which);
                    }

                    // Apply the value, trigger a rerender and then hide
                    // the control again. YES it's hacky. No, there's no better
                    // way to do this.
                    shadow.text(value);
                    shadow.css('display', 'inline-block');
                    try {
                        $element[0].offsetWidth = shadow[0].offsetWidth;
                    }
                    finally {
                        shadow.css('display', 'none');
                    }
                }

                $element.bind('keypress keyup', recalculateSize);
                recalculateSize();
            }
        };
    });

/*
 * Copyright (c) 2013 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * This directive adds a simple ng-delete directive that allows us to trap
 * the "delete" key press. It only works when a control is focused, so
 * will only work on textareas, textinputs, etc.
 */
angular.module('sb.util').directive('ngDelete', function () {
    'use strict';

    return function (scope, element, attrs) {

        element.bind('keydown keypress', function (event) {
            if (event.which === 8) {
                var preventDefault = false;

                scope.$apply(function () {
                    preventDefault = scope.$eval(attrs.ngDelete);
                });

                if (preventDefault) {
                    event.preventDefault();
                }
            }
        });
    };
});

/*
 * Copyright (c) 2013 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * This directive adds the often sought, but never found, ng-enter directive.
 * It intercepts keystrokes and will execute the bound method if that keystroke
 * is the enter key.
 *
 * @author Michael Krotscheck
 */
angular.module('sb.util').directive('ngEnter', function () {
    'use strict';

    return function (scope, element, attrs) {

        element.bind('keydown keypress', function (event) {
            if (event.which === 13) {
                scope.$apply(function () {
                    scope.$eval(attrs.ngEnter);
                });

                event.preventDefault();
            }
        });
    };
});

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * This directive may be attached to input elements, and calls the 'focus()'
 * method on that element whenever it is added to the DOM (whenever the link())
 * phase is run)
 */
angular.module('sb.util').directive('focus',
    function ($timeout) {
        'use strict';

        return {
            link: function ($scope, $element, $attrs) {

                var focus = $scope.$eval($attrs.focus);
                if (typeof focus === 'undefined') {
                    focus = true;
                }

                // If focus is set to false, don't actually do anything.
                if (!focus) {
                    return;
                }

                // Extract the element...
                var e = $element[0];

                if (!!e && !!e.focus) {
                    $timeout(function () {
                        e.focus();
                    }, 10);
                }
            }
        };
    });

/*
 * Copyright (c) 2016 Codethink Limited.
 *
 * Licensed under the Apache License, Version 2.0 (the 'License'); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Service for displaying a date/time picker.
 */
angular.module('sb.util')
    .directive('focusOnShow', function($timeout) {
        'use strict';

        return {
            restrict: 'A',
            link: function(scope, element, attrs) {
                if (attrs.ngShow) {
                    scope.$watch(attrs.ngShow, function(show) {
                        if (show) {
                            $timeout(function() {
                                element[0].focus();
                            }, 0);
                        }
                    });
                }
            }
        };
    }
);

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A directive which encapsulates typeahead logic for inline user selection.
 */
angular.module('sb.util').directive('inputInline',
    function () {
        'use strict';

        return {
            require: 'ngModel',
            restrict: 'E',
            templateUrl: 'app/util/template/input_inline.html',
            scope: {
                enabled: '=',
                asInline: '=',
                autoFocus: '=',
                onChange: '&',
                emptyPrompt: '@',
                emptyDisabledPrompt: '@',
                maxLength: '=',
                placeholder: '@placeholder'
            },
            link: function ($scope, element, attrs, ngModel) {
                /**
                 * Flag that indicates whether the form is visible.
                 *
                 * @type {boolean}
                 */
                $scope.showForm = !$scope.asInline;

                /**
                 * Toggle the display of the form.
                 */
                $scope.toggleForm = function () {
                    if (!!$scope.asInline) {
                        $scope.showForm =
                            $scope.enabled ? !$scope.showForm : false;
                    } else {
                        $scope.showForm = true;
                    }
                };

                /**
                 * Updates the view value (if necessary) and notifies external
                 * event listeners.
                 *
                 * @param value
                 */
                $scope.onInputBlur = function (value) {
                    if (value !== ngModel.$viewValue) {
                        ngModel.$setViewValue(value);
                        $scope.onChange();
                    }
                    $scope.toggleForm();
                };

                /**
                 * Watch the ng-model controller for data changes.
                 */
                $scope.$watch(function () {
                    return ngModel.$viewValue;
                }, function (value) {
                    if ($scope.inputText !== value) {
                        $scope.inputText = !!value ? value : '';
                    }
                });
            }
        };
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A directive which encapsulates typeahead logic for inline project selection.
 */
angular.module('sb.util').directive('projectTypeahead',
    function (Project, $timeout) {
        'use strict';

        return {
            require: 'ngModel',
            restrict: 'E',
            templateUrl: 'app/util/template/project_typeahead.html',
            scope: {
                enabled: '=',
                asInline: '=',
                autoFocus: '=',
                onChange: '&',
                emptyPrompt: '@',
                emptyDisabledPrompt: '@',
                placeholder: '@placeholder'
            },
            link: function ($scope, element, attrs, ngModel) {
                /**
                 * Flag that indicates whether the form is visible.
                 *
                 * @type {boolean}
                 */
                $scope.showForm = !$scope.asInline;

                /**
                 * Toggle the display of the form.
                 */
                $scope.toggleForm = function () {
                    if (!!$scope.asInline) {
                        if ($scope.showForm) {
                            $timeout(function () {
                                $scope.showForm =
                                    $scope.enabled ? !$scope.showForm : false;
                            }, 200);
                        }
                        else {
                            $scope.showForm = true;
                        }
                    } else {
                        $scope.showForm = true;
                    }
                };

                /**
                 * Project typeahead search method.
                 */
                $scope.searchProjects = function (value) {
                    return Project.browse({
                        name: value,
                        limit: 10
                    }).$promise;
                };

                /**
                 * Load the currently configured project.
                 */
                $scope.loadProject = function () {
                    var projectId = ngModel.$viewValue || null;
                    if (!!projectId) {
                        $scope.project = Project.get({id: projectId},
                            function (project) {
                                $scope.selectedProject = project;
                            }, function () {
                                $scope.project = null;
                                $scope.selectedProject = null;
                            });
                    } else {
                        $scope.project = null;
                        $scope.selectedProject = null;
                    }
                };

                /**
                 * Updates the view value (if necessary) and notifies external
                 * event listeners.
                 *
                 * @param value
                 */
                $scope.updateViewValue = function (value) {
                    if (value !== ngModel.$viewValue) {
                        ngModel.$setViewValue(value);
                        $scope.onChange();
                    }
                };

                /**
                 * Formats the project name.
                 */
                $scope.formatProjectName = function (model) {
                    if (!!model) {
                        return model.name;
                    }
                    return '';
                };

                /**
                 * Watch the ng-model controller for data changes.
                 */
                $scope.$watch(function () {
                    return ngModel.$viewValue;
                }, $scope.loadProject);
            }
        };
    });

/*
 * Copyright (c) 2013 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A convenience directive that resolves a story into the scope.
 */
angular.module('sb.util').directive('resolveStory',
    function ($parse, Story) {
        'use strict';

        return {
            restrict: 'A',
            scope: true,
            link: function ($scope, $element, $attrs) {
                var storyId = $parse($attrs.resolveStory)($scope.$parent);

                if (!!storyId) {
                    $scope.story = Story.get({id: storyId});
                } else {
                    $scope.story = null;
                }
            }
        };
    });

/*
 * Copyright (c) 2013 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * This directive adds the ng-shift-enter directive. It intercepts keystrokes
 * and will execute the bound method if that keystroke is the enter key.
 *
 * @author Michael Krotscheck
 */
angular.module('sb.util').directive('ngShiftEnter', function () {
    'use strict';

    return function (scope, element, attrs) {

        element.bind('keydown keypress', function (event) {
            if (event.which === 13 && event.shiftKey) {
                scope.$apply(function () {
                    scope.$eval(attrs.ngShiftEnter);
                });

                event.preventDefault();
            }
        });
    };
});

/*
 * Copyright (c) 2013 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A story status label that automatically selects color and text based on
 * the bound-in story.
 */
angular.module('sb.util').directive('storyStatusLabel',
    function () {
        'use strict';

        return {
            restrict: 'E',
            templateUrl: 'app/util/template/story_status_label.html',
            scope: {
                story: '='
            },
            controller: function ($scope) {

                /**
                 * Helper method to return the story status.
                 */
                function getStoryStatus() {
                    if (!$scope.story) {
                        return null;
                    } else {
                        return $scope.story.status;
                    }
                }

                /**
                 * Helper method to update the label style of the story.
                 */
                function updateStoryLabel() {
                    switch (getStoryStatus()) {
                        case 'invalid':
                            $scope.labelStyle = 'label-default';
                            break;
                        case 'active':
                            $scope.labelStyle = 'label-info';
                            break;
                        case 'merged':
                            $scope.labelStyle = 'label-success';
                            break;
                        default:
                            $scope.labelStyle = 'label-default';
                    }
                }

                var unwatch = $scope.$watch(getStoryStatus, updateStoryLabel);
                $scope.$on('$destroy', unwatch);

                updateStoryLabel();
            }
        };
    });

/*
 * Copyright (c) 2013 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A story status label that automatically selects color and text based on
 * the bound-in story.
 */
angular.module('sb.util').directive('storyTaskStatus',
    function (TaskStatus) {
        'use strict';

        return {
            restrict: 'E',
            templateUrl: 'app/util/template/story_task_status.html',
            scope: {
                story: '='
            },
            link: function ($scope) {
                TaskStatus.query({}, function (items) {
                    var statuses = [];
                    for (var i = 0;i < items.length; i++) {
                        statuses[items[i].key] = items[i].name;
                    }

                    $scope.status_texts = statuses;
                });

                $scope.orderStatus = function(item) {
                    var order = [
                        'todo',
                        'inprogress',
                        'review',
                        'invalid',
                        'merged'
                    ];

                    return order.indexOf(item.key);
                };
            }
        };
    });

/*
 * Copyright (c) 2013 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A convenience directive that allows us to bind the current task priority onto
 * a control. This control will automatically render itself either as a
 * dropdown (if editable) or as a label (if not).
 *
 */
angular.module('sb.util').directive('taskPriorityDropdown',
    function () {
        'use strict';

        return {
            restrict: 'E',
            templateUrl: 'app/util/template/task_priority_dropdown.html',
            scope: {
                priority: '@',
                onChange: '&',
                editable: '@'
            },
            link: function ($scope) {

                // Initialize the style.
                $scope.style = 'default';

                // Make sure our scope can set its own priority
                $scope.setPriority = function (newPriority) {
                    if (newPriority !== $scope.priority) {
                        $scope.priority = newPriority;
                        $scope.onChange({priority: newPriority});
                    }
                };
            }
        };
    });

/*
 * Copyright (c) 2013 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A convenience directive that allows us to bind the current task status onto
 * a control. This control will automatically render itself either as a
 * dropdown (if editable) or as a label (if not). It will also automatically
 * color itself based on the status.
 *
 * TODO(krotscheck): Once we can load task status types from the database,
 * allow binding that to this control. At that point we can revisit the color
 * mapping too, because it might be possible to genericize this.
 */
angular.module('sb.util').directive('taskStatusDropdown',
    function (TaskStatus) {
        'use strict';


        /**
         * Map our task status to a display style.
         */
        function setStyle(status) {
            switch (status) {
                case 'invalid':
                    return 'default';
                case 'merged':
                    return 'success';
                case 'inprogress':
                    return 'info';
                case 'review':
                    return 'warning';
                case 'todo':
                    return 'default';
                default:
                    return 'default';
            }
        }

        return {
            restrict: 'E',
            templateUrl: 'app/util/template/task_status_dropdown.html',
            scope: {
                status: '@',
                onChange: '&',
                editable: '@'
            },
            link: function ($scope) {
                TaskStatus.query({}, function (items) {
                        $scope.taskStatuses = items;
                        $scope.statusName = $scope.status;

                        // check if we can set current status name
                        for (var i = 0;i < items.length; i++) {
                            if (items[i].key === $scope.status) {
                                $scope.statusName = items[i].name;
                            }
                        }
                    }
                );

                // Initialize the style.
                $scope.style = setStyle($scope.status);

                // Make sure our scope can set its own status
                $scope.setStatus = function (newStatus) {
                    if (newStatus !== $scope.status) {
                        $scope.style = setStyle(newStatus);
                        $scope.status = newStatus;

                        // update status name as well
                        for (var i = 0;i < $scope.taskStatuses.length; i++) {
                            if ($scope.taskStatuses[i].key === $scope.status) {
                                $scope.statusName = $scope.taskStatuses[i].name;
                            }
                        }

                        $scope.onChange({status: newStatus});
                    }
                };
            }
        };
    });

/*
 * Copyright (c) 2013 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A story status label that automatically selects color and text based on
 * the bound-in story.
 */
angular.module('sb.util').directive('timeMoment',
    function (DateUtil) {
        'use strict';

        return {
            restrict: 'A',
            templateUrl: 'app/util/template/time_moment.html',
            scope: {
                eventdate: '=',
                shortDate: '=',
                formatString: '=',
                noTimeAgo: '@'
            },
            controller: function ($scope) {

                /**
                 * Helper method to update the needs_timeago propery
                 */
                function updateTimeAgo() {
                    if (!$scope.noTimeAgo) {
                        $scope.needsTimeAgo =
                            DateUtil.needsTimeAgo($scope.eventdate);
                    } else {
                        $scope.needsTimeAgo = false;
                    }
                }

                var unwatch = $scope.$watch(updateTimeAgo);
                $scope.$on('$destroy', unwatch);

                updateTimeAgo();
            }
        };
    }
);

/*
 * Copyright (c) 2014 Mirantis Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

angular.module('sb.util').directive('timelineEvent', function($log) {
    'use strict';

    return {
            restrict: 'E',
            replace: true,
            link: function(scope, element, attrs) {
                var tlEvent;
                try {
                    tlEvent = JSON.parse(attrs.tlEvent);
                    scope.event_type = tlEvent.event_type;
                } catch (error) {
                    $log.warn(error);
                    scope.event_type = 'unknown';
                }
            },
            templateUrl: 'app/stories/template/comments/template_switch.html'
        };
});

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A directive which encapsulates typeahead logic for inline user selection.
 */
angular.module('sb.util').directive('userTypeahead',
    function (User, $timeout) {
        'use strict';

        return {
            require: 'ngModel',
            restrict: 'E',
            templateUrl: 'app/util/template/user_typeahead.html',
            scope: {
                enabled: '=',
                asInline: '=',
                autoFocus: '=',
                onChange: '&',
                onBlur: '&',
                onFocus: '&',
                emptyPrompt: '@',
                emptyDisabledPrompt: '@',
                placeholder: '@placeholder'
            },
            link: function ($scope, element, attrs, ngModel) {
                /**
                 * Flag that indicates whether the form is visible.
                 *
                 * @type {boolean}
                 */
                $scope.showForm = !$scope.asInline;

                /**
                 * Toggle the display of the form.
                 */
                $scope.toggleForm = function (evt) {
                    // If we blur while the input is empty, try to set that
                    // value.
                    if (evt && !evt.target.value) {
                        $scope.updateViewValue(null);
                    }

                    if (!!$scope.asInline) {
                        if ($scope.showForm) {
                            $timeout(function () {
                                $scope.showForm =
                                    $scope.enabled ? !$scope.showForm : false;
                            }, 200);
                            $scope.onBlur();
                        }
                        else {
                            $scope.showForm = true;
                            $scope.onFocus();
                        }
                    } else {
                        $scope.showForm = true;
                    }
                };

                /**
                 * User typeahead search method.
                 */
                $scope.searchUsers = function (value) {
                    return User.browse({full_name: value, limit: 10}).$promise;
                };

                /**
                 * Load the currently configured user.
                 */
                $scope.loadUser = function () {
                    var userId = ngModel.$viewValue || null;
                    if (!!userId) {
                        User.get({id: userId},
                            function (user) {
                                $scope.user = user;
                                $scope.selectedUser = user;
                            }, function () {
                                $scope.user = null;
                                $scope.selectedUser = null;
                            });
                    } else {
                        $scope.user = null;
                        $scope.selectedUser = null;
                    }
                };

                /**
                 * Updates the view value (if necessary) and notifies external
                 * event listeners.
                 *
                 * @param value
                 */
                $scope.updateViewValue = function (value) {
                    if (value !== ngModel.$viewValue) {
                        ngModel.$setViewValue(value);
                        $scope.onChange();
                    }
                };

                /**
                 * Blur on escape.
                 */
                $scope.handleEscapeKey = function (evt) {
                    // Escape key
                    if (evt.keyCode === 27) {
                        evt.target.blur();
                    }
                };

                /**
                 * Save and blur on enter.
                 */
                $scope.handleEnterKey = function (evt) {

                    // The enter key
                    if (evt.keyCode === 13) {
                        // If the field is empty, try to set the view value (if
                        // there's something in it, we have to trust
                        // that the field renderer knows what it's doing).

                        if (evt && !evt.target.value) {
                            $scope.updateViewValue(null);
                        }
                        evt.target.blur();
                    }
                };

                /**
                 * Formats the user name.
                 */
                $scope.formatUserName = function (model) {
                    if (!!model) {
                        return model.full_name + ' <' + model.email + '>';
                    }
                    return '';
                };

                /**
                 * Watch the ng-model controller for data changes.
                 */
                $scope.$watch(function () {
                    return ngModel.$viewValue;
                }, $scope.loadUser);
            }
        };
    });

/*
 * Copyright (c) 2013 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A capitalization filter.
 */
angular.module('sb.util').filter('capitalize',
    function () {
        'use strict';

        return function (value) {
            try {
                return value.charAt(0).toUpperCase() + value.slice(1);
            } catch (e) {
                return value;
            }
        };
    });

/*
 * Copyright (c) 2013 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A helpful development filter that will console.log out any value you bind
 * to it in the DOM. You should probably only use this while debugging.
 *
 * @author Michael Krotscheck
 */
angular.module('sb.util').filter('debug',
    function ($log) {
        'use strict';

        return function (value) {
            $log.debug('DEBUG', value);
            return value;
        };
    });

/*
 * Copyright (c) 2013 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Converts a task status label to a human readable string.
 */
angular.module('sb.util').filter('taskPriorityLabel',
    function () {
        'use strict';

        return function (value) {

            switch( value ) {
                case 'high':
                    return 'High';
                case 'medium':
                    return 'Medium';
                case 'low':
                    return 'Low';
                default:
                    return 'Unknown Priority';
            }
        };
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A truncation filter.
 */
angular.module('sb.util').filter('truncate',
    function () {
        'use strict';

        return function (value, length) {
            if (value && value.length > length) {
                value = value.substr(0, length - 3) + '...';
            }
            return value;
        };
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Array utilities.
 */
angular.module('sb.util').factory('ArrayUtil',
    function () {
        'use strict';

        return {

            /**
             * Performs a logical intersection on two arrays. Given A, and B,
             * returns A∩B, the set of all objects that are in both A and B.
             *
             * @param A
             * @param B
             */
            intersection: function (A, B) {
                var result = [];
                A.forEach(function (item) {
                    if (B.indexOf(item) > -1) {
                        result.push(item);
                    }
                });

                return result;
            },

            /**
             * Performs a logical difference operation on the two
             * arrays. Given sets U and A it will return U\A, the set of all
             * members of U that are not members of A.
             *
             * @param U
             * @param A
             */
            difference: function (U, A) {
                var result = [];
                U.forEach(function (item) {
                    if (A.indexOf(item) === -1) {
                        result.push(item);
                    }
                });

                return result;
            }
        };
    }
);

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A collection of date utilities.
 *
 * @author Yolanda Robla
 */
angular.module('sb.util').factory('DateUtil',
    function () {
        'use strict';

        return {

            /**
             * Helper to check if a date needs to be formatted using
             * TimeAgo plugion, or displaying UTC date
             *
             * @param date The date to be checked.
             * @returns {boolean} True if time ago needs to be used.
             */
            needsTimeAgo: function (targetDate) {
                if (targetDate) {
                    var currentDate = new Date().getTime();
                    var daydiff = (currentDate - Date.parse(targetDate)) /
                        (1000 * 60 * 60 * 24);
                    return (daydiff < 1 && daydiff >= 0);
                }
                else {
                    return true;
                }
            }
        };
    }
);

/*
 * Copyright (c) 2014 Mirantis Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/*
 * A collection of string utilities.
 *
 * @author Nikita Konovalov
 */

angular.module('sb.util').factory('StringUtil',
    function () {
        'use strict';

        var defaultLength = 32; // MD5 length. Seems decent.
        var alphaNumeric = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ' +
            'abcdefghijklmnopqrstuvwxyz' +
            '0123456789';

        return {
            /**
             * Helper to generate a random alphanumeric string for the state
             * parameter.
             *
             * @param length The length of the string to generate.
             * @returns {string} A random alphanumeric string.
             */
            randomAlphaNumeric: function (length) {
                return this.random(length, alphaNumeric);
            },

            /**
             * Helper to generate a random string of specified length, using a
             * provided list of characters.
             *
             * @param length The length of the string to generate.
             * @param characters The list of valid characters.
             * @returns {string} A random string composed of provided
             * characters.
             */
            random: function (length, characters) {
                length = length || defaultLength;
                characters = characters || alphaNumeric;

                var text = '';

                for (var i = 0; i < length; i++) {
                    text += characters.charAt(Math.floor(
                            Math.random() * characters.length));
                }

                return text;
            }
        };
    }
);

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * URL and location manipulation utilities.
 *
 * @author Nikita Konovalov
 */
angular.module('sb.util').factory('UrlUtil',
    function ($window) {
        'use strict';

        return {
            /**
             * Return the full URL prefix of the application, without the #!
             * component.
             */
            getFullUrlPrefix: function () {
                var origin = $window.location.origin;
                var path = $window.location.pathname;
                return origin + path;
            },

            /**
             * Build a HashBang url for this application given the provided
             * fragment.
             */
            buildApplicationUrl: function (fragment) {
                return this.getFullUrlPrefix() + '#!' + fragment;
            },

            /**
             * Serialize an object into HTTP parameters.
             */
            serializeParameters: function (params) {
                var pairs = [];
                for (var prop in params) {
                    // Filter out system params.
                    if (!params.hasOwnProperty(prop)) {
                        continue;
                    }
                    pairs.push(
                            encodeURIComponent(prop) +
                            '=' +
                            encodeURIComponent(params[prop])
                    );
                }
                return pairs.join('&');
            },


            /**
             * Deserialize URI query parameters into an object.
             */
            deserializeParameters: function (queryString) {

                var params = {};
                var queryComponents = queryString.split('&');
                for (var i = 0; i < queryComponents.length; i++) {
                    var parts = queryComponents[i].split('=');
                    var key = decodeURIComponent(parts[0]) || null;
                    var value = decodeURIComponent(parts[1]) || null;

                    if (!!key && !!value) {
                        params[key] = value;
                    }
                }
                return params;
            }
        };
    }
);

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Utility injector, injects the query parameters from the NON-hashbang URL as
 * $searchParams.
 */
angular.module('sb.util').provider('$searchParams',
    function ($windowProvider) {
        'use strict';

        var pageParams = {};

        this.extractSearchParameters = function () {
            var window = $windowProvider.$get();
            var search = window.location.search;
            if (search.charAt(0) === '?') {
                search = search.substr(1);
            }
            var queryComponents = search.split('&');
            for (var i = 0; i < queryComponents.length; i++) {
                var parts = queryComponents[i].split('=');
                var key = decodeURIComponent(parts[0]) || null;
                var value = decodeURIComponent(parts[1]) || null;

                if (!!key && !!value) {
                    pageParams[key] = value;
                }
            }
        };
        this.$get = function () {
            return angular.copy(pageParams);
        };
    })
    .config(function ($searchParamsProvider, $windowProvider) {
        'use strict';

        // Make sure we save the search parameters so they can be used later.
        $searchParamsProvider.extractSearchParameters();

        // Overwrite the URL's current state.
        var window = $windowProvider.$get();
        var url = new URL(window.location.toString());
        url.search = '';
        if (window.location.toString() !== url.toString()) {
            window.history.replaceState({},
                window.document.title,
                url.toString());
        }
    });

/*
 * Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A service that keeps track of the last page we visited.
 */
angular.module('sb.util').factory('LastLocation',
    function ($rootScope, localStorageService, $state) {
        'use strict';

        // The published API.
        return {

            /**
             * Navigate to the last recorded state.
             *
             * @param defaultStateName A fallback state.
             * @param defaultStateParams Default state parameters.
             */
            go: function (defaultStateName, defaultStateParams) {
                var last = localStorageService.get('lastLocation');
                if (!last) {
                    $state.go(defaultStateName, defaultStateParams);
                } else {
                    last = angular.fromJson(last);
                    $state.go(last.name, last.params);
                }
            },

            /**
             * onStateChange handler. Stores the next destination state, and its
             * parameters, so we can keep revisit the history after bouncing out
             * for authentication.
             *
             * @param transition The transition to record the state from.
             */
            onStateChange: function(transition) {
                if (transition.$to().name.indexOf('sb.auth') === -1) {
                    var data = {
                        'name': transition.$to().name,
                        'params': transition.params()
                    };
                    localStorageService.set('lastLocation',
                        angular.toJson(data));
                }
            }
        };
    });

/*
 * Copyright (c) 2015-2016 Codethink Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Controller for the "new worklist" modal popup.
 */
angular.module('sb.worklist').controller('AddWorklistController',
    function ($scope, $modalInstance, $state, params, redirect, Worklist) {
        'use strict';

        var blankFilter = {
            type: 'Story',
            filter_criteria: [{
                negative: false,
                field: null,
                value: null,
                title: null
            }]
        };

        /**
         * Saves the worklist.
         */
        $scope.save = function () {
            $scope.isSaving = true;
            $scope.worklist.$create(
                function (result) {
                    $scope.isSaving = false;
                    $modalInstance.close(result);
                    if (redirect) {
                        $state.go('sb.worklist.detail',
                                  {worklistID: result.id});
                    }
                }
            );
        };

        /**
         * Close this modal without saving.
         */
        $scope.close = function () {
            $modalInstance.dismiss('cancel');
        };

        $scope.setType = function(type) {
            $scope.newFilter.type = type;
            $scope.resourceTypes = [type];
            $scope.$broadcast('refresh-types');
        };

        $scope.setCriterion = function(criterion, tag) {
            criterion.field = tag.type;
            criterion.value = tag.value.toString();
            criterion.title = tag.title;
        };

        $scope.addCriterion = function(filter) {
            filter.filter_criteria.push({
                negative: false,
                field: null,
                value: null,
                title: null
            });
        };

        $scope.removeTag = function(criterion) {
            if ($scope.newFilter.filter_criteria.length > 1) {
                var idx = $scope.newFilter.filter_criteria.indexOf(criterion);
                $scope.newFilter.filter_criteria.splice(idx, 1);
            } else {
                criterion.field = null;
                criterion.value = null;
                criterion.title = null;
            }
        };

        $scope.checkNewFilter = function() {
            var valid = true;
            angular.forEach(
                $scope.newFilter.filter_criteria,
                function(criterion) {
                    if (criterion.field == null ||
                        criterion.value == null ||
                        criterion.title == null) {
                        valid = false;
                    }
                }
            );
            return valid;
        };

        $scope.remove = function(filter) {
            var idx = $scope.worklist.filters.indexOf(filter);
            $scope.worklist.filters.splice(idx, 1);
        };

        $scope.saveNewFilter = function() {
            var added = angular.copy($scope.newFilter);
            $scope.worklist.filters.push(added);
            $scope.showAddFilter = false;
            $scope.newFilter = angular.copy(blankFilter);
        };

        $scope.isSaving = false;
        $scope.worklist = new Worklist({title: '', filters: []});
        $scope.resourceTypes = ['Story'];
        $scope.showAddFilter = true;
        $scope.newFilter = angular.copy(blankFilter);
        $scope.modalTitle = 'New Worklist';
    });

/*
 * Copyright (c) 2015 Codethink Limited
 *
 * Licensed under the Apache License, Version 2.0 (the 'License'); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Controller for "delete worklist" modal
 */
angular.module('sb.worklist').controller('WorklistAddItemController',
    function ($log, $scope, $state, worklist, $modalInstance, Story, Task,
              NewStoryService, Criteria, Worklist, $q, valid) {
        'use strict';

        $scope.worklist = worklist;
        $scope.items = [];

        // Set our progress flags and clear previous error conditions.
        $scope.saving = false;
        $scope.loadingItems = false;
        $scope.error = {};

        $scope.newStory = function () {
            NewStoryService.showNewStoryModal();
        };

        var unstaged = function(item) {
            var accept = true;
            angular.forEach($scope.items, function(selectedItem) {
                if (!selectedItem.hasOwnProperty('value')) {
                    selectedItem.value = selectedItem.id;
                }
                if (!item.hasOwnProperty('value')) {
                    item.value = item.id;
                }

                if (selectedItem.type === item.type &&
                    selectedItem.value === item.value) {
                    accept = false;
                    item.invalid = item.type +
                                   ' is already waiting to be added.';
                }
            });
            return accept;
        };

        $scope.save = function() {
            $scope.saving = true;
            var creates = [];
            var offset = $scope.worklist.items.length;
            for (var i = 0; i < $scope.items.length; i++) {
                var item = $scope.items[i];
                var item_type = '';

                if (item.type === 'Task') {
                    item_type = 'task';
                } else if (item.type === 'Story') {
                    item_type = 'story';
                }

                if (!item.hasOwnProperty('value')) {
                    item.value = item.id;
                }

                var params = {
                    item_id: item.value,
                    id: $scope.worklist.id,
                    list_position: offset + i,
                    item_type: item_type
                };

                if (valid(item)) {
                    creates.push(
                        Worklist.ItemsController.create(params).$promise
                    );
                }
            }
            $q.all(creates).then(function() {
                $scope.saving = false;
                $modalInstance.dismiss('success');
            });
        };

        /**
         * Remove an item from the list of items to add.
         */
        $scope.removeItem = function (item) {
            var idx = $scope.items.indexOf(item);
            $scope.items.splice(idx, 1);
        };

        /**
         * Item search method.
         */
        $scope.targets = ['Stories', 'Tasks'];
        $scope.searchTarget = 'Tasks';
        $scope.searchQuery = '';
        $scope.searchItems = function (value) {
            var searchString = value || '';

            var searches = [];
            if (searchString !== '') {
                if ($scope.searchTarget === 'Stories') {
                    searches.push(Story.criteriaResolver(searchString, 50));
                } else if ($scope.searchTarget === 'Tasks') {
                    searches.push(Task.criteriaResolver(searchString, 50));
                }
            }

            $q.all(searches).then(function (searchResults) {
                var validated = [];
                var invalid = [];

                var addResult = function (item) {
                    if (valid(item) && unstaged(item)) {
                        validated.push(item);
                    } else {
                        invalid.push(item);
                    }
                };

                for (var i = 0; i < searchResults.length; i++) {
                    var results = searchResults[i];

                    if (!results) {
                        continue;
                    }

                    if (!!results.forEach) {
                        results.forEach(addResult);
                    } else {
                        addResult(results);
                    }
                }

                $scope.searchResults = validated;
                $scope.invalidSearchResults = invalid;
            });
        };

        $scope.setSearchTarget = function(target) {
            $scope.searchTarget = target;
            $scope.searchItems($scope.searchQuery);
        };

        $scope.loadTasks = function(story) {
            story.loadingTasks = true;
            Task.browse({story_id: story.value}, function(tasks) {
                var results = [];
                var invalid = [];

                angular.forEach(tasks, function(task) {
                    task.type = 'Task';
                    if (valid(task) && unstaged(task)) {
                        results.push(task);
                    } else {
                        invalid.push(task);
                    }
                });

                story.tasks = results;
                story.invalidTasks = invalid;
                story.loadingTasks = false;
            });
        };

        /**
         * Formats the item name.
         */
        $scope.formatItemName = function (model) {
            if (!!model) {
                return model.title;
            }
            return '';
        };

        /**
         * Select a new item.
         */
        $scope.selectTask = function (task, source, event) {
            event.stopPropagation();
            task.type = 'Task';
            $scope.items.push(task);
            if (source.length > 0) {
                var idx = source.indexOf(task);
                source.splice(idx, 1);
            }
        };

        $scope.selectStory = function (story, source, event) {
            event.stopPropagation();
            story.type = 'Story';
            $scope.items.push(story);
            if (source.length > 0) {
                var idx = source.indexOf(story);
                source.splice(idx, 1);
            }
        };

        $scope.close = function () {
            $modalInstance.dismiss('cancel');
        };
    });

/*
 * Copyright (c) 2015 Codethink Limited
 *
 * Licensed under the Apache License, Version 2.0 (the 'License'); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Controller for "delete worklist" modal
 */
angular.module('sb.worklist').controller('WorklistDeleteController',
    function ($log, $scope, $state, worklist, redirect, $modalInstance,
              Worklist) {
        'use strict';

        $scope.worklist = worklist;

        // Set our progress flags and clear previous error conditions.
        $scope.isUpdating = true;
        $scope.error = {};

        $scope.remove = function () {
            Worklist.delete({id: $scope.worklist.id},
                function () {
                    if (!!redirect) {
                        $state.go('sb.dashboard.boards');
                    }
                    return $modalInstance.close('success');
                }
            );
        };

        $scope.close = function () {
            $modalInstance.dismiss('cancel');
        };
    });

/*
 * Copyright (c) 2015-2016 Codethink Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A controller that manages the worklist detail page.
 */
angular.module('sb.worklist').controller('WorklistDetailController',
    function ($scope, $modal, $timeout, $stateParams, Worklist, BoardHelper,
              $document, User, $q, worklist, permissions) {
        'use strict';

        function resolvePermissions() {
            $scope.owners = [];
            $scope.users = [];
            angular.forEach($scope.worklist.owners, function(id) {
                $scope.owners.push(User.get({id: id}));
            });
            angular.forEach($scope.worklist.users, function(id) {
                $scope.users.push(User.get({id: id}));
            });
        }

        /**
         * Load the worklist and its contents.
         */
        function loadWorklist() {
            var params = {id: $stateParams.worklistID};
            Worklist.Permissions.get(params, function(perms) {
                $scope.permissions = {
                    editWorklist: perms.indexOf('edit_worklist') > -1,
                    moveItems: perms.indexOf('move_items') > -1
                };
            });
            Worklist.get(params, function(result) {
                $scope.worklist = result;
                resolvePermissions();
            });
        }

        /**
         * Save the worklist.
         */
        $scope.update = function() {
            var params = {id: $scope.worklist.id};
            var owners = {
                codename: 'edit_worklist',
                users: $scope.worklist.owners
            };
            var users = {
                codename: 'move_items',
                users: $scope.worklist.users
            };
            $scope.worklist.$update().then(function() {
                var updating = [
                    Worklist.Permissions.update(params, owners).$promise,
                    Worklist.Permissions.update(params, users).$promise
                ];
                angular.forEach($scope.worklist.filters, function(filter) {
                    var filterParams = {
                        id: $scope.worklist.id,
                        filter_id: filter.id
                    };
                    updating.push(
                        Worklist.Filters.update(filterParams, filter).$promise
                    );
                });
                $q.all(updating).then(function() {
                    $scope.toggleEditMode();
                });
            });
        };

        $scope.unarchive = function() {
            $scope.worklist.archived = false;
            $scope.worklist.$update();
        };

        /**
         * Toggle edit mode on the worklist. If going on->off then
         * save changes.
         */
        $scope.toggleEditMode = function() {
            if ($scope.editing) {
                loadWorklist();
            }
            $scope.editing = !$scope.editing;
        };

        /**
         * Show a modal to handle adding items to the worklist.
         */
        function showAddItemModal() {
            var modalInstance = $modal.open({
                size: 'lg',
                templateUrl: 'app/worklists/template/additem.html',
                backdrop: 'static',
                controller: 'WorklistAddItemController',
                resolve: {
                    worklist: function() {
                        return $scope.worklist;
                    },
                    valid: function() {
                        return function() {
                            // No limit on the contents of worklists
                            return true;
                        };
                    }
                }
            });

            return modalInstance.result;
        }

        /**
         * Display the add-item modal and reload the worklist when
         * it is closed.
         */
        $scope.addItem = function() {
            showAddItemModal().finally(loadWorklist);
        };

        /**
         * Remove an item from the worklist.
         */
        $scope.removeListItem = function(item) {
            Worklist.ItemsController.delete({
                id: $scope.worklist.id,
                item_id: item.id
            }).$promise.then(function() {
                var idx = $scope.worklist.items.indexOf(item);
                $scope.worklist.items.splice(idx, 1);
            });
        };

        /**
         * Show a modal to handle archiving the worklist.
         */
        $scope.remove = function() {
            var modalInstance = $modal.open({
                templateUrl: 'app/worklists/template/delete.html',
                backdrop: 'static',
                controller: 'WorklistDeleteController',
                resolve: {
                    worklist: function() {
                        return $scope.worklist;
                    },
                    redirect: function() {
                        return true;
                    }
                }
            });
            return modalInstance.result;
        };

        /**
         * User typeahead search method.
         */
        $scope.searchUsers = function (value, array) {
            var deferred = $q.defer();

            User.browse({full_name: value, limit: 10},
                function(searchResults) {
                    var results = [];
                    angular.forEach(searchResults, function(result) {
                        if (array.indexOf(result.id) === -1) {
                            results.push(result);
                        }
                    });
                    deferred.resolve(results);
                }
            );
            return deferred.promise;
        };

        /**
         * Formats the user name.
         */
        $scope.formatUserName = function (model) {
            if (!!model) {
                return model.name;
            }
            return '';
        };

        /**
         * Add a new user to one of the permission levels.
         */
        $scope.addUser = function (model, modelArray, idArray) {
            idArray.push(model.id);
            modelArray.push(model);
        };

        /**
         * Remove a user from one of the permission levels.
         */
        $scope.removeUser = function (model, modelArray, idArray) {
            var idIdx = idArray.indexOf(model.id);
            idArray.splice(idIdx, 1);

            var modelIdx = modelArray.indexOf(model);
            modelArray.splice(modelIdx, 1);
        };

        /**
         * Criteria editing
         */
        var blankFilter = {
            type: 'Story',
            filter_criteria: [{
                negative: false,
                field: null,
                value: null,
                title: null
            }]
        };
        $scope.setType = function(type) {
            $scope.newFilter.type = type;
            $scope.resourceTypes = [type];
            $scope.$broadcast('refresh-types');
        };

        $scope.setCriterion = function(criterion, tag) {
            criterion.field = tag.type;
            criterion.value = tag.value.toString();
            criterion.title = tag.title;
        };

        $scope.addCriterion = function(filter) {
            filter.filter_criteria.push({
                negative: false,
                field: null,
                value: null,
                title: null
            });
        };

        $scope.removeTag = function(criterion) {
            if ($scope.newFilter.filter_criteria.length > 1) {
                var idx = $scope.newFilter.filter_criteria.indexOf(criterion);
                $scope.newFilter.filter_criteria.splice(idx, 1);
            } else {
                criterion.field = null;
                criterion.value = null;
                criterion.title = null;
            }
        };

        $scope.checkNewFilter = function() {
            var valid = true;
            angular.forEach(
                $scope.newFilter.filter_criteria,
                function(criterion) {
                    if (criterion.field == null ||
                        criterion.value == null ||
                        criterion.title == null) {
                        valid = false;
                    }
                }
            );
            return valid;
        };

        $scope.removeFilter = function(filter) {
            var idx = $scope.worklist.filters.indexOf(filter);
            Worklist.Filters.delete({
                id: $scope.worklist.id,
                filter_id: filter.id
            });
            $scope.worklist.filters.splice(idx, 1);
        };

        $scope.saveNewFilter = function() {
            var added = angular.copy($scope.newFilter);
            Worklist.Filters.create(
                {id: $scope.worklist.id}, added, function(result) {
                    $scope.worklist.filters.push(result);
                }
            );
            $scope.showAddFilter = false;
            $scope.newFilter = angular.copy(blankFilter);
        };

        $scope.isSaving = false;
        $scope.resourceTypes = ['Story'];
        $scope.showAddFilter = false;
        $scope.newFilter = angular.copy(blankFilter);


        /**
         * Config for worklist sortable.
         */
        $scope.sortableOptions = {
            accept: function (sourceHandle, dest) {
                return sourceHandle.itemScope.sortableScope.$id === dest.$id;
            },
            orderChanged: function (result) {
                var list = result.dest.sortableScope.$parent.worklist;
                var position = result.dest.index;
                var item = list.items[position];

                item.list_position = position;
                Worklist.ItemsController.update({
                    id: list.id,
                    item_id: item.id,
                    list_position: item.list_position
                });
            }
        };

        /**
         * Add an event listener to prevent default dragging behaviour from
         * interfering with dragging items around.
         */
        $document[0].ondragstart = function (event) {
            event.preventDefault();
        };

        $scope.worklist = worklist;
        $scope.permissions = {
            editWorklist: permissions.indexOf('edit_worklist') > -1,
            moveItems: permissions.indexOf('move_items') > -1
        };
        resolvePermissions();
    });

/*
 * Copyright (c) 2015-2016 Codethink Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Controller for the "new worklist" modal popup.
 */
angular.module('sb.worklist').controller('WorklistEditController',
    function ($scope, $modalInstance, $state, worklist, board, Worklist) {
        'use strict';

        var blankFilter = {
            type: 'Story',
            filter_criteria: [{
                negative: false,
                field: null,
                value: null,
                title: null
            }]
        };

        /**
         * Saves the worklist.
         */
        $scope.save = function () {
            $scope.isSaving = true;
            Worklist.update($scope.worklist, function () {
                $scope.isSaving = false;
                $modalInstance.dismiss('success');
            });
        };

        /**
         * Close this modal without saving.
         */
        $scope.close = function () {
            $modalInstance.dismiss('cancel');
        };

        $scope.setType = function(type) {
            $scope.newFilter.type = type;
            $scope.resourceTypes = [type];
            $scope.$broadcast('refresh-types');
        };

        $scope.setCriterion = function(criterion, tag) {
            criterion.field = tag.type;
            criterion.value = tag.value.toString();
            criterion.title = tag.title;
        };

        $scope.addCriterion = function(filter) {
            filter.filter_criteria.push({
                negative: false,
                field: null,
                value: null,
                title: null
            });
        };

        $scope.removeTag = function(criterion) {
            if ($scope.newFilter.filter_criteria.length > 1) {
                var idx = $scope.newFilter.filter_criteria.indexOf(criterion);
                $scope.newFilter.filter_criteria.splice(idx, 1);
            } else {
                criterion.field = null;
                criterion.value = null;
                criterion.title = null;
            }
        };

        $scope.checkNewFilter = function() {
            var valid = true;
            angular.forEach(
                $scope.newFilter.filter_criteria,
                function(criterion) {
                    if (criterion.field == null ||
                        criterion.value == null ||
                        criterion.title == null) {
                        valid = false;
                    }
                }
            );
            return valid;
        };

        $scope.remove = function(filter) {
            var idx = $scope.worklist.filters.indexOf(filter);
            Worklist.Filters.delete({
                id: $scope.worklist.id,
                filter_id: filter.id
            });
            $scope.worklist.filters.splice(idx, 1);
        };

        $scope.saveNewFilter = function() {
            var added = angular.copy($scope.newFilter);
            Worklist.Filters.create(
                {id: $scope.worklist.id}, added, function(result) {
                    $scope.worklist.filters.push(result);
                }
            );
            $scope.showAddFilter = false;
            $scope.newFilter = angular.copy(blankFilter);
        };

        $scope.isSaving = false;
        $scope.worklist = worklist;
        $scope.resourceTypes = ['Story'];
        $scope.showAddFilter = false;
        $scope.newFilter = angular.copy(blankFilter);
        $scope.modalTitle = 'Edit Worklist';
    });

/*
 * Copyright (c) 2015 Codethink Limited.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */


angular.module('sb.worklist').factory('NewWorklistService',
    function ($modal, $log, Session, SessionModalService) {
        'use strict';

        return {
            showNewWorklistModal: function (userId) {
                if (!Session.isLoggedIn()) {
                    return SessionModalService.showLoginRequiredModal();
                } else {
                    var modalInstance = $modal.open(
                        {
                            size: 'lg',
                            templateUrl: 'app/worklists/template/new.html',
                            backdrop: 'static',
                            controller: 'AddWorklistController',
                            resolve: {
                                params: function () {
                                    return {
                                        userId: userId || null
                                    };
                                },
                                redirect: function() {
                                    return true;
                                }
                            }
                        }
                    );

                    // Return the modal's promise.
                    return modalInstance.result;
                }
            }
        };
    }
);
