# Copyright 2013 Red Hat, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

"""
Client side of the rock agent RPC API.
"""

from oslo.config import cfg
from oslo import messaging

from rock import exception
from rock.i18n import _, _LW
from rock import objects
from rock.objects import base as objects_base
from rock.openstack.common import jsonutils
from rock.openstack.common import log as logging
from rock import rpc
from rock import utils

rpcapi_opts = [
    cfg.StrOpt('agent_topic',
               default='agent',
               help='The topic rock-agent nodes listen on'),
]

CONF = cfg.CONF
CONF.register_opts(rpcapi_opts)

rpcapi_cap_opt = cfg.StrOpt('agent',
        help='Set a version cap for messages sent to rock services. ')
CONF.register_opt(rpcapi_cap_opt, 'upgrade_levels')

LOG = logging.getLogger(__name__)

def _rock_host(host, accelerator):
    '''Get the destination host for a message.

    :param host: explicit host to send the message to.
    :param accelerator: If an explicit host was not specified, use
                     accelerator['host']

    :returns: A host
    '''
    if host:
        return host
    if not accelerator:
        raise exception.RockException(_('No accelerator host specified'))
    if not accelerator['host']:
        raise exception.RockException(_('Unable to find host for '
                                        'accelerator %s') % accelerator['uuid'])
    return accelerator['host']

class AgentAPI(object):
    '''Client side of the rock-agent rpc API.

    API version history:

        * 1.0 - Initial version.
    '''

    def __init__(self):
        super(AgentAPI, self).__init__()
        target = messaging.Target(topic=CONF.agent_topic, version='1.0')
        serializer = objects_base.RockObjectSerializer()
        self.client = self.get_client(target, None, serializer)


    def get_client(self, target, version_cap, serializer):
        return rpc.get_client(target,
                              version_cap=version_cap,
                              serializer=serializer)

    def target_host(self, acc_list):
        target_host = None
        if isinstance(acc_list, list) or isinstance(acc_list, objects_base.ObjectListBase):
            for acc in acc_list:
                temp_host = _rock_host(None, accelerator=acc)
                if target_host and temp_host !=target_host:
                    LOG.error("request acc_list is not in the same host[%s].acc-%s" % (target_host, acc))

                else:
                    target_host = temp_host
        else:
            target_host=_rock_host(None, accelerator=acc_list)
        return target_host

    def allocate_acc(self, context, acc_list):
        version = '1.0'
        if isinstance(acc_list, dict):
            acc_list = acc_list['update_vf_list']
        cctxt = self.client.prepare(server=self.target_host(acc_list), version=version)
        return cctxt.call(context, 'allocate_acc', req_accelerator=acc_list)

    def release_acc(self, context, acc_list):
        version = '1.0'
        cctxt = self.client.prepare(server=self.target_host(acc_list),
                version=version)
        return cctxt.call(context, 'release_acc', req_accelerator=acc_list)

class SecurityGroupAPI(object):
    '''Client side of the security group rpc API.

    API version history:

        1.0 - Initial version.
    '''

    def __init__(self):
        super(SecurityGroupAPI, self).__init__()
        target = messaging.Target(topic=CONF.compute_topic, version='1.0')
        self.client = rpc.get_client(target, None)

    def refresh_security_group_rules(self, ctxt, security_group_id, host):
        version = '1.0'
        cctxt = self.client.prepare(server=host, version=version)
        cctxt.cast(ctxt, 'refresh_security_group_rules',
                   security_group_id=security_group_id)

    def refresh_security_group_members(self, ctxt, security_group_id,
            host):
        version = '1.0'
        cctxt = self.client.prepare(server=host, version=version)
        cctxt.cast(ctxt, 'refresh_security_group_members',
                   security_group_id=security_group_id)

    def refresh_instance_security_rules(self, ctxt, host, instance):
        version = '1.0'
        instance_p = jsonutils.to_primitive(instance)
        cctxt = self.client.prepare(server=_rock_host(None, instance),
                version=version)
        cctxt.cast(ctxt, 'refresh_instance_security_rules',
                   instance=instance_p)
