#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

"""Handles database requests from other rock services."""

import copy
import itertools

from oslo import messaging
import six

from rock.agent import api as agent_api
from rock.agent import rpcapi as agent_rpcapi
from rock.db import base
from rock import exception
from rock.i18n import _
from rock import manager
from rock import conductor
from rock import objects
from rock.objects import base as rock_object

from rock.openstack.common import excutils
from rock.openstack.common import jsonutils
from rock.openstack.common import log as logging

LOG = logging.getLogger(__name__)

_ALIAS_ACC_TYPE = ['ipsec', 'sa', 'gb']
_ALIAS_SCHEMA = {
    "type": "array",
    "properties": {
        "type": "object",
        "additionalProperties": False,
        "properties":{
            "acc_type": {
                "type": "string",
                "pattern": _ALIAS_ACC_TYPE,
            },
            "capabilitys": {
                "type": "array",
                "properties":{
                    "name": {
                        "type": "string",
                        "minLength": 1,
                        "maxLength": 255,
                    },
                    "num":{
                        "type": "number",
                    },
                    "unit":{
                        "type": "string",
                        "minLength": 1,
                        "maxLength": 255,
                    },
                }
            },
            "remote_disable":{
                "type": "boolean",
            }
        },
    },
    "required": ["acc_type"],
}

def _is_valid_attr(attr, attr_name):
    if not (attr and attr_name in attr):
        return False

    def is_dict(d):
        try:
            d.get(None)
            return True
        except AttributeError:
            return False

    if not is_dict(attr):
        return False

    return True
def _decode_accelerator_attr(accelerator_attr):
    acc_desc = accelerator_attr.get


class RockManager(manager.Manager):
    """Manages the running instances from creation to destruction."""

    target = messaging.Target(version='1.0')

    def __init__(self, compute_driver=None, *args, **kwargs):
        self.agent_rpcapi = agent_rpcapi.AgentAPI()
        self.conductor_api = conductor.API()

        super(RockManager, self).__init__(service_name="rock",
                                             *args, **kwargs)

    def allocate_accelerator(self, context, instance, accelerator_request):
        """Allocate accelerator to instance."""
        if accelerator_request is None:
            raise exception.AcceleratorRequestIsNone()

        LOG.info("Get request instance::::%s, accelerator==> %s" % (instance, accelerator_request))
        request_acc_list = []
        acc_capability_list = []
        for accelerator in accelerator_request:
            try:
                acc = {}
                acc['acc_type'] = accelerator.get("acc_type", None)
                if not acc['acc_type']:
                    raise exception.AccTypeNotFound()
                acc['remotable'] = accelerator.get("remotable", 0)
                acc['acc_capability'] = accelerator.get("acc_capability", {})
                acc_capability_list.append(acc['acc_capability'])
                acc['device_type'] = accelerator.get("device_type", None)
                request_acc_list.append(acc)
            except Exception as e:
                raise exception.AllocateAcceleratorInvalidRequest(reason=six.text_type(e))

        if request_acc_list:
            # result form:
            #    update_info['update_acc_board_pf_list'] = update_acc_board_pf_list
            #    update_info['update_vf_list'] = update_vf_list
            LOG.info("=========> api manager -------acceelerators--%s"%request_acc_list)
            instance_claim_result = objects.Accelerator.instance_claim(context, instance, acc_list=request_acc_list)
            LOG.info("=========> instance_claim_result--%s"%instance_claim_result)

        instance_allocate_result = None
        try:
            allocate_acc_result = None
            object_list = []
            if instance_claim_result:
                vf_list = instance_claim_result['update_vf_list']
                for index, vf in enumerate(vf_list):
                    accelerator = objects.Accelerator.get_by_dev_id(context, vf)
                    accelerator['acc_capability'] = acc_capability_list[index]
                    object_list.append(accelerator)
                allocate_acc_result = self.agent_rpcapi.allocate_acc(context, object_list)
                LOG.info("allocate result ====> %s" % allocate_acc_result)

            if allocate_acc_result:
                instance_allocate_result = objects.Accelerator.instance_allocate(context, acc_list=object_list)
                LOG.info("instance_allocate_result ====> %s" % instance_allocate_result)
        except Exception as e:
            self.detach_accelerator(context, instance, 'all')
            raise exception.AllocateAcceleratorInvalidRequest(reason=six.text_type(e))
        return instance_allocate_result


    def detach_accelerator(self, context, instance, pci_address):
        """free accelerator from instance.
        if pci_address value is 'all', free all accelerators attached in instance"""
        LOG.info("detach accelerator with instance %s and address %s" % (instance, pci_address))
        instance_uuid = instance['uuid']
        if instance_uuid is None:
            LOG.warn("detach accelerator with instance[Instance UUID is none] %s" % instance)
            return None
        acc_list = []
        if 'all' == pci_address:
            acc_list = objects.AcceleratorList.get_by_instance_uuid(context, instance_uuid)
        else:
            accelerator = objects.Accelerator.get_by_address(context, pci_address)
            if accelerator and accelerator['status'] in ['claimed', 'allocated']:
                acc_list = [accelerator]
        if acc_list:
            self.agent_rpcapi.release_acc(context, acc_list)
            acc_list = objects.AcceleratorList.release_by_uuid_address(context, instance_uuid, pci_address)
        return acc_list

