# Copyright (c) 2011 X.commerce, a business unit of eBay Inc.
# Copyright 2010 United States Government as represented by the
# Administrator of the National Aeronautics and Space Administration.
# Copyright 2011 Piston Cloud Computing, Inc.
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
"""
SQLAlchemy models for rock data.
"""

from oslo.config import cfg
from oslo.db.sqlalchemy import models
from sqlalchemy import Column, Index, Integer, BigInteger, Enum, String, schema
from sqlalchemy.dialects.mysql import MEDIUMTEXT
from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy import orm
from sqlalchemy import ForeignKey, DateTime, Boolean, Text, Float

from rock.db.sqlalchemy import types
from rock.openstack.common import timeutils

CONF = cfg.CONF
BASE = declarative_base()


def MediumText():
    return Text().with_variant(MEDIUMTEXT(), 'mysql')


class RockBase(models.SoftDeleteMixin,
               models.TimestampMixin,
               models.ModelBase):
    metadata = None

    # TODO(ekudryashova): remove this after both rock and oslo.db
    # will use oslo.utils library
    # NOTE: Both projects(rock and oslo.db) use `timeutils.utcnow`, which
    # returns specified time(if override_time is set). Time overriding is
    # only used by unit tests, but in a lot of places, temporarily overriding
    # this columns helps to avoid lots of calls of timeutils.set_override
    # from different places in unit tests.
    created_at = Column(DateTime, default=lambda: timeutils.utcnow())
    updated_at = Column(DateTime, onupdate=lambda: timeutils.utcnow())

    def save(self, session=None):
        from rock.db.sqlalchemy import api

        if session is None:
            session = api.get_session()

        super(RockBase, self).save(session=session)


class Service(BASE, RockBase):
    """Represents a running service on a host."""

    __tablename__ = 'services'
    __table_args__ = (
        schema.UniqueConstraint("host", "topic", "deleted",
                                name="uniq_services0host0topic0deleted"),
        schema.UniqueConstraint("host", "binary", "deleted",
                                name="uniq_services0host0binary0deleted")
        )

    id = Column(Integer, primary_key=True)
    host = Column(String(255))  # , ForeignKey('hosts.id'))
    binary = Column(String(255))
    topic = Column(String(255))
    report_count = Column(Integer, nullable=False, default=0)
    disabled = Column(Boolean, default=False)
    disabled_reason = Column(String(255))

class Accelerator(BASE, RockBase):
    __tablename__ = 'accelerators'
    __table_args__ = (
        Index('ix_accelerators_compute_node_id_deleted',
              'compute_node_id', 'deleted'),
        Index('ix_accelerators_instance_uuid_deleted',
              'instance_uuid', 'deleted'),
        schema.UniqueConstraint(
            "compute_node_id", "address", "deleted",
            name="uniq_accelerators0compute_node_id0address0deleted")
    )

    id = Column(Integer ,primary_key=True, autoincrement=True)

    belong_pf_id = Column(Integer, nullable=True)
    vf_number = Column(Integer, nullable=True)
    dev_id = Column(String(255), nullable=True)
    mac = Column(String(32), nullable=True)
    device_type = Column(String(12), nullable=False)
    acc_type = Column(String(8), nullable=False)
    acc_sub_type = Column(String(8), nullable=False)
    # acc_algorithm_type = Column(String(8), nullable=False)
    acc_capability = Column(Text, nullable=True)
    queue_num = Column(Integer, nullable=True)
    queue_type = Column(String(8), nullable=True)
    # physical address of device domain:bus:slot.func (0000:09:01.1)
    address = Column(String(12), nullable=False)
    vendor_id = Column(String(4), nullable=True)
    product_id = Column(String(4), nullable=True)
    function_type = Column(String(8), nullable=False)
    remotable = Column(Integer, nullable=True)
    remote_mac = Column(String(32), nullable=True)
    numa_node = Column(Integer, nullable=True)
    status = Column(String(36), nullable=False)
    belong_network = Column(String(255), nullable=True)
    config_flag = Column(Integer, nullable=True)

    # numa_topology = Column(Text)
    instance_uuid = Column(String(36))
    #belong_network_id = Column(String(36), ForeignKey())
    extra_info = Column(Text)

    compute_node_id = Column(Integer, ForeignKey('compute_nodes.id'),
                             nullable=False)
    host = Column(String(255), nullable=True)

    # the request_id is used to identify a device that is allocated for a
    # particular request
    request_id = Column(String(36), nullable=True)


class ComputeNode(BASE, RockBase):
    """Represents a running compute service on a host."""

    __tablename__ = 'compute_nodes'
    __table_args__ = ()
    id = Column(Integer, primary_key=True)
    service_id = Column(Integer, ForeignKey('services.id'), nullable=False)
    service = orm.relationship(Service,
                           backref=orm.backref('compute_node'),
                           foreign_keys=service_id,
                           primaryjoin='and_('
                                'ComputeNode.service_id == Service.id,'
                                'ComputeNode.deleted == 0)')

    hypervisor_hostname = Column(String(255))
    host_ip = Column(types.IPAddress())
    mac = Column(String(32))

    # '{"vendor_id":"8086", "product_id":"1234", "count":3 }'
    accelerator_stats = Column(Text)

