# Copyright (c) 2011 X.commerce, a business unit of eBay Inc.
# Copyright 2010 United States Government as represented by the
# Administrator of the National Aeronautics and Space Administration.
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

"""Defines interface for DB access.

Functions in this module are imported into the rock.db namespace. Call these
functions from rock.db namespace, not the rock.db.api namespace.

All functions in this module return objects that implement a dictionary-like
interface. Currently, many of these objects are sqlalchemy objects that
implement a dictionary interface. However, a future goal is to have all of
these objects be simple dictionaries.

"""

from oslo.config import cfg
from oslo.db import concurrency

from rock.i18n import _
from rock.openstack.common import log as logging


db_opts = [
    cfg.BoolOpt('enable_new_services',
                default=True,
                help='Services to be added to the available pool on create'),
]

CONF = cfg.CONF
CONF.register_opts(db_opts)

_BACKEND_MAPPING = {'sqlalchemy': 'rock.db.sqlalchemy.api'}


IMPL = concurrency.TpoolDbapiWrapper(CONF, backend_mapping=_BACKEND_MAPPING)

LOG = logging.getLogger(__name__)

# The maximum value a signed INT type may have
MAX_INT = 0x7FFFFFFF

###################


def constraint(**conditions):
    """Return a constraint object suitable for use with some updates."""
    return IMPL.constraint(**conditions)


def equal_any(*values):
    """Return an equality condition object suitable for use in a constraint.

    Equal_any conditions require that a model object's attribute equal any
    one of the given values.
    """
    return IMPL.equal_any(*values)


def not_equal(*values):
    """Return an inequality condition object suitable for use in a constraint.

    Not_equal conditions require that a model object's attribute differs from
    all of the given values.
    """
    return IMPL.not_equal(*values)


###################


def service_destroy(context, service_id):
    """Destroy the service or raise if it does not exist."""
    return IMPL.service_destroy(context, service_id)


def service_get(context, service_id, with_compute_node=False,
                use_slave=False):
    """Get a service or raise if it does not exist."""
    return IMPL.service_get(context, service_id,
                            with_compute_node=with_compute_node,
                            use_slave=use_slave)


def service_get_by_host_and_topic(context, host, topic):
    """Get a service by host it's on and topic it listens to."""
    return IMPL.service_get_by_host_and_topic(context, host, topic)


def service_get_all(context, disabled=None):
    """Get all services."""
    return IMPL.service_get_all(context, disabled)


def service_get_all_by_topic(context, topic):
    """Get all services for a given topic."""
    return IMPL.service_get_all_by_topic(context, topic)


def service_get_all_by_host(context, host):
    """Get all services for a given host."""
    return IMPL.service_get_all_by_host(context, host)


def service_get_by_compute_host(context, host, use_slave=False):
    """Get the service entry for a given compute host.

    Returns the service entry joined with the compute_node entry.
    """
    return IMPL.service_get_by_compute_host(context, host,
                                            use_slave=use_slave)


def service_get_by_args(context, host, binary):
    """Get the state of a service by node name and binary."""
    return IMPL.service_get_by_args(context, host, binary)


def service_create(context, values):
    """Create a service from the values dictionary."""
    return IMPL.service_create(context, values)


def service_update(context, service_id, values):
    """Set the given properties on a service and update it.

    Raises NotFound if service does not exist.

    """
    return IMPL.service_update(context, service_id, values)


####################


def task_log_end_task(context, task_name,
                        period_beginning,
                        period_ending,
                        host,
                        errors,
                        message=None):
    """Mark a task as complete for a given host/time period."""
    return IMPL.task_log_end_task(context, task_name,
                                  period_beginning,
                                  period_ending,
                                  host,
                                  errors,
                                  message)


def task_log_begin_task(context, task_name,
                        period_beginning,
                        period_ending,
                        host,
                        task_items=None,
                        message=None):
    """Mark a task as started for a given host/time period."""
    return IMPL.task_log_begin_task(context, task_name,
                                    period_beginning,
                                    period_ending,
                                    host,
                                    task_items,
                                    message)


def task_log_get_all(context, task_name, period_beginning,
                 period_ending, host=None, state=None):
    return IMPL.task_log_get_all(context, task_name, period_beginning,
                 period_ending, host, state)


def task_log_get(context, task_name, period_beginning,
                 period_ending, host, state=None):
    return IMPL.task_log_get(context, task_name, period_beginning,
                 period_ending, host, state)


###################


def compute_node_get(context, compute_id):
    """Get a compute node by its id.

    :param context: The security context
    :param compute_id: ID of the compute node

    :returns: Dictionary-like object containing properties of the compute node,
              including its corresponding service

    Raises ComputeHostNotFound if compute node with the given ID doesn't exist.
    """
    return IMPL.compute_node_get(context, compute_id)


def compute_node_get_by_service_id(context, service_id):
    """Get a compute node by its associated service id.

    :param context: The security context
    :param service_id: ID of the associated service

    :returns: Dictionary-like object containing properties of the compute node,
              including its corresponding service and statistics

    Raises ServiceNotFound if service with the given ID doesn't exist.
    """
    return IMPL.compute_node_get_by_service_id(context, service_id)


def compute_node_get_all(context, no_date_fields=False):
    """Get all computeNodes.

    :param context: The security context
    :param no_date_fields: If set to True, excludes 'created_at', 'updated_at',
                           'deleted_at' and 'deleted' fields from the output,
                           thus significantly reducing its size.
                           Set to False by default

    :returns: List of dictionaries each containing compute node properties,
              including corresponding service
    """
    return IMPL.compute_node_get_all(context, no_date_fields)


def compute_node_search_by_hypervisor(context, hypervisor_match):
    """Get compute nodes by hypervisor hostname.

    :param context: The security context
    :param hypervisor_match: The hypervisor hostname

    :returns: List of dictionary-like objects each containing compute node
              properties, including corresponding service
    """
    return IMPL.compute_node_search_by_hypervisor(context, hypervisor_match)


def compute_node_create(context, values):
    """Create a compute node from the values dictionary.

    :param context: The security context
    :param values: Dictionary containing compute node properties

    :returns: Dictionary-like object containing the properties of the created
              node, including its corresponding service and statistics
    """
    return IMPL.compute_node_create(context, values)


def compute_node_update(context, compute_id, values):
    """Set the given properties on a compute node and update it.

    :param context: The security context
    :param compute_id: ID of the compute node
    :param values: Dictionary containing compute node properties to be updated

    :returns: Dictionary-like object containing the properties of the updated
              compute node, including its corresponding service and statistics

    Raises ComputeHostNotFound if compute node with the given ID doesn't exist.
    """
    return IMPL.compute_node_update(context, compute_id, values)


def compute_node_delete(context, compute_id):
    """Delete a compute node from the database.

    :param context: The security context
    :param compute_id: ID of the compute node

    Raises ComputeHostNotFound if compute node with the given ID doesn't exist.
    """
    return IMPL.compute_node_delete(context, compute_id)


####################


def accelerator_add(context, values):
    """Add accelerator."""
    return IMPL.accelerator_create(context, values)

def accelerator_get_by_id(context, id):
    """Get accelerator device by id."""
    return IMPL.accelerator_get_by_id(context, id)

def accelerator_get_by_address(context, address):
    """Get accelerator device by address."""
    return IMPL.accelerator_get_by_address(context, address)

def accelerator_get_all_by_node(context, node_id):
    """Get all accelerators devices for one host."""
    return IMPL.accelerator_get_all_by_node(context, node_id)

def accelerator_get_all_by_instance_uuid(context, instance_uuid):
    """Get all accelerators devices for one instance."""
    return IMPL.accelerator_get_all_by_instance_uuid(context, instance_uuid)

def accelerator_update(context, node_id, address, value):
    """Update a pci device."""
    return IMPL.accelerator_update(context, node_id, address, value)

"""def accelerator_delete_by_id(context, id):
    return IMPL.accelerator_delete_by_id(context, id)

def accelerator_delete_by_node_id(context, node_id):
    return IMPL.accelerator_delete_by_node_id(context, node_id)"""

def accelerator_destroy(context, node_id, address):
    """Delete a accelerators record."""
    return IMPL.accelerator_destroy(context, node_id, address)

def instance_claim(context=None, instance=None, acc_list=None):
    """claim accelerators for instance"""
    return IMPL.instance_claim(context, instance, acc_list)

def instance_allocate(context=None, acc_list=None):
    return IMPL.instance_allocate(context, acc_list)

def accelerator_filter(cls, context, host_name, req_list):
    return IMPL.accelerator_filter(cls, context, host_name, req_list)

def release_by_uuid_address(context, uuid, address):
    return IMPL.release_by_uuid_address(context, uuid, address)
