# Copyright 2010 United States Government as represented by the
# Administrator of the National Aeronautics and Space Administration.
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

"""rock base exception handling.

Includes decorator for re-raising rock-type exceptions.

SHOULD include dedicated exception logging.

"""

import functools
import sys

from oslo.config import cfg
import webob.exc

from rock.i18n import _
from rock.openstack.common import excutils
from rock.openstack.common import log as logging
from rock import safe_utils

LOG = logging.getLogger(__name__)

exc_log_opts = [
    cfg.BoolOpt('fatal_exception_format_errors',
                default=False,
                help='Make exception message format errors fatal'),
]

CONF = cfg.CONF
CONF.register_opts(exc_log_opts)


class ConvertedException(webob.exc.WSGIHTTPException):
    def __init__(self, code=0, title="", explanation=""):
        self.code = code
        self.title = title
        self.explanation = explanation
        super(ConvertedException, self).__init__()


def _cleanse_dict(original):
    """Strip all admin_password, new_pass, rescue_pass keys from a dict."""
    return dict((k, v) for k, v in original.iteritems() if "_pass" not in k)


def wrap_exception(notifier=None, get_notifier=None):
    """This decorator wraps a method to catch any exceptions that may
    get thrown. It logs the exception as well as optionally sending
    it to the notification system.
    """
    def inner(f):
        def wrapped(self, context, *args, **kw):
            # Don't store self or context in the payload, it now seems to
            # contain confidential information.
            try:
                return f(self, context, *args, **kw)
            except Exception as e:
                with excutils.save_and_reraise_exception():
                    if notifier or get_notifier:
                        payload = dict(exception=e)
                        call_dict = safe_utils.getcallargs(f, context,
                                                           *args, **kw)
                        cleansed = _cleanse_dict(call_dict)
                        payload.update({'args': cleansed})

                        # If f has multiple decorators, they must use
                        # functools.wraps to ensure the name is
                        # propagated.
                        event_type = f.__name__

                        (notifier or get_notifier()).error(context,
                                                           event_type,
                                                           payload)

        return functools.wraps(f)(wrapped)
    return inner


class RockException(Exception):
    """Base Rock Exception

    To correctly use this class, inherit from it and define
    a 'msg_fmt' property. That msg_fmt will get printf'd
    with the keyword arguments provided to the constructor.

    """
    msg_fmt = _("An unknown exception occurred.")
    code = 500
    headers = {}
    safe = False

    def __init__(self, message=None, **kwargs):
        self.kwargs = kwargs

        if 'code' not in self.kwargs:
            try:
                self.kwargs['code'] = self.code
            except AttributeError:
                pass

        if not message:
            try:
                message = self.msg_fmt % kwargs

            except Exception:
                exc_info = sys.exc_info()
                # kwargs doesn't match a variable in the message
                # log the issue and the kwargs
                LOG.exception(_('Exception in string format operation'))
                for name, value in kwargs.iteritems():
                    LOG.error("%s: %s" % (name, value))    # noqa

                if CONF.fatal_exception_format_errors:
                    raise exc_info[0], exc_info[1], exc_info[2]
                else:
                    # at least get the core message out if something happened
                    message = self.msg_fmt

        super(RockException, self).__init__(message)

    def format_message(self):
        # NOTE(mrodden): use the first argument to the python Exception object
        # which should be our full RockException message, (see __init__)
        return self.args[0]


class EncryptionFailure(RockException):
    msg_fmt = _("Failed to encrypt text: %(reason)s")


class DecryptionFailure(RockException):
    msg_fmt = _("Failed to decrypt text: %(reason)s")


class RevokeCertFailure(RockException):
    msg_fmt = _("Failed to revoke certificate for %(project_id)s")

class DBNotAllowed(RockException):
    msg_fmt = _('%(binary)s attempted direct database access which is '
                'not allowed by policy')

class Forbidden(RockException):
    ec2_code = 'AuthFailure'
    msg_fmt = _("Not authorized.")
    code = 403


class AdminRequired(Forbidden):
    msg_fmt = _("User does not have admin privileges")


class PolicyNotAuthorized(Forbidden):
    msg_fmt = _("Policy doesn't allow %(action)s to be performed.")


class Invalid(RockException):
    msg_fmt = _("Unacceptable parameters.")
    code = 400


class InvalidKeypair(Invalid):
    ec2_code = 'InvalidKeyPair.Format'
    msg_fmt = _("Keypair data is invalid: %(reason)s")


class InvalidRequest(Invalid):
    msg_fmt = _("The request is invalid.")


class InvalidInput(Invalid):
    msg_fmt = _("Invalid input received: %(reason)s")


class InvalidMetadata(Invalid):
    msg_fmt = _("Invalid metadata: %(reason)s")


class InvalidMetadataSize(Invalid):
    msg_fmt = _("Invalid metadata size: %(reason)s")


class InvalidPortRange(Invalid):
    ec2_code = 'InvalidParameterValue'
    msg_fmt = _("Invalid port range %(from_port)s:%(to_port)s. %(msg)s")


class InvalidIpProtocol(Invalid):
    msg_fmt = _("Invalid IP protocol %(protocol)s.")


class InvalidContentType(Invalid):
    msg_fmt = _("Invalid content type %(content_type)s.")


class InvalidUnicodeParameter(Invalid):
    msg_fmt = _("Invalid Parameter: "
                "Unicode is not supported by the current database.")


# Cannot be templated as the error syntax varies.
# msg needs to be constructed when raised.
class InvalidParameterValue(Invalid):
    ec2_code = 'InvalidParameterValue'
    msg_fmt = _("%(err)s")


class InvalidAggregateAction(Invalid):
    msg_fmt = _("Cannot perform action '%(action)s' on aggregate "
                "%(aggregate_id)s. Reason: %(reason)s.")


class InvalidGroup(Invalid):
    msg_fmt = _("Group not valid. Reason: %(reason)s")


class InvalidSortKey(Invalid):
    msg_fmt = _("Sort key supplied was not valid.")


class InvalidStrTime(Invalid):
    msg_fmt = _("Invalid datetime string: %(reason)s")


class InstanceInvalidState(Invalid):
    msg_fmt = _("Instance %(instance_uuid)s in %(attr)s %(state)s. Cannot "
                "%(method)s while the instance is in this state.")


class ServiceUnavailable(Invalid):
    msg_fmt = _("Service is unavailable at this time.")


class ComputeResourcesUnavailable(ServiceUnavailable):
    msg_fmt = _("Insufficient compute resources: %(reason)s.")


class HypervisorUnavailable(RockException):
    msg_fmt = _("Connection to the hypervisor is broken on host: %(host)s")


class ComputeServiceUnavailable(ServiceUnavailable):
    msg_fmt = _("Compute service of %(host)s is unavailable at this time.")


class ComputeServiceInUse(RockException):
    msg_fmt = _("Compute service of %(host)s is still in use.")


class InvalidUUID(Invalid):
    msg_fmt = _("Expected a uuid but received %(uuid)s.")


class InvalidID(Invalid):
    msg_fmt = _("Invalid ID received %(id)s.")


class ConstraintNotMet(RockException):
    msg_fmt = _("Constraint not met.")
    code = 412


class NotFound(RockException):
    msg_fmt = _("Resource could not be found.")
    code = 404


class KeypairNotFound(NotFound):
    ec2_code = 'InvalidKeyPair.NotFound'
    msg_fmt = _("Keypair %(name)s not found for user %(user_id)s")


class ServiceNotFound(NotFound):
    msg_fmt = _("Service %(service_id)s could not be found.")


class ServiceBinaryExists(RockException):
    msg_fmt = _("Service with host %(host)s binary %(binary)s exists.")


class ServiceTopicExists(RockException):
    msg_fmt = _("Service with host %(host)s topic %(topic)s exists.")


class HostNotFound(NotFound):
    msg_fmt = _("Host %(host)s could not be found.")


class ComputeHostNotFound(HostNotFound):
    msg_fmt = _("Compute host %(host)s could not be found.")


class ComputeHostNotCreated(HostNotFound):
    msg_fmt = _("Compute host %(name)s needs to be created first"
                " before updating.")


class HostBinaryNotFound(NotFound):
    msg_fmt = _("Could not find binary %(binary)s on host %(host)s.")


class InvalidReservationExpiration(Invalid):
    msg_fmt = _("Invalid reservation expiration %(expire)s.")


class UnsupportedObjectError(RockException):
    msg_fmt = _('Unsupported object type %(objtype)s')


class OrphanedObjectError(RockException):
    msg_fmt = _('Cannot call %(method)s on orphaned %(objtype)s object')


class IncompatibleObjectVersion(RockException):
    msg_fmt = _('Version %(objver)s of %(objname)s is not supported')


class ReadOnlyFieldError(RockException):
    msg_fmt = _('Cannot modify readonly field %(field)s')


class ObjectActionError(RockException):
    msg_fmt = _('Object action %(action)s failed because: %(reason)s')


class ObjectFieldInvalid(RockException):
    msg_fmt = _('Field %(field)s of %(objname)s is not an instance of Field')


class CoreAPIMissing(RockException):
    msg_fmt = _("Core API extensions are missing: %(missing_apis)s")

class InstanceNotFound(NotFound):
    ec2_code = 'InvalidInstanceID.NotFound'
    msg_fmt = _("Instance %(instance_id)s could not be found.")

class UnexpectedTaskStateError(RockException):
    msg_fmt = _("Unexpected task state: expecting %(expected)s but "
                "the actual state is %(actual)s")

class TaskAlreadyRunning(RockException):
    msg_fmt = _("Task %(task_name)s is already running on host %(host)s")


class TaskNotRunning(RockException):
    msg_fmt = _("Task %(task_name)s is not running on host %(host)s")

class AcceleratorNotFoundById(NotFound):
    msg_fmt = _("Accelerator device %(id)s not found")

class AcceleratorNotFoundByAddress(NotFound):
    msg_fmt = _("Accelerator device %(address)s not found")

class AcceleratorNotFound(RockException):
    msg_fmt = _("Accelerator Device %(node_id)s %(address)s not found.")

class AcceleratorDeviceInvalidStatus(RockException):
    msg_fmt = _(
        "Accelerator device %(compute_node_id)s:%(address)s is %(status)s "
        "instead of %(hopestatus)s")

class AcceleratorDeviceInvalidOwner(RockException):
    msg_fmt = _(
        "Accelerator device %(compute_node_id)s:%(address)s is owned by %(owner)s "
        "instead of %(hopeowner)s")

class AcceleratorDeviceRequestFailed(RockException):
    msg_fmt = _(
        "Accelerator device request (%requests)s failed")

class AcceleratorDevicePoolEmpty(RockException):
    msg_fmt = _(
        "Attempt to consume Accelerator device %(compute_node_id)s:%(address)s "
        "from empty pool")

class InstanceIsLocked(InstanceInvalidState):
    msg_fmt = _("Instance %(instance_uuid)s is locked")

class AccTypeNotFound(RockException):
    msg_fmt = _("accelerator_request error, acc_type is none")

class AllocateAcceleratorInvalidRequest(RockException):
    msg_fmt = _("Invalid accelerator request definition: %(reason)s")

class InstanceNodeNameIsNone(RockException):
    msg_fmt = _("Instance node name is none: %(instance_uuid)s")

class InstanceUUIDIsNone(RockException):
    msg_fmt = _("Instance UUID is none")

class InstanceHostIsNone(RockException):
    msg_fmt = _("Instance host is none: %(instance_uuid)s")

class ComputeNodeNotFoundByNode(RockException):
    msg_fmt = _("Compute node not found by node_name:%(node_name)s, host:%(host)s")

class GetAvailableSubTypeVFFailed(RockException):
    msg_fmt = _("Instance claim get available vf failed by compute_node_id:%(compute_node_id)s, "
                "acc_sub_type:%(acc_sub_type)s")

class GetAvailableSubTypePFFailed(RockException):
    msg_fmt = _("Instance claim  get available pf failed by compute_node_id:%(compute_node_id)s, "
                "acc_sub_type:%(acc_sub_type)s")

class InstanceClaimFailed(RockException):
    msg_fmt = _("Instance claim failed by compute_node_id:%(compute_node_id)s, "
                "acc_sub_type:%(acc_sub_type)s")

class UpdatePFCapabilityFailed(RockException):
    msg_fmt = _("Instance claim  update pf capability failed by compute_node_id:%(compute_node_id)s, "
                "pf id:%(id)s")

class UpdateVFClaimFailed(RockException):
    msg_fmt = _("Instance claim  update vf failed by compute_node_id:%(compute_node_id)s, "
                "vf id:%(id)s")

class UpdateVFNoneClaimFailed(RockException):
    msg_fmt = _("Instance claim  update vf failed . VF info is none:%(compute_node_id)s")

class AcceleratorRequestIsNone(RockException):
    msg_fmt = _("Instance claim  accelerator request is none failed .")

class AcceleratorDriverFailed(RockException):
    msg_fmt = _("Get accelerator driver failed.Can't find driver for device type: %(dev_type)s")

class AcceleratorStateFailed(RockException):
    msg_fmt = _("Get resource device state for accelerator is failed. Accelerator-id:%(accelerator_id)s")

class AcceleratorCapInsufficient(RockException):
    msg_fmt = _("LocalHost capability %(pool)s is insufficient for the request %(req_total)s "
                "with accelerator list:%(req_list)s")

class AcceleratorPoolFailed(RockException):
    msg_fmt = _(
        "LocalHost capability %(pool)s is insufficient for request capability with capability item::%(capability)s")

class AcceleratorNumFailed(RockException):
    msg_fmt = _(
        "LocalHost capability free-vf-num %(pool)s is not enough for request accelerator")


class BadRequest(RockException):
    message = _('Bad %(resource)s request: %(msg)s')


class Conflict(RockException):
    pass


class NotAuthorized(RockException):
    message = _("Not authorized.")


class InUse(RockException):
    message = _("The resource is inuse")