# Copyright 2013 Intel Corporation
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

from rock import db
from rock import objects
from rock.objects import base
from rock.objects import fields
from rock.openstack.common import jsonutils
from rock.openstack.common import log as logging
from rock import utils


LOG = logging.getLogger(__name__)


class Accelerator(base.RockPersistentObject, base.RockObject):

    """Object to represent a accelerator device on a compute node.

    Accelerator devices are managed by the compute resource tracker, which discovers
    the devices from the hardware platform, claims, allocates and frees
    devices for instances.

    The Accelerator device information is permanently maintained in a database.
    This makes it convenient to get Accelerator device information, like physical
    function for a VF device, adjacent switch IP address for a NIC,
    hypervisor identification for a Accelerator device, etc. It also provides a
    convenient way to check device allocation information for administrator
    purposes.

    A device can be in available/claimed/allocated/deleted/removed state.

    A device is available when it is discovered..

    A device is claimed prior to being allocated to an instance. Normally the
    transition from claimed to allocated is quick. However, during a resize
    operation the transition can take longer, because devices are claimed in
    prep_resize and allocated in finish_resize.

    A device becomes removed when hot removed from a node (i.e. not found in
    the next auto-discover) but not yet synced with the DB. A removed device
    should not be allocated to any instance, and once deleted from the DB,
    the device object is changed to deleted state and no longer synced with
    the DB.

    Filed notes::

        | 'dev_id':
        |   Hypervisor's identification for the device, the string format
        |   is hypervisor specific
        | 'extra_info':
        |   Device-specific properties like PF address, switch ip address etc.

    """

    VERSION = '1.0'

    fields = {
        'id': fields.IntegerField(),
        'belong_pf_id': fields.IntegerField(nullable=True),
        'vf_number': fields.IntegerField(nullable=True),  # valid in pf. capability of vf maybe empty
        # Note: the compute_node_id may be None because the accelerators
        # device objects are created before the compute node is created in DB
        'dev_id': fields.StringField(nullable=True),         # reserve
        'mac': fields.StringField(nullable=True),
        'device_type': fields.StringField(),                # ACC_BOARD, QAT_device , GPU
        'acc_type': fields.StringField(),                   # EMPTY, CRYPTO, VTC, DC
        'acc_sub_type': fields.StringField(),              # CRYPTO include sa, ipsec, gb
        # ["count": "2", "algorithm0": {"name":"aes", "Mpps":1024,"Mbps":10},"algorithm1": {"name":"3des", "Mpps":10240,"Mbps":10}]
        # [{"name":"aes", "Mpps":1024,"Mbps":10},{"name":"3des", "Mpps":10240,"Mbps":10}]
        'acc_capability': fields.DictOfStringsField(nullable=True),
        'queue_num': fields.StringField(nullable=True),      # reserve
        'queue_type': fields.StringField(nullable=True),     # reserve
        'address': fields.StringField(),                     # pci_address    domain:bus:device.function  0000:05:00.1
        #'pf_address': fields.StringField(nullable=True),
        'vendor_id': fields.StringField(nullable=True),      # reserve
        'product_id': fields.StringField(nullable=True),     # reserve
        'function_type': fields.StringField(),               # type-VF or type-PF
        'remotable': fields.IntegerField(nullable=True),     # 0:local;  1:local or remote; 2:only remote
        'remote_mac': fields.StringField(nullable=True),     # remote vf mac
        'numa_node':fields.IntegerField(nullable=True),      # 0 or 1 or ...
        'status': fields.StringField(),                       # invalid, available, claimed , allocated.  device is valid when the info is unanimous  form hpi and nova
        'instance_uuid': fields.StringField(nullable=True),
        'belong_network': fields.StringField(nullable=True),
        'extra_info': fields.DictOfStringsField(nullable=True),    # {"key": "value"}
        'compute_node_id': fields.IntegerField(nullable=False),
        'host': fields.StringField(nullable=True),
        'config_flag': fields.IntegerField(nullable=True),         # 0 or 1
        'request_id': fields.StringField(nullable=True),           # reserve
    }

    def obj_make_compatible(self, primitive, target_version):
        target_version = utils.convert_version_to_tuple(target_version)
        if target_version < (1, 2) and 'request_id' in primitive:
            del primitive['request_id']

    def update_device(self, dev_dict):
        """Sync the content from device dictionary to device object.

        The resource tracker updates the available devices periodically.
        To avoid meaningless syncs with the database, we update the device
        object only if a value changed.
        """

        # Note: status/instance_uuid should only be updated by
        # functions like claim/allocate etc. The id is allocated by
        # database. The extra_info is created by the object.
        no_changes = ('status', 'instance_uuid', 'id', 'extra_info')
        map(lambda x: dev_dict.pop(x, None),
            [key for key in no_changes])

        for k, v in dev_dict.items():
            if k in self.fields.keys():
                if ('acc_capability' == k):
                    temp_capability = self.acc_capability
                    for key, element in v.items():
                        if isinstance(element, str):
                            temp_capability[key] = element
                        else:
                            temp_capability[key] = jsonutils.dumps(element)
                    self.acc_capability = temp_capability
                    LOG.info('======acc_capability====> %s' % self.acc_capability)
                else:
                    self[k] = v
                    LOG.info('======key====> %s -----%s' % (k, v))
            else:
                # Note: extra_info.update does not update
                # obj_what_changed, set it explicitely
                extra_info = self.extra_info
                extra_info.update({k: v})
                LOG.info('====extra_info======> %s ' % extra_info)
                LOG.info('====extra_info======> %s -----%s' % (k, v))
                self.extra_info = extra_info
                LOG.info('====extra_info======> %s ' % extra_info)

    def __init__(self, *args, **kwargs):
        super(Accelerator, self).__init__(*args, **kwargs)
        self.obj_reset_changes()
        self.acc_capability = {}
        self.extra_info = {}

    @staticmethod
    def _from_db_object(context, accelerator, db_dev):
        for key in accelerator.fields:
            if key == 'extra_info':
                extra_info = db_dev.get("extra_info")
                accelerator.extra_info = jsonutils.loads(extra_info)
            elif key == 'acc_capability':
                db_capability = db_dev.get("acc_capability")
                acc_capability = jsonutils.loads(db_capability)
                accelerator.acc_capability = acc_capability
            else:
                accelerator[key] = db_dev[key]
        accelerator._context = context
        accelerator.obj_reset_changes()
        return accelerator

    # @base.remotable_classmethod
    # def get_by_dev_addr(cls, context, compute_node_id, dev_addr):
    #     db_dev = db.accelerator_get_by_address(
    #         context, compute_node_id, dev_addr)
    #     return cls._from_db_object(context, cls(), db_dev)

    @base.remotable_classmethod
    def get_by_address(cls, context, address):
        db_dev = db.accelerator_get_by_address(context, address)
        return cls._from_db_object(context, cls(), db_dev)

    @base.remotable_classmethod
    def get_by_dev_id(cls, context, id):
        db_dev = db.accelerator_get_by_id(context, id)
        return cls._from_db_object(context, cls(), db_dev)

    @classmethod
    def create(cls, dev_dict):
        """Create a Accelerator device based on hypervisor information.

        As the device object is just created and is not synced with db yet
        thus we should not reset changes here for fields from dict.
        """
        accelerator = cls()
        accelerator.update_device(dev_dict)
        accelerator.status = 'available'
        return accelerator

    @base.remotable
    def save(self, context):
        if self.status == 'removed':
            self.status = 'deleted'
            db.accelerator_destroy(context, self.compute_node_id, self.address)
        elif self.status != 'deleted':
            updates = self.obj_get_changes()
            if 'extra_info' in updates:
                updates['extra_info'] = jsonutils.dumps(updates['extra_info'])
            if 'acc_capability' in updates:
                updates['acc_capability'] = jsonutils.dumps(updates['acc_capability'])
            if updates:
                db_acc = db.accelerator_update(context, self.compute_node_id,
                                              self.address, updates)
                self._from_db_object(context, self, db_acc)

    @base.remotable_classmethod
    def _instance_claim(cls, context, instance, acc_list=None):
        result = db.instance_claim(context=context, instance=instance, acc_list=acc_list)
        return result

    @classmethod
    def instance_claim(cls, context, instance, acc_list=None):
        return cls._instance_claim(context, instance, acc_list=acc_list)

    @base.remotable_classmethod
    def _instance_allocate(cls, context, acc_list=None):
        result = db.instance_allocate(context=context, acc_list=acc_list)
        return result

    @classmethod
    def instance_allocate(cls, context, acc_list):
        return cls._instance_allocate(context, acc_list=acc_list)

class AcceleratorList(base.ObjectListBase, base.RockObject):

    VERSION = '1.0'

    fields = {
        'objects': fields.ListOfObjectsField('Accelerator'),
        }

    def __init__(self, *args, **kwargs):
        super(AcceleratorList, self).__init__(*args, **kwargs)
        self.objects = []
        self.obj_reset_changes()

    @base.remotable_classmethod
    def get_by_compute_node(cls, context, node_id):
        db_dev_list = db.accelerator_get_all_by_node(context, node_id)
        return base.obj_make_list(context, cls(context), objects.Accelerator,
                                  db_dev_list)

    @base.remotable_classmethod
    def get_by_instance_uuid(cls, context, uuid):
        db_dev_list = db.accelerator_get_all_by_instance_uuid(context, uuid)
        return base.obj_make_list(context, cls(context), objects.Accelerator,
                                  db_dev_list)

    @base.remotable_classmethod
    def release_by_uuid_address(cls, context, uuid, address):
        db_dev_list = db.release_by_uuid_address(context, uuid, address)
        return base.obj_make_list(context, cls(context), objects.Accelerator, db_dev_list)
