# vim: tabstop=4 shiftwidth=4 softtabstop=4

# Copyright 2012 OpenStack LLC
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.
import re

from keystone.common import logging
from keystone import config


CONF = config.CONF
LOG = logging.getLogger(__name__)


class Error(StandardError):
    """Base error class.

    Child classes should define an HTTP status code, title, and a doc string.

    """
    code = None
    title = None

    def __init__(self, message=None, **kwargs):
        """Use the doc string as the error message by default."""

        try:
            message = self._build_message(message, **kwargs)
        except KeyError:
            # if you see this warning in your logs, please raise a bug report
            LOG.warning('missing expected exception kwargs (programmer error)')
            message = self.__doc__

        super(Error, self).__init__(message)

    def _build_message(self, message, **kwargs):
        """Builds and returns an exception message.

        :raises: KeyError given insufficient kwargs

        """
        return message or self.__doc__ % kwargs

    def __str__(self):
        """Cleans up line breaks and indentation from doc strings."""
        string = super(Error, self).__str__()
        string = re.sub('[ \n]+', ' ', string)
        string = string.strip()
        return string


class ValidationError(Error):
    """Expecting to find %(attribute)s in %(target)s.

    The server could not comply with the request since it is either malformed
    or otherwise incorrect.

    The client is assumed to be in error.

    """
    code = 400
    title = 'Bad Request'


class SecurityError(Error):
    """Avoids exposing details of security failures, unless in debug mode."""

    def _build_message(self, message, **kwargs):
        """Only returns detailed messages in debug mode."""
        if CONF.debug:
            return message or self.__doc__ % kwargs
        else:
            return self.__doc__ % kwargs


class Unauthorized(SecurityError):
    """The request you have made requires authentication."""
    code = 401
    title = 'Not Authorized'


class Forbidden(SecurityError):
    """You are not authorized to perform the requested action."""
    code = 403
    title = 'Not Authorized'


class ForbiddenAction(Forbidden):
    """You are not authorized to perform the requested action: %(action)s"""


class NotFound(Error):
    """Could not find: %(target)s"""
    code = 404
    title = 'Not Found'


class EndpointNotFound(NotFound):
    """Could not find endpoint: %(endpoint_id)s"""


class MetadataNotFound(NotFound):
    """An unhandled exception has occurred: Could not find metadata."""
    # (dolph): metadata is not a user-facing concept,
    #          so this exception should not be exposed


class PolicyNotFound(NotFound):
    """Could not find policy: %(policy_id)s"""


class RoleNotFound(NotFound):
    """Could not find role: %(role_id)s"""


class ServiceNotFound(NotFound):
    """Could not find service: %(service_id)s"""


class DomainNotFound(NotFound):
    """Could not find domain: %(domain_id)s"""


class TenantNotFound(NotFound):
    """Could not find tenant: %(tenant_id)s"""


class ProjectNotFound(TenantNotFound):
    """Could not find project: %(project_id)s"""


class TokenNotFound(NotFound):
    """Could not find token: %(token_id)s"""


class UserNotFound(NotFound):
    """Could not find user: %(user_id)s"""


class Conflict(Error):
    """Conflict occurred attempting to store %(type)s.

    %(details)s

    """
    code = 409
    title = 'Conflict'


class UnexpectedError(Error):
    """An unexpected error prevented the server from fulfilling your request.

    %(exception)s

    """
    code = 500
    title = 'Internal Server Error'


class MalformedEndpoint(UnexpectedError):
    """Malformed endpoint URL (see ERROR log for details): %(endpoint)s"""


class NotImplemented(Error):
    """The action you have requested has not been implemented."""
    code = 501
    title = 'Not Implemented'
