# vim: tabstop=4 shiftwidth=4 softtabstop=4

# Copyright 2011 OpenStack LLC.
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
"""
Tests For Chance Scheduler.
"""

import random

import mox

from nova.compute import rpcapi as compute_rpcapi
from nova import context
from nova import exception
from nova.scheduler import chance
from nova.scheduler import driver
from nova.tests.scheduler import test_scheduler


class ChanceSchedulerTestCase(test_scheduler.SchedulerTestCase):
    """Test case for Chance Scheduler."""

    driver_cls = chance.ChanceScheduler

    def test_filter_hosts_avoid(self):
        """Test to make sure _filter_hosts() filters original hosts if
        avoid_original_host is True."""

        hosts = ['host1', 'host2', 'host3']
        request_spec = dict(instance_properties=dict(host='host2'))
        filter_properties = {'ignore_hosts': ['host2']}

        filtered = self.driver._filter_hosts(request_spec, hosts,
                filter_properties=filter_properties)
        self.assertEqual(filtered, ['host1', 'host3'])

    def test_filter_hosts_no_avoid(self):
        """Test to make sure _filter_hosts() does not filter original
        hosts if avoid_original_host is False."""

        hosts = ['host1', 'host2', 'host3']
        request_spec = dict(instance_properties=dict(host='host2'))
        filter_properties = {'ignore_hosts': []}

        filtered = self.driver._filter_hosts(request_spec, hosts,
                filter_properties=filter_properties)
        self.assertEqual(filtered, hosts)

    def test_basic_schedule_run_instance(self):
        ctxt = context.RequestContext('fake', 'fake', False)
        ctxt_elevated = 'fake-context-elevated'
        fake_args = (1, 2, 3)
        instance_opts = {'fake_opt1': 'meow'}
        request_spec = {'num_instances': 2,
                        'instance_properties': instance_opts}
        instance1 = {'uuid': 'fake-uuid1'}
        instance2 = {'uuid': 'fake-uuid2'}
        instance1_encoded = {'uuid': 'fake-uuid1', '_is_precooked': False}
        instance2_encoded = {'uuid': 'fake-uuid2', '_is_precooked': False}
        reservations = ['resv1', 'resv2']

        # create_instance_db_entry() usually does this, but we're
        # stubbing it.
        def _add_uuid1(ctxt, request_spec, reservations):
            request_spec['instance_properties']['uuid'] = 'fake-uuid1'

        def _add_uuid2(ctxt, request_spec, reservations):
            request_spec['instance_properties']['uuid'] = 'fake-uuid2'

        self.mox.StubOutWithMock(ctxt, 'elevated')
        self.mox.StubOutWithMock(self.driver, 'hosts_up')
        self.mox.StubOutWithMock(random, 'random')
        self.mox.StubOutWithMock(self.driver, 'create_instance_db_entry')
        self.mox.StubOutWithMock(driver, 'encode_instance')
        self.mox.StubOutWithMock(driver, 'instance_update_db')
        self.mox.StubOutWithMock(compute_rpcapi.ComputeAPI, 'run_instance')

        ctxt.elevated().AndReturn(ctxt_elevated)
        # instance 1
        self.driver.hosts_up(ctxt_elevated, 'compute').AndReturn(
                ['host1', 'host2', 'host3', 'host4'])
        random.random().AndReturn(.5)
        self.driver.create_instance_db_entry(ctxt, request_spec,
                reservations).WithSideEffects(_add_uuid1).AndReturn(
                instance1)
        driver.instance_update_db(ctxt, instance1['uuid'],
                'host3').AndReturn(instance1)
        compute_rpcapi.ComputeAPI.run_instance(ctxt, host='host3',
                instance=instance1, requested_networks=None,
                injected_files=None, admin_password=None, is_first_time=None,
                request_spec=request_spec, filter_properties={})

        driver.encode_instance(instance1).AndReturn(instance1_encoded)
        # instance 2
        ctxt.elevated().AndReturn(ctxt_elevated)
        self.driver.hosts_up(ctxt_elevated, 'compute').AndReturn(
                ['host1', 'host2', 'host3', 'host4'])
        random.random().AndReturn(.2)
        self.driver.create_instance_db_entry(ctxt, request_spec,
                reservations).WithSideEffects(_add_uuid2).AndReturn(
                instance2)
        driver.instance_update_db(ctxt, instance2['uuid'],
                'host1').AndReturn(instance2)
        compute_rpcapi.ComputeAPI.run_instance(ctxt, host='host1',
                instance=instance2, requested_networks=None,
                injected_files=None, admin_password=None, is_first_time=None,
                request_spec=request_spec, filter_properties={})

        driver.encode_instance(instance2).AndReturn(instance2_encoded)

        self.mox.ReplayAll()
        result = self.driver.schedule_run_instance(ctxt, request_spec,
                None, None, None, None, {}, reservations)
        expected = [instance1_encoded, instance2_encoded]
        self.assertEqual(result, expected)

    def test_scheduler_includes_launch_index(self):
        ctxt = context.RequestContext('fake', 'fake', False)
        instance_opts = {'fake_opt1': 'meow'}
        request_spec = {'num_instances': 2,
                        'instance_properties': instance_opts}
        instance1 = {'uuid': 'fake-uuid1'}
        instance2 = {'uuid': 'fake-uuid2'}

        # create_instance_db_entry() usually does this, but we're
        # stubbing it.
        def _add_uuid(num):
            """Return a function that adds the provided uuid number."""
            def _add_uuid_num(_, spec, reservations):
                spec['instance_properties']['uuid'] = 'fake-uuid%d' % num
            return _add_uuid_num

        def _has_launch_index(expected_index):
            """Return a function that verifies the expected index."""
            def _check_launch_index(value):
                if 'instance_properties' in value:
                    if 'launch_index' in value['instance_properties']:
                        index = value['instance_properties']['launch_index']
                        if index == expected_index:
                            return True
                return False
            return _check_launch_index

        self.mox.StubOutWithMock(self.driver, '_schedule')
        self.mox.StubOutWithMock(self.driver, 'create_instance_db_entry')
        self.mox.StubOutWithMock(driver, 'encode_instance')
        self.mox.StubOutWithMock(driver, 'instance_update_db')
        self.mox.StubOutWithMock(compute_rpcapi.ComputeAPI, 'run_instance')

        # instance 1
        self.driver._schedule(ctxt, 'compute', request_spec,
                              {}).AndReturn('host')
        self.driver.create_instance_db_entry(
            ctxt, mox.Func(_has_launch_index(0)), None
            ).WithSideEffects(_add_uuid(1)).AndReturn(instance1)
        driver.instance_update_db(ctxt, instance1['uuid'],
                'host').AndReturn(instance1)
        compute_rpcapi.ComputeAPI.run_instance(ctxt, host='host',
                instance=instance1, requested_networks=None,
                injected_files=None, admin_password=None, is_first_time=None,
                request_spec=request_spec, filter_properties={})
        driver.encode_instance(instance1).AndReturn(instance1)
        # instance 2
        self.driver._schedule(ctxt, 'compute', request_spec,
                              {}).AndReturn('host')
        self.driver.create_instance_db_entry(
            ctxt, mox.Func(_has_launch_index(1)), None
            ).WithSideEffects(_add_uuid(2)).AndReturn(instance2)
        driver.instance_update_db(ctxt, instance2['uuid'],
                'host').AndReturn(instance2)
        compute_rpcapi.ComputeAPI.run_instance(ctxt, host='host',
                instance=instance2, requested_networks=None,
                injected_files=None, admin_password=None, is_first_time=None,
                request_spec=request_spec, filter_properties={})
        driver.encode_instance(instance2).AndReturn(instance2)
        self.mox.ReplayAll()

        self.driver.schedule_run_instance(ctxt, request_spec, None, None,
                None, None, {}, None)

    def test_basic_schedule_run_instance_no_hosts(self):
        ctxt = context.RequestContext('fake', 'fake', False)
        ctxt_elevated = 'fake-context-elevated'
        fake_args = (1, 2, 3)
        instance_opts = 'fake_instance_opts'
        request_spec = {'num_instances': 2,
                        'instance_properties': instance_opts}

        self.mox.StubOutWithMock(ctxt, 'elevated')
        self.mox.StubOutWithMock(self.driver, 'hosts_up')

        # instance 1
        ctxt.elevated().AndReturn(ctxt_elevated)
        self.driver.hosts_up(ctxt_elevated, 'compute').AndReturn([])

        self.mox.ReplayAll()
        self.assertRaises(exception.NoValidHost,
                self.driver.schedule_run_instance, ctxt, request_spec,
                None, None, None, None, {}, None)

    def test_basic_schedule_fallback(self):
        ctxt = context.RequestContext('fake', 'fake', False)
        ctxt_elevated = 'fake-context-elevated'
        topic = 'fake_topic'
        method = 'fake_method'
        fake_args = (1, 2, 3)
        fake_kwargs = {'fake_kwarg1': 'fake_value1',
                       'fake_kwarg2': 'fake_value2'}

        self.mox.StubOutWithMock(ctxt, 'elevated')
        self.mox.StubOutWithMock(self.driver, 'hosts_up')
        self.mox.StubOutWithMock(random, 'random')
        self.mox.StubOutWithMock(driver, 'cast_to_host')

        ctxt.elevated().AndReturn(ctxt_elevated)
        self.driver.hosts_up(ctxt_elevated, topic).AndReturn(
                ['host1', 'host2', 'host3', 'host4'])
        random.random().AndReturn(.5)
        driver.cast_to_host(ctxt, topic, 'host3', method, **fake_kwargs)

        self.mox.ReplayAll()
        self.driver.schedule(ctxt, topic, method, *fake_args, **fake_kwargs)

    def test_basic_schedule_fallback_no_hosts(self):
        ctxt = context.RequestContext('fake', 'fake', False)
        ctxt_elevated = 'fake-context-elevated'
        topic = 'fake_topic'
        method = 'fake_method'
        fake_args = (1, 2, 3)
        fake_kwargs = {'fake_kwarg1': 'fake_value1',
                       'fake_kwarg2': 'fake_value2'}

        self.mox.StubOutWithMock(ctxt, 'elevated')
        self.mox.StubOutWithMock(self.driver, 'hosts_up')

        ctxt.elevated().AndReturn(ctxt_elevated)
        self.driver.hosts_up(ctxt_elevated, topic).AndReturn([])

        self.mox.ReplayAll()
        self.assertRaises(exception.NoValidHost,
                self.driver.schedule, ctxt, topic, method,
                *fake_args, **fake_kwargs)

    def test_schedule_prep_resize_doesnt_update_host(self):
        fake_context = context.RequestContext('user', 'project',
                is_admin=True)

        def _return_host(*args, **kwargs):
            return 'host2'

        self.stubs.Set(self.driver, '_schedule', _return_host)

        info = {'called': 0}

        def _fake_instance_update_db(*args, **kwargs):
            # This should not be called
            info['called'] = 1

        self.stubs.Set(driver, 'instance_update_db',
                _fake_instance_update_db)

        instance = {'uuid': 'fake-uuid', 'host': 'host1'}

        self.driver.schedule_prep_resize(fake_context, {}, {}, {},
                instance, {})
        self.assertEqual(info['called'], 0)
