# Copyright (c) 2011 OpenStack, LLC.
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

from nova.openstack.common import cfg
from nova.openstack.common import log as logging
from nova.scheduler import filters

LOG = logging.getLogger(__name__)

cpu_usage_ratio_opt = cfg.FloatOpt("cpu_usage_ratio",
        default=1.0,
        help="physical cpu usage ratio")

CONF = cfg.CONF
CONF.register_opt(cpu_usage_ratio_opt)


class CpuUsageFilter(filters.BaseHostFilter):
    """CPU filter based on actual usage"""

    def host_passes(self, host_state, filter_properties):
        """Only return hosts with sufficient available CPU."""
        instance_type = filter_properties.get('instance_type')
        vcpu_mhz = host_state.cpu_mhz_total / host_state.vcpus_total
        requested_cpu = instance_type['vcpus'] * vcpu_mhz
        used_cpu_mhz = host_state.cpu_mhz_used
        total_usable_cpu_mhz = host_state.cpu_mhz_total

        cpu_mhz_limit = total_usable_cpu_mhz * CONF.cpu_usage_ratio
        usable_cpu = cpu_mhz_limit - used_cpu_mhz
        if not usable_cpu >= requested_cpu:
            LOG.debug(_("%(host_state)s does not have %(requested_cpu)s MHz "
                    "usable cpu, it only has %(usable_cpu)s MHz usable cpu."),
                    locals())
            return False

        # save usage limit for compute node to test against:
        host_state.limits['cpu_mhz_usage'] = cpu_mhz_limit
        return True
