# Copyright 2011-2012 OpenStack LLC.
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
"""
Tests For Scheduler weights.
"""

from nova import context
from nova.scheduler import weights
from nova import test
from nova.tests import matchers
from nova.tests.scheduler import fakes


class TestWeighedHost(test.TestCase):
    def test_dict_conversion(self):
        host_state = fakes.FakeHostState('somehost', None, {})
        host = weights.WeighedHost(host_state, 'someweight')
        expected = {'weight': 'someweight',
                    'host': 'somehost'}
        self.assertThat(host.to_dict(), matchers.DictMatches(expected))

    def test_all_weighers(self):
        classes = weights.all_weighers()
        class_names = [cls.__name__ for cls in classes]
        self.assertEqual(len(classes), 3)
        self.assertIn('RAMWeigher', class_names)
        self.assertIn('LoadBalancingWeigher', class_names)
        self.assertIn('ServerConsolidationWeigher', class_names)

    def test_all_weighers_with_deprecated_config1(self):
        self.flags(compute_fill_first_cost_fn_weight=-1.0)
        classes = weights.all_weighers()
        class_names = [cls.__name__ for cls in classes]
        self.assertEqual(len(classes), 1)
        self.assertIn('_LeastCostWeigher', class_names)

    def test_all_weighers_with_deprecated_config2(self):
        self.flags(least_cost_functions=['something'])
        classes = weights.all_weighers()
        class_names = [cls.__name__ for cls in classes]
        self.assertEqual(len(classes), 1)
        self.assertIn('_LeastCostWeigher', class_names)


class BaseWeigherTestCase(test.TestCase):

    def setUp(self):
        super(BaseWeigherTestCase, self).setUp()
        self.host_manager = fakes.FakeHostManager()
        self.weight_handler = weights.HostWeightHandler()

    def _get_weighed_host(self, hosts, weight_properties=None):
        if weight_properties is None:
            weight_properties = {}
        return self.weight_handler.get_weighed_objects(self.weight_classes,
                hosts, weight_properties)[0]

    def _get_all_hosts(self):
        ctxt = context.get_admin_context()
        fakes.mox_host_manager_db_calls(self.mox, ctxt)
        self.mox.ReplayAll()
        host_states = self.host_manager.get_all_host_states(ctxt)
        self.mox.VerifyAll()
        self.mox.ResetAll()
        return host_states


class RamWeigherTestCase(test.TestCase):

    def setUp(self):
        super(RamWeigherTestCase, self).setUp()
        self.weight_classes = self.weight_handler.get_matching_classes(
                ['nova.scheduler.weights.ram.RAMWeigher'])

    def test_default_of_spreading_first(self):
        hostinfo_list = self._get_all_hosts()

        # host1: free_ram_mb=512
        # host2: free_ram_mb=1024
        # host3: free_ram_mb=3072
        # host4: free_ram_mb=8192

        # so, host4 should win:
        weighed_host = self._get_weighed_host(hostinfo_list)
        self.assertEqual(weighed_host.weight, 8192)
        self.assertEqual(weighed_host.obj.host, 'host4')

    def test_ram_filter_multiplier1(self):
        self.flags(ram_weight_multiplier=-1.0)
        hostinfo_list = self._get_all_hosts()

        # host1: free_ram_mb=-512
        # host2: free_ram_mb=-1024
        # host3: free_ram_mb=-3072
        # host4: free_ram_mb=-8192

        # so, host1 should win:
        weighed_host = self._get_weighed_host(hostinfo_list)
        self.assertEqual(weighed_host.weight, -512)
        self.assertEqual(weighed_host.obj.host, 'host1')

    def test_ram_filter_multiplier2(self):
        self.flags(ram_weight_multiplier=2.0)
        hostinfo_list = self._get_all_hosts()

        # host1: free_ram_mb=512 * 2
        # host2: free_ram_mb=1024 * 2
        # host3: free_ram_mb=3072 * 2
        # host4: free_ram_mb=8192 * 2

        # so, host4 should win:
        weighed_host = self._get_weighed_host(hostinfo_list)
        self.assertEqual(weighed_host.weight, 8192 * 2)
        self.assertEqual(weighed_host.obj.host, 'host4')


class LoadBalancingWeigherTestCase(test.TestCase):

    def setUp(self):
        super(LoadBalancingWeigherTestCase, self).setUp()
        self.weight_classes = self.weight_handler.get_matching_classes(
                ['nova.scheduler.weights.utilization.LoadBalancingWeigher'])

    def test_load_balancing_weigher(self):
        hostinfo_list = self._get_all_hosts()

        # Instance predictions are included in usage values
        # host1: ram_usage=1.0, cpu_usage=2.0
        # host2: ram_usage=0.75, cpu_usage=1.5
        # host3: ram_usage=0.375, cpu_usage=0.5
        # host4: ram_usage=0.0625, cpu_usage=0.125

        # so, host4 should win:
        instance_type = {'memory_mb': 512, 'vcpus': 1}
        weight_properties = {'instance_type': instance_type}
        weighed_host = self._get_weighed_host(hostinfo_list, weight_properties)
        self.assertEqual(weighed_host.weight, 0.875)
        self.assertEqual(weighed_host.obj.host, 'host4')


class ServerConsolidationWeigherTestCase(test.TestCase):

    def setUp(self):
        super(ServerConsolidationWeigherTestCase, self).setUp()
        self.weight_classes = self.weight_handler.get_matching_classes(
            ['nova.scheduler.weights.utilization.ServerConsolidationWeigher'])

    def test_server_consolidation_weigher(self):
        hostinfo_list = self._get_all_hosts()

        # Instance predictions are included in usage values
        # host1: ram_usage=1.0, cpu_usage=2.0
        # host2: ram_usage=0.75, cpu_usage=1.5
        # host3: ram_usage=0.375, cpu_usage=0.5
        # host4: ram_usage=0.0625, cpu_usage=0.125

        # so, host1 should win:
        # NOTE: usage of winning host is above 1 and should normally be
        # filtered out prior to weighing.
        instance_type = {'memory_mb': 512, 'vcpus': 1}
        weight_properties = {'instance_type': instance_type}
        weighed_host = self._get_weighed_host(hostinfo_list, weight_properties)
        self.assertEqual(weighed_host.weight, 1.0)
        self.assertEqual(weighed_host.obj.host, 'host1')
