#!/bin/bash
# The following environment variables may be set to substitute in a
# custom bridge or interface name.  Normally these are provided by the calling
# SoftwareConfig resource, but they may also be set manually for testing.
# $bridge_name : The bridge device name to apply
# $interface_name : The interface name to apply
#
# Also this token is replaced via a str_replace in the SoftwareConfig running
# the script - in future we may extend this to also work with a variable, e.g
# a deployment input via input_values
# $network_config : the json serialized os-net-config config to apply
#
set -eux

function configure_safe_defaults() {

[[ $? == 0 ]] && return 0

cat > /etc/os-net-config/dhcp_all_interfaces.yaml <<EOF_CAT
# This file is an autogenerated safe defaults file for os-net-config
# which runs DHCP on all discovered interfaces to ensure connectivity
# back to the undercloud for updates
network_config:
EOF_CAT

    for iface in $(ls /sys/class/net | grep -v -e ^lo$ -e ^vnet$); do
        local mac_addr_type="$(cat /sys/class/net/${iface}/addr_assign_type)"
        local vf_parent="/sys/class/net/${iface}/device/physfn"
        if [ "$mac_addr_type" != "0" ]; then
            echo "Device has generated MAC, skipping."
        elif [[ -d $vf_parent ]]; then
            echo "Device (${iface}) is a SR-IOV VF, skipping."
        else
            HAS_LINK="$(cat /sys/class/net/${iface}/carrier || echo 0)"

            TRIES=10
            while [ "$HAS_LINK" == "0" -a $TRIES -gt 0 ]; do
                # Need to set the link up on each iteration
                ip link set dev $iface up &>/dev/null
                HAS_LINK="$(cat /sys/class/net/${iface}/carrier || echo 0)"
                if [ "$HAS_LINK" == "1" ]; then
                    break
                else
                    sleep 1
                fi
                TRIES=$(( TRIES - 1 ))
            done
            if [ "$HAS_LINK" == "1" ] ; then
cat >> /etc/os-net-config/dhcp_all_interfaces.yaml <<EOF_CAT
  -
    type: interface
    name: $iface
    use_dhcp: true
EOF_CAT
            fi
        fi
    done
    set +e
    os-net-config -c /etc/os-net-config/dhcp_all_interfaces.yaml -v --detailed-exit-codes --cleanup
    RETVAL=$?
    set -e
    if [[ $RETVAL != 2 || $RETVAL != 0 ]]; then
        echo "ERROR: configuration of safe defaults failed."
    fi
}

if [ -n '$network_config' ]; then
    if [ -z "${disable_configure_safe_defaults:-}" ]; then
        trap configure_safe_defaults EXIT
    fi

    # Backup the old /etc/os-net-config/config.json, if it exists
    DATETIME=`date +"%Y-%m-%dT%H:%M:%S"`
    if [ -f /etc/os-net-config/config.json ]; then
        mv /etc/os-net-config/config.json /etc/os-net-config/config.json.$DATETIME
    fi

    mkdir -p /etc/os-net-config
    # Note these variables come from the calling heat SoftwareConfig
    echo '$network_config' > /etc/os-net-config/config.json

    sed -i "s/: \"bridge_name/: \"${bridge_name:-''}/g" /etc/os-net-config/config.json
    sed -i "s/interface_name/${interface_name:-''}/g" /etc/os-net-config/config.json

    set +e
    os-net-config -c /etc/os-net-config/config.json -v --detailed-exit-codes
    RETVAL=$?
    set -e

    if [[ $RETVAL == 2 ]]; then
        #NOTE: dprince this udev rule can apparently leak DHCP processes?
        # https://bugs.launchpad.net/tripleo/+bug/1538259
        # until we discover the root cause we can simply disable the
        # rule because networking has already been configured at this point
        if [ -f /etc/udev/rules.d/99-dhcp-all-interfaces.rules ]; then
            rm /etc/udev/rules.d/99-dhcp-all-interfaces.rules
        fi

    elif [[ $RETVAL != 0 ]]; then
        echo "ERROR: os-net-config configuration failed." >&2
        exit 1
    fi

    # Remove files used by os-apply-config for old style configs
    if [ -f /usr/libexec/os-apply-config/templates/etc/os-net-config/config.json ]; then
        mv /usr/libexec/os-apply-config/templates/etc/os-net-config/config.json /usr/libexec/os-apply-config/templates/etc/os-net-config/config.json.$DATETIME
    fi
    if [ -f /usr/libexec/os-apply-config/templates/etc/os-net-config/element_config.json ]; then
        mv /usr/libexec/os-apply-config/templates/etc/os-net-config/element_config.json /usr/libexec/os-apply-config/templates/etc/os-net-config/element_config.json.$DATETIME
    fi
fi
