# Licensed under the Apache License, Version 2.0 (the "License"); you may not
# use this file except in compliance with the License. You may obtain a copy
# of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Hosts resource and resource shell wrapper."""
from __future__ import print_function

import argparse
import sys

from cratonclient.common import cliutils
from cratonclient import exceptions as exc

DEFAULT_CLOUD_FIELDS = [
    'id',
    'name',
    'created_at',
]

CLOUD_FIELDS = DEFAULT_CLOUD_FIELDS + [
    'updated_at',
    'note',
    'project_id',
]


@cliutils.arg('-n', '--name',
              metavar='<name>',
              required=True,
              help='Name of the host.')
@cliutils.arg('--note',
              help='Note about the host.')
def do_cloud_create(cc, args):
    """Register a new cloud with the Craton service."""
    fields = {k: v for (k, v) in vars(args).items()
              if k in CLOUD_FIELDS and not (v is None)}

    cloud = cc.clouds.create(**fields)
    args.formatter.configure(wrap=72).handle(cloud)


@cliutils.arg('--fields',
              nargs='+',
              metavar='<fields>',
              default=DEFAULT_CLOUD_FIELDS,
              help='Comma-separated list of fields to display. '
                   'Only these fields will be fetched from the server. '
                   'Can not be used when "--detail" is specified')
@cliutils.arg('--all',
              action='store_true',
              default=False,
              help='Retrieve and show all clouds. This will override '
                   'the provided value for --limit and automatically '
                   'retrieve each page of results.')
@cliutils.arg('--detail',
              action='store_true',
              default=False,
              help='Show detailed information about all clouds.')
@cliutils.arg('--limit',
              metavar='<limit>',
              type=int,
              help='Maximum number of clouds to return.')
@cliutils.arg('--marker',
              metavar='<marker>',
              default=None,
              help='ID of the cell to use to resume listing clouds.')
def do_cloud_list(cc, args):
    """List all clouds."""
    params = {}
    if args.limit is not None:
        if args.limit < 0:
            raise exc.CommandError('Invalid limit specified. Expected '
                                   'non-negative limit, got {0}'
                                   .format(args.limit))
        params['limit'] = args.limit
    if args.all is True:
        params['limit'] = 100

    if args.detail:
        if args.fields and args.fields == DEFAULT_CLOUD_FIELDS:
            args.fields = CLOUD_FIELDS
        else:
            raise exc.CommandError(
                'Cannot specify both --fields and --detail.'
            )
        params['detail'] = args.detail

    fields = args.fields
    for field in args.fields:
        if field not in CLOUD_FIELDS:
            raise exc.CommandError(
                'Invalid field "{}"'.format(field)
            )

    params['marker'] = args.marker
    params['autopaginate'] = args.all

    clouds_list = cc.clouds.list(**params)
    args.formatter.configure(fields=list(fields)).handle(clouds_list)


@cliutils.arg('id',
              metavar='<cloud>',
              type=int,
              help='ID of the cloud.')
def do_cloud_show(cc, args):
    """Show detailed information about a cloud."""
    cloud = cc.clouds.get(args.id)
    args.formatter.configure(wrap=72).handle(cloud)


@cliutils.arg('id',
              metavar='<cloud>',
              type=int,
              help='ID of the cloud')
@cliutils.arg('-n', '--name',
              metavar='<name>',
              help='Name of the cloud.')
@cliutils.arg('--note',
              help='Note about the cloud.')
def do_cloud_update(cc, args):
    """Update a cloud that is registered with the Craton service."""
    fields = {k: v for (k, v) in vars(args).items()
              if k in CLOUD_FIELDS and not (v is None)}
    item_id = fields.pop('id')
    if not fields:
        raise exc.CommandError(
            'Nothing to update... Please specify one or more of --name, or '
            '--note'
        )
    cloud = cc.clouds.update(item_id, **fields)
    args.formatter.configure(wrap=72).handle(cloud)


@cliutils.arg('id',
              metavar='<cloud>',
              type=int,
              help='ID of the cloud.')
def do_cloud_delete(cc, args):
    """Delete a cloud that is registered with the Craton service."""
    try:
        response = cc.clouds.delete(args.id)
    except exc.ClientException as client_exc:
        raise exc.CommandError(
            'Failed to delete cloud {} due to "{}:{}"'.format(
                args.id, client_exc.__class__, str(client_exc),
            )
        )
    else:
        print("Cloud {0} was {1} deleted.".
              format(args.id, 'successfully' if response else 'not'))


@cliutils.arg('id',
              metavar='<cloud>',
              type=int,
              help='ID or name of the cloud.')
@cliutils.handle_shell_exception
def do_cloud_vars_get(cc, args):
    """Get variables for a cloud."""
    variables = cc.clouds.get(args.id).variables.get()
    formatter = args.formatter.configure(dict_property="Variable", wrap=72)
    formatter.handle(variables)


@cliutils.arg('id',
              metavar='<cloud>',
              type=int,
              help='ID of the cloud.')
@cliutils.arg('variables', nargs=argparse.REMAINDER)
@cliutils.handle_shell_exception
def do_cloud_vars_set(cc, args):
    """Set variables for a cloud."""
    cloud_id = args.id
    if not args.variables and sys.stdin.isatty():
        raise exc.CommandError(
            'Nothing to update... Please specify variables to set in the '
            'following format: "key=value". You may also specify variables to '
            'delete by key using the format: "key="'
        )
    adds, deletes = cliutils.variable_updates(args.variables)
    variables = cc.clouds.get(cloud_id).variables
    if deletes:
        variables.delete(*deletes)
    variables.update(**adds)
    formatter = args.formatter.configure(wrap=72, dict_property="Variable")
    formatter.handle(variables.get())


@cliutils.arg('id',
              metavar='<cloud>',
              type=int,
              help='ID of the cloud.')
@cliutils.arg('variables', nargs=argparse.REMAINDER)
@cliutils.handle_shell_exception
def do_cloud_vars_delete(cc, args):
    """Delete variables for a cloud by key."""
    cloud_id = args.id
    if not args.variables and sys.stdin.isatty():
        raise exc.CommandError(
            'Nothing to delete... Please specify variables to delete by '
            'listing the keys you wish to delete separated by spaces.'
        )
    deletes = cliutils.variable_deletes(args.variables)
    variables = cc.clouds.get(cloud_id).variables
    response = variables.delete(*deletes)
    print("Variables {0} deleted.".
          format('successfully' if response else 'not'))
