# Copyright 2019 Intel, Inc.
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

import json
import mock
import testtools

from rsd_lib.resources.v2_1.system import processor


class ProcessorTestCase(testtools.TestCase):
    def setUp(self):
        super(ProcessorTestCase, self).setUp()
        self.conn = mock.Mock()
        with open(
            "rsd_lib/tests/unit/json_samples/v2_1/processor.json", "r"
        ) as f:
            self.conn.get.return_value.json.return_value = json.loads(f.read())

        self.processor_inst = processor.Processor(
            self.conn,
            "/redfish/v1/Systems/System1/Processors/CPU1",
            redfish_version="1.0.2",
        )

    def test__parse_attributes(self):
        self.processor_inst._parse_attributes()
        self.assertEqual("CPU1", self.processor_inst.identity)
        self.assertEqual("Processor", self.processor_inst.name)
        self.assertEqual(None, self.processor_inst.description)
        self.assertEqual("CPU 1", self.processor_inst.socket)
        self.assertEqual("CPU", self.processor_inst.processor_type)
        self.assertEqual("x86", self.processor_inst.processor_architecture)
        self.assertEqual("x86-64", self.processor_inst.instruction_set)
        self.assertEqual(
            "Intel(R) Corporation", self.processor_inst.manufacturer
        )
        self.assertEqual(
            "Multi-Core Intel(R) Xeon(R) processor 7xxx Series",
            self.processor_inst.model,
        )
        self.assertEqual(3700, self.processor_inst.max_speed_mhz)
        self.assertEqual(
            "0x42", self.processor_inst.processor_id.effective_family
        )
        self.assertEqual(
            "0x61", self.processor_inst.processor_id.effective_model
        )
        self.assertEqual(
            "0x34AC34DC8901274A",
            self.processor_inst.processor_id.identification_registers,
        )
        self.assertEqual(
            "0x429943", self.processor_inst.processor_id.microcode_info
        )
        self.assertEqual("0x1", self.processor_inst.processor_id.step)
        self.assertEqual(
            "GenuineIntel", self.processor_inst.processor_id.vendor_id
        )
        self.assertEqual("OK", self.processor_inst.status.health)
        self.assertEqual(None, self.processor_inst.status.health_rollup)
        self.assertEqual("Enabled", self.processor_inst.status.state)
        self.assertEqual(8, self.processor_inst.total_cores)
        self.assertEqual(16, self.processor_inst.total_threads)
        self.assertEqual("E5", self.processor_inst.oem.intel_rackscale.brand)
        self.assertEqual(
            ["sse", "sse2", "sse3"],
            self.processor_inst.oem.intel_rackscale.capabilities,
        )


class ProcessorCollectionTestCase(testtools.TestCase):
    def setUp(self):
        super(ProcessorCollectionTestCase, self).setUp()
        self.conn = mock.Mock()
        with open(
            "rsd_lib/tests/unit/json_samples/v2_1/"
            "processor_collection.json",
            "r",
        ) as f:
            self.conn.get.return_value.json.return_value = json.loads(f.read())
        self.processor_col = processor.ProcessorCollection(
            self.conn,
            "/redfish/v1/Systems/System1/Processors",
            redfish_version="1.1.0",
        )

    def test__parse_attributes(self):
        self.processor_col._parse_attributes()
        self.assertEqual("1.1.0", self.processor_col.redfish_version)
        self.assertEqual(
            (
                "/redfish/v1/Systems/System1/Processors/CPU1",
                "/redfish/v1/Systems/System1/Processors/CPU2",
            ),
            self.processor_col.members_identities,
        )

    @mock.patch.object(processor, "Processor", autospec=True)
    def test_get_member(self, mock_processor):
        self.processor_col.get_member(
            "/redfish/v1/Systems/System1/Processors/CPU1"
        )
        mock_processor.assert_called_once_with(
            self.processor_col._conn,
            "/redfish/v1/Systems/System1/Processors/CPU1",
            redfish_version=self.processor_col.redfish_version,
        )

    @mock.patch.object(processor, "Processor", autospec=True)
    def test_get_members(self, mock_processor):
        members = self.processor_col.get_members()
        calls = [
            mock.call(
                self.processor_col._conn,
                "/redfish/v1/Systems/System1/Processors/CPU1",
                redfish_version=self.processor_col.redfish_version,
            ),
            mock.call(
                self.processor_col._conn,
                "/redfish/v1/Systems/System1/Processors/CPU2",
                redfish_version=self.processor_col.redfish_version,
            ),
        ]
        mock_processor.assert_has_calls(calls)
        self.assertIsInstance(members, list)
        self.assertEqual(2, len(members))
